module Types where

{----------------------------------------------------------------------------+
| All the basic types. All other modules for the project should import this  |
| Also some non-derived Show instances here to make debugging simpler: parse |
| trees pretty-printed with 'disp'/'showProg'                                |
+----------------------------------------------------------------------------}

{- additional types that are needed globally -}
type ErrorMsg = String
type Arity = Int

{- types/datas specified for the project -}
type FunName = String
type Formal = String
type FunDef = ([Formal], Exp)
type FunEqn = (FunName, FunDef)

type Prog = [FunEqn]

{- expressions -}
data Exp = Num Int 
         | Var String
         | App String [Exp]
         | If Bexp Exp Exp
         | Add Exp Exp
         | Sub Exp Exp
         | Mul Exp Exp
         | Div Exp Exp
           
{- boolean expressions -}
data Bexp = Or Bexp Bexp
          | And Bexp Bexp      -- connectives
          | Rel Exp Comp Exp   -- relations

{- built-in relations -}
data Comp = Equal | NEqual | Less | LessEq | Greater | GreaterEq


{- Show instances -}

instance Show Exp where
    show (Num x) = show x
    show (Var x) = x
    show (App f xs) = f ++ brack (show xs)
    show (If a b c) = "if " ++ show a ++ " then " ++ show b ++ " else " ++ show c
    show (Add a b) = showOp "+" a b
    show (Sub a b) = showOp "-" a b
    show (Mul a b) = showOp "*" a b
    show (Div a b) = showOp "/" a b


instance Show Bexp where
    show (Or a b) = showOp "||" a b
    show (And a b) = showOp "&&" a b
    show (Rel exp comp exp2) = showOp (show comp) exp exp2

instance Show Comp where
    show Equal = "=="
    show NEqual = "/="
    show Less = "<"
    show LessEq = "<="
    show Greater = ">"
    show GreaterEq = ">="

brack :: String -> String
brack x = "(" ++ x ++ ")"

disp :: Prog -> IO ()
disp = putStr . showProg

showProg :: Prog -> String
showProg = unlines . map showEqn 
    where
    showEqn (fun,(args,def)) = fun ++ brack (unwords args) ++ " = " ++ show def

showOp :: (Show a, Show b) => String -> a -> b -> String
showOp op arg1 arg2 = brack $ show arg1 ++ op ++ show arg2


{---------------------------------------+
| Basic Operations to do with the types |
+---------------------------------------}


{- given an expression, find the variables referred to in the expression -}
getVars :: Exp -> [String]
getVars (Num _) = []
getVars (Var x) = [x]
getVars (App _ args) = concatMap getVars args
getVars (If  a b c)  = getBVars a ++ concatMap getVars [b,c]
getVars (Add a b)    = concatMap getVars [a,b]
getVars (Sub a b)    = concatMap getVars [a,b]
getVars (Mul a b)    = concatMap getVars [a,b]
getVars (Div a b)    = concatMap getVars [a,b]

getBVars :: Bexp -> [String]
getBVars (Or  a b) = concatMap getBVars [a,b]
getBVars (And a b) = concatMap getBVars [a,b]
getBVars (Rel a _ b) = concatMap getVars [a,b]

{- return a list of (function, numberArgs) pairs for each occurrence of an App
   in the expression -}
getFuns :: Exp -> [(String, Int)]
getFuns (Num _) = []
getFuns (Var x) = []
getFuns (App f args) = (f,length args):(concatMap getFuns args)
getFuns (If  a b c)  = getBFuns a ++ concatMap getFuns [b,c]
getFuns (Add a b)    = concatMap getFuns [a,b]
getFuns (Sub a b)    = concatMap getFuns [a,b]
getFuns (Mul a b)    = concatMap getFuns [a,b]
getFuns (Div a b)    = concatMap getFuns [a,b]

getBFuns :: Bexp -> [(String, Int)]
getBFuns (Or  a b) = concatMap getBFuns [a,b]
getBFuns (And a b) = concatMap getBFuns [a,b]
getBFuns (Rel a _ b) = concatMap getFuns [a,b]

