module Basics where

import Types
import Array
import FiniteMap
-- import IOExts(unsafePerformIO)
import List(sort)

-- flag to say whether we can print lots or not
debug :: Bool
debug = True

infinity :: Int
infinity = 10000000

{- primitive functions that were in Types but because that takes too long to compile, they're here -}

gameToCode :: Game -> GameLookup
codeToGame :: GameLookup -> Game
gameToCode game
    = let ((_DenLoc,a,b,c,d),(_DenLoc',e,f,g,h)) = pieceLocs game
          toNum (Onboard x y) = 10 * x + y  -- coordinates are one digit
          toNum (Offboard) = 99
          -- convert four locations into one Int
          makeElem x y z w = 1000000 * (toNum x) + 10000 * (toNum y) + 
                                 100 * (toNum z) + toNum w
      in
          ((if whoseTurn game == White then (-1) else 1) * makeElem a b c d, 
            makeElem e f g h)

codeToGame _ = error "codeToGame not supported yet"

-- note that numPosEvals must be strict for this to work -- otherwise HUGE unevaluated
-- 1 + (1 + (1 + ... ))s get created
incPosEval :: Search ()
incPosEval = Search (\state -> ((), state {numPosEvals = let x = numPosEvals state in 
                                                             {- x `seq` -} (x + 1)}))

resetMonad :: Search ()
resetMonad = Search (\_state -> ((), initState))
             where initState = State { numPosEvals = 0, hashTable = emptyFM }

getHashScore :: Game -> Search (Maybe (Depth, Score, Move))
getHashScore game 
    = Search (\state -> (lookupFM (hashTable state) (gameToCode game), state))

setHashScore :: Game -> Depth -> Score -> Move -> Search ()
setHashScore game depth score best_response
    = let gameCode = gameToCode game
      in Search (\state -> ((), 
                            let hash = hashTable state
                            in case lookupFM hash gameCode of
                                    Just (dep, _, _) | dep > depth -> debugPutStr "setHashScore: foo" state
                                    _ -> state {hashTable = addToFM hash gameCode (depth, score, best_response)}
             ))



{- End primitive functions from Types.hs -}

{-------------------+
| Dodgy Debug Stuff |
+-------------------}

{- caveat: must be used in the form () <- debugRetVoid 123 -}
debugRetVoid :: (Show a, Monad m) => a -> m ()
debugRetVoid val = debugPrint val (return ())

debugPrint :: Show a => a -> b -> b
debugPrint val r = debugPutStr (show val) r

debugPutStr :: String -> a -> a
-- debugPutStr str r =  unsafePerformIO $ do { putStrLn str; return r }
debugPutStr str r =  undefined 

{- in INCREASING order -}
sortByFun :: (Ord a, Ord b) => (a -> b) -> [a] -> [a]
sortByFun f xs = map snd $ sort [(f x, x) | x <- xs]


-- count the pieces other than the den on the baord
countPieces :: Game -> Int
countPieces game = length $ filter (stillOnBoard game) allMoveablePieces

swapPlayers :: GameInfo -> GameInfo
swapPlayers info 
    = Info { player = player' } 
      where player' White = (player info) Black
            player' Black = (player info) White
      

{- the taxicab distance between two squares on the board -}
distance :: Location -> Location -> Int
distance (Onboard a b) (Onboard c d)
    = abs (a - c) + abs (b - d)
distance _ _ = error "distance only works between two things on board!"

properLastMoveWin :: Game -> Bool
properLastMoveWin game
    = quickLastMoveWin game || length (legalMoves game) == 0

{- Test if the den has been captured 
   Don't test if no legal moves left, as this is too slow generally -}
quickLastMoveWin :: Game -> Bool
quickLastMoveWin game
    = findPiece game (whoseTurn game, Den) == Offboard

canCapture :: Piece -> Square -> Bool
canCapture (_, Den) _ = error "canCapture Den sth -- shouldn't happen"
canCapture _ Empty = True
canCapture (turn, species) (Piece (turn', species'))
    = turn /= turn' && 
      canCaptureSpecies species species'

canCaptureSpecies :: Species -> Species -> Bool
canCaptureSpecies s1 s2
    = case (s1,s2) of
           (Mouse, Elephant) -> True
           (Elephant, Mouse) -> False
           _ -> s1 >= s2

legalMoves :: Game -> [Move]
legalMoves game = filter (legalMove game) allMoves

legalMove :: Game -> Move -> Bool
legalMove game (Move species dir)
    = let turn = whoseTurn game
          piece = (turn, species)
          currSquare = findPiece game piece
          destSquare = adjacentSquare currSquare dir
          offBoard = case currSquare of {Offboard -> True; _ -> False}
      in
      not offBoard && 
      inBounds game destSquare &&
      canCapture piece (readSquare game destSquare)

stillOnBoard :: Game -> Piece -> Bool
stillOnBoard game piece
    = case findPiece game piece of
           Offboard -> False
           _ -> True

readSquare :: Game -> Location -> Square
readSquare game = readBoardSquare (board game)

inBounds :: Game -> Location -> Bool
inBounds game (Onboard x y) 
    = let (h,w) = (height game, width game) in
      x <= h && x >= 1 && 
      y <= w && y >= 1
inBounds _    Offboard = False

openWith :: [Move] -> Game
openWith moves = makeMoves moves newGame

makeMoves :: [Move] -> Game -> Game
makeMoves moves game = foldl (flip makeMove) game moves

makeMove :: Move -> Game -> Game
makeMove mv@(Move species dir) game
    = game { board = board',
             pieceLocs = pieceLocs',
             history = mv:(history game),
             numMoves = numMoves game + 1,
             whoseTurn = toggle (whoseTurn game)
           }
      where
      board' = writeBoardSquares (board game) 
                [(fromLocation, Empty), (toLocation, Piece pieceMoved)]
      pieceLocs' = 
          (case takenSquare of 
               Piece p -> setPieceLoc p Offboard
               Empty -> id)  $
          setPieceLoc pieceMoved toLocation (pieceLocs game)
      
      pieceMoved = (whoseTurn game, species)
      fromLocation = findPiece game pieceMoved
      toLocation = adjacentSquare fromLocation dir
      takenSquare = readBoardSquare (board game) toLocation
    
-- this is extremely inefficient but is only used so humans can undo
-- moves -- we just replay all moves up to the last n
unmakePlies :: Int -> Game -> Game
unmakePlies num game 
    = openWith $ reverse $ drop num (history game)

unmakeMove :: Game -> Game
unmakeMove = unmakePlies 2



adjacentSquare :: Location -> Direction -> Location
adjacentSquare (Onboard x y) dir
    = case dir of 
           L -> Onboard x (y - 1)
           R -> Onboard x (y + 1)
           U -> Onboard (x - 1) y
           D -> Onboard (x + 1) y
adjacentSquare Offboard _ = error "adjacentSquare of something off the board"

findPiece :: Game -> Piece -> Location
findPiece game p = findPieceInLocs (pieceLocs game) p


{--------------------------------+
| Setting up an initial position |
+--------------------------------}

newGame :: Game
newGame = newGameDims 9 7

-- just give piece locations other than den locations
setupPosition :: [(Piece, (Int, Int))] -> Turn -> Game
setupPosition pieceSquares turn
    = let g = newGameParams 9 7 pieceLocs in 
          g { whoseTurn = turn }
      where
      pieceLocs = 
          foldr ($) start [setPieceLoc p (uncurry Onboard pos)
                              | (p, pos) <- (denPlaces ++ pieceSquares) ]
          where start = dup (Offboard, Offboard, Offboard, Offboard, Offboard)
                dup a = (a,a)
                denPlaces = [((White, Den), (9, 4)), ((Black, Den), (1,4))]

-- allows you to set the dimensions; use default piece locations
-- 4 7 are the smallest allowed dimensions
newGameDims :: Int -> Int -> Game
newGameDims x y | x < 4 || y < 6 = error "Dimensions too small!"
                | even y = error "Must be odd width" -- or else looks silly!
                | otherwise = newGameParams x y (normalPieceLocs x y)
    where
    normalPieceLocs :: Int -> Int -> PieceLocs
    normalPieceLocs h w
        =  (White, Den)      ==> (h, 1 + w `div` 2)   $
           (White, Mouse)    ==> (h, w - 1)           $
           (White, Wolf)     ==> (h - 1, 3)           $
           (White, Tiger)    ==> (h - 1, w - 2)       $
           (White, Elephant) ==> (h, 2)               $
           (Black, Elephant) ==> (1, w - 1)           $
           (Black, Tiger)    ==> (2, 3)               $
           (Black, Wolf)     ==> (2, w - 2)           $
           (Black, Mouse)    ==> (1, 2)               $
           (Black, Den)      ==> (1, 1 + w `div` 2)   $
           start
          where piece ==> (x',y') = setPieceLoc piece (Onboard x' y')
                start = dup (Offboard, Offboard, Offboard, Offboard, Offboard)
                dup a = (a,a)
    
-- allows you to set height/width/initial pos
newGameParams :: Int -> Int -> PieceLocs -> Game
newGameParams h w piecelocs
    = Game { height = h, width = w, board = brd,
             numMoves = 0, whoseTurn = White, pieceLocs = piecelocs,
             history = []
           }
    where
    brd = emptyBoard // differences
          where 
          emptyBoard = array ((1,1),(h,w)) 
                         [((x,y), Empty) | x <- [1..h], y <- [1..w]]
          differences = [((a,b), Piece piece) | 
                                 turn <- [White .. Black],
                                 species <- [Den .. Elephant],
                                 let piece = (turn, species),
                                 let pieceLoc = findPieceInLocs piecelocs piece,
                                 pieceLoc /= Offboard,
                                 let Onboard a b = pieceLoc]
