module CPU where

import Types
import Basics
import CPUTime(getCPUTime)
import PosEval(positionEvaluate)
import Monad(when, liftM)
-- import Exception(assert)
-- import qualified Posix(getProcessTimes,elapsedTime)

default ()

simple :: CPUPrefs
simple = CPUPrefs {
            speciesValue = val,
            magicNums = [1000,50,23,10,100000,41,2,78,3,8004,2, 83, 4, 94, 32,22],
            --           0    1  2  3  4      5  6 7  8  9    10 11 12 13 14 15
           -- magicNums = [1000, 1, 5, 3, 100000, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2],
            {- greed = 1000, denCloseness = 1 aggressiveness = 5, defensiveness = 3, forceMateTrust = 100000, -}
            name = "Simple", 
            chooseSearchDepth = depth'

         }
         where
         depth' game | countPieces game <= 4 = 9
                     | countPieces game <= 6 = 7
                     | numMoves game <= 5 = 3
                     | otherwise = 4


         val Tiger = 7
         val Elephant = 6
         val Mouse = 4
         val Wolf = 3
         val Den = infinity `div` 10

getCPUMove :: Game -> CPUPrefs -> IO Move
getCPUMove game prefs =
    do 
    -- beforeCPU     <- CPUTime.getCPUTime   
    -- beforeProcess <- liftM Posix.elapsedTime Posix.getProcessTimes
    beforeCPU     <- CPUTime.getCPUTime   
    beforeProcess <- return 0 

    let depth = chooseSearchDepth prefs game
        ((score, move), state) = runSearch $
                                  scoreAndMove game depth prefs (whoseTurn game)
    when debug (
        do
        putStrLn $ "Depth: " ++ show depth
        putStrLn $ "Expected score is " ++ show score
        putStrLn $ "Positions evaluated: " ++ show (numPosEvals state)
        )
        {- putStrLn $ "State: " ++ show state -}

    afterCPU     <- CPUTime.getCPUTime   
    -- afterProcess <- liftM Posix.elapsedTime Posix.getProcessTimes
    afterProcess <- return 0 

    let picoSecToMilli x = (fromIntegral x :: Float) / (10 ** 9)
        cpusecDiff = 0.001 * picoSecToMilli (afterCPU - beforeCPU)
        -- this seems to be the most accurate: centiseconds since start
        procTime :: Float
        procTime = (0.01 :: Float) * ((fromIntegral ((afterProcess::Integer) - (beforeProcess ::Integer))) :: Float)
        truncateDecPlaces dps str
            = if 'e' `elem` str 
              then str
              else let (before, after) = break (== '.') str 
                       take' num xs = let len = length xs in 
                            if num <= len 
                                then take num xs 
                                else xs ++ replicate (num - len) '0'

                   in before ++ take' (dps + 1) after

        foo = truncateDecPlaces 2
        
    when debug (do
                putStrLn $ "Time: " ++ foo (show procTime) ++ "s (CPU "
                                        ++ foo (show cpusecDiff) ++ "s)"

                putStrLn $ "#" ++ smallShow move ++ "#"
               )
    return move

{- this is useful for doing shorter searches and just getting the score
   back from them (or the best move) -}
scoreAndMove :: Game -> Depth -> CPUPrefs -> Turn -> Search (Score, Move)
scoreAndMove game depth prefs perspective
    = let alpha' = -10 * infinity
          beta'  =  10 * infinity
      in do 
         {- Earlier versions:

         scores_and_moves <- negamax depth prefs game
         return (minimum scores_and_moves)

         scores_and_moves <- minimax depth prefs game
         return (maximum scores_and_moves)
         -}

         scores_and_moves <- alphabeta depth prefs game (whoseTurn game) (alpha', beta')
         () <- debugRetVoid ("alpha-beta", scores_and_moves)
         return (maximum scores_and_moves)

quickGoodnessScore :: CPUPrefs -> Move -> Game -> Score
quickGoodnessScore prefs (Move species dir) game 
    =          sum [ if movesTowardsDen then 10 else -10,
                     3 * moveBigPiece,
                     1000 * takesPiece,
                     {- 10 * followFlow, -}
                     0
                   ]
      where
      turn = whoseTurn game

      -- encourage moving the same piece as you did on
      -- the last turn: useful in simulating a consistent plan

      followFlow :: Score
      followFlow
         = case history game of
                (_:(Move s d):_) | s /= species    -> 0 
                                 | d == toggle dir -> -1  -- undoes last move!
                                 | otherwise -> 1
                _ -> 0


      currentLoc@(Onboard x _) = findPiece game (turn, species)
      Onboard den_x _ = findPiece game (opp, Den)
      newLoc@(Onboard _ _) = adjacentSquare currentLoc dir

      moveBigPiece = speciesValue prefs species
      opp = toggle turn

      movesTowardsDen = (dir == U && turn == White) ||
                        (dir == D && turn == Black) || 
                        (dir == L && x > den_x)     ||
                        (dir == R && x < den_x)

      takesPiece = case readSquare game newLoc of
                        Empty -> 0
                        (Piece (_,s)) -> (speciesValue prefs) s


negamax :: Depth -> CPUPrefs -> Game -> Search [(Score, Move)] 
negamax depth prefs game
    = do
      let legal = legalMoves game

          nextLevelScore :: Move -> Search (Score, Move)
          nextLevelScore mv = let gm = makeMove mv game
                              in if depth == 1 then do score <- positionEvaluate prefs gm (whoseTurn gm) 
                                                       return (score, mv)
                                               else do s_m <- negamax (depth - 1) prefs gm
                                                       let (score, _response) = minimum s_m
                                                       return (-1 * score, mv)

      scoresAndMoves <- mapM nextLevelScore legal
      return scoresAndMoves

      
minimax :: Depth -> CPUPrefs -> Game -> Turn -> Search [(Score, Move)]
minimax depth prefs game perspective
    = do
      let legal = legalMoves game
          is_max_turn = perspective == whoseTurn game

          moveScore :: Move -> Search Score
          moveScore mv = let gm = makeMove mv game
                         in if depth == 1 then positionEvaluate prefs gm perspective
                                          else do scores_moves <- minimax (depth - 1) prefs gm perspective
                                                  {- remember we're getting scores_moves from the next level down so if it's max's turn
                                                     at this level then min will choose the _minimum_ of his scores -}
                                                  let (score, _response) = (if is_max_turn then minimum else maximum) scores_moves
                                                  return score
                                                           
      scores <- mapM moveScore legal
      let scoresAndMoves = zip scores legal
      return scoresAndMoves


alphabeta :: Depth -> CPUPrefs -> Game -> Turn -> (Score, Score) -> Search [(Score, Move)]
alphabeta depth prefs game perspective (alpha, beta)
    = do 
      legal <- let goodness move = -1 * (quickGoodnessScore prefs move game)
               in  return $ sortByFun goodness (legalMoves game) 

      let is_max_turn = perspective == whoseTurn game
          infinity_whoseturn = if is_max_turn then infinity else -infinity -- infinity for whoever's on turn

          moveScore :: Move -> (Score, Score) -> Search Score
          moveScore mv (alpha', beta') 
               = let gm = makeMove mv game
                 in if quickLastMoveWin gm 
                      then return infinity_whoseturn
                      else 
                        do
                        hashTableScore <- getHashScore gm
                        case hashTableScore of 
                            Just (depthSearched, score', _response) | depthSearched >= depth -> return score'
                            _ -> if depth == 1 then positionEvaluate prefs gm perspective
                                    else do 
                                         scores_moves <- alphabeta (depth - 1) prefs gm perspective (alpha', beta')
                                         let score = if null scores_moves 
                                                        then infinity_whoseturn  -- no legal responses
                                                        else fst ((if is_max_turn then minimum else maximum) scores_moves)
                                         when (depth >= 0)
                                              (setHashScore gm depth score (error "Foo"))
                                         return score

          for_loop :: [Move] -> (Score, Score) -> Search [(Score, Move)]
          for_loop []       _ = return []
          for_loop (mv:mvs) (alpha', beta')
              = do
                score <- moveScore mv (alpha', beta')
                let (newAlpha, newBeta) = if is_max_turn then (max score alpha', beta')
                                                         else (alpha', min score beta')
                if newAlpha >= newBeta then                                             
                    return [(score, mv)]  -- this is effectively the end of the movelist: TODO: maybe this should also add on the
                                          -- rest of the moves with very low scores? -- would help iter. deep.
                    else do scoresMoves <- for_loop mvs (newAlpha, newBeta)
                            return $ (score, mv):scoresMoves
                
      scoresAndMoves <- for_loop legal (alpha, beta)
      return scoresAndMoves
      

{-
alphabeta :: Game -> (Score, Score) -> Depth -> 
                      CPUPrefs -> Turn -> Search (Score, Move)
alphabeta game (init_alpha, init_beta) depth prefs turn
    = if legalMoveList == []
      then if is_max_turn then return (-infinity, undefined) 
                          else return (infinity, undefined)
      else scores (init_alpha, init_beta) undefined legalMoveList
    where
    legalMoveList = sortByFun goodness (legalMoves game)
        where
        goodness move =  -1 * (quickGoodnessScore prefs move game)
                                              

    is_max_turn = whoseTurn game == turn

    {- this does the looping through the moves and explicitly stores
       alpha and beta: return the current position's score and the best move -}
    scores :: (Score, Score) -> Move -> [Move] -> Search (Score, Move)
    scores (alpha, beta) best_move [] 
        = if is_max_turn then return (alpha, best_move)
                         else return (beta, best_move)
    scores (alpha, beta) best_move (move:moves)
        = do
          let game' = makeMove move game
          score <- if quickLastMoveWin game' then
                      return (if is_max_turn then infinity else -infinity)
                   else
                      do 
                      hashTableScore <- getHashScore game'
                      case hashTableScore of
                          Just (depthSearched, score', _response) 
                              | depthSearched >= depth -> {- debugPutStr "success" $ -} return score'
                          _ -> if depth == 1 then
                                   positionEvaluate prefs game' turn
                               else
                                    do (sc,mv) <-  alphabeta game' (alpha, beta)
                                                                      (depth - 1) prefs turn
                                       -- (main_dep - 5 seems the right amount here)
                                       when (depth >= 0)  
                                            (setHashScore game' depth sc mv)
                                       
                                       return sc
                          

          let (alpha', beta', best_move') 
               | is_max_turn = if score > alpha then (score, beta, move)
                                                   else (alpha, beta, best_move)
               | otherwise   = if score < beta  then (alpha, score, move)
                                                   else (alpha, beta, best_move)
                    
          if (alpha' >= beta') then
              if is_max_turn then return (beta', move) else return (alpha', move)
            else
              scores (alpha', beta') best_move' moves

-}







{-
alphabeta :: Game -> (Score, Score) -> Depth -> 
                      CPUPrefs -> Turn -> Search (Score, Move)
alphabeta game (init_alpha, init_beta) depth prefs turn
    = if legalMoveList == []
      then if is_max_turn then return (-infinity, undefined) 
                          else return (infinity, undefined)
      else scores (init_alpha, init_beta) undefined legalMoveList
    where
    legalMoveList = sortByFun goodness (legalMoves game)
        where
        goodness move =  -1 * (quickGoodnessScore prefs move game)
                                              

    is_max_turn = whoseTurn game == turn

    {- this does the looping through the moves and explicitly stores
       alpha and beta: return the current position's score and the best move -}
    scores :: (Score, Score) -> Move -> [Move] -> Search (Score, Move)
    scores (alpha, beta) best_move [] 
        = if is_max_turn then return (alpha, best_move)
                         else return (beta, best_move)
    scores (alpha, beta) best_move (move:moves)
        = do
          let game' = makeMove move game
          score <- if quickLastMoveWin game' then
                      return (if is_max_turn then infinity else -infinity)
                   else
                      do 
                      hashTableScore <- return Nothing -- getHashScore game'
                      case hashTableScore of
                          Just (depthSearched, score') 
                              | depthSearched >= depth -> {- debugPutStr "success" $ -} return score'
                          _ -> if depth == 1 then
                                   positionEvaluate prefs game' turn
                               else
                                    do (sc,_move) <-  alphabeta game' (alpha, beta)
                                                                      (depth - 1) prefs turn
                                       -- (main_dep - 5 seems the right amount here)
                                       when (depth >= 0)  
                                            (setHashScore game' depth sc)
                                       
                                       return sc
                          

          let (alpha', beta', best_move') 
               | is_max_turn = if score > alpha then (score, beta, move)
                                                   else (alpha, beta, best_move)
               | otherwise   = if score < beta  then (alpha, score, move)
                                                   else (alpha, beta, best_move)
                    
          if (alpha' >= beta') then
              if is_max_turn then return (beta', move) else return (alpha', move)
            else
              scores (alpha', beta') best_move' moves

-}
