module PosEval where

import Types
import Array
import Basics
import Maybe(fromJust, isJust)

{- for debugging mainly: don't worry about the monadic stuff -}
nonMonadicPosEval :: CPUPrefs -> Game -> Turn -> Score
nonMonadicPosEval a b = fst . runSearch . positionEvaluate a b

-- try to get value as close to ideal as possible.
minimise :: Score -> Score -> Score -> Score
minimise ideal gradient value
    = -1 * gradient * (abs (ideal - value))

positionEvaluate :: CPUPrefs -> Game -> Turn -> Search Score
positionEvaluate prefs game turn
    = do
      let score | not (stillOnBoard game (turn, Den)) = -1 * infinity  
                | not (stillOnBoard game (toggle turn, Den)) = infinity  
                | otherwise = basicPositionScore turn - basicPositionScore (toggle turn) +  
                              magic 4 * (posDenRace turn - posDenRace (toggle turn)) +
                              0  
      incPosEval
      return score

      where
         magic :: Int -> Weight
         magic number = (magicNums prefs)!!number

         isItOurTurn :: Bool
         isItOurTurn = whoseTurn game == turn

         maxDist :: Int
         maxDist = height game + width game

         basicPositionScore turn' = magic 5 * sum pieceScores + 
                                    magic 6 * pieceCoherence + 
                                    magic 7 * bothSideReln
            where
            piecesStillOnBoard :: Turn -> [(Piece, Location)]
            piecesStillOnBoard t = [(p, loc) | s <- allMoveableSpecies, 
                                               let p = (t, s),
                                               let loc = findPiece game p,
                                               loc /= Offboard]

            myPiecesOnBoard = piecesStillOnBoard turn'
            oppPiecesOnBoard = piecesStillOnBoard (toggle turn')

            opponentDenLoc = findPiece game (toggle turn', Den)

            -- how many pieces and how close they are to the opponent's den
            pieceScores = [ magic 0 * speciesValue prefs species + 
                                      minimise 0 (magic 1) denDistance 
                                  | ((_, species), pieceLoc) <- myPiecesOnBoard, 
                                  let denDistance = distance pieceLoc opponentDenLoc
                          ]
            
            -- distance between pairs of pieces
            pieceCoherence = if null supportList then 0
                                else minimise (magic 8) (magic 9) (maximum supportList) + 
                                     minimise (magic 10) (magic 11) (minimum supportList) +
                                     minimise (magic 12) (magic 13) (average supportList)
            average dists = (sum dists) `div` (length dists)
            -- to speed up, we shouldn't count each pair twice (as this does)
            supportList = [ distance loc loc' | (p, loc) <- myPiecesOnBoard,
                                                (p', loc') <- myPiecesOnBoard, 
                                                p /= p' ]
            iff x y = x == y

            bothSideReln = sum [ if spec == spec' then 
                                  if ((dist `mod` 2 == 0) `iff` isItOurTurn) then magic 14 else magic 15
                              else if canCaptureSpecies spec spec' then 
                                  (magic 2) * minimise 0 1 dist
                              else
                                  (magic 3) * minimise maxDist 1 dist 
                                        | ((_, spec), loc) <- myPiecesOnBoard,
                                          ((_, spec'), loc') <- oppPiecesOnBoard,
                                          let dist = distance loc loc' ]

         posDenRace turn'
            = -- this is a bit risky -- we might miss some den races, but
              -- generally they will be later than after 10 moves in
              if numMoves game >= 10 && possibleDenRace game then 
                    case denRace game of 
                              Nothing -> 0
                              Just (x, plies) ->
                                (2 * maxDist - plies) * if x == turn' then 1 else (-1)
                    else 0  -- this is to speed things up: hopefully it's not 
                            -- often wrong! 



{- TODO: Make this work again (faster and maybe a _real_ nec. condition) -}
{- provide an almost necessary condition for a denRace to be viable 
   (this is a quicker test to try and discard lots of positions) -}
possibleDenRace :: Game -> Bool
possibleDenRace game = True -- possWin
    where
    colourHomeAway :: Turn -> [(Species, (Int, Int))]
    colourHomeAway turn = [(s, (pliesHome, pliesAway)) | 
                              s <- allMoveableSpecies,
                              let p = (turn, s),
                              let loc = findPiece game p, 
                              loc /= Offboard,
                              let pliesHome = penalty + 2 * distance loc homeDen,
                              let pliesAway = penalty + 2 * distance loc awayDen]
                    where
                    homeDen = findPiece game (turn, Den)
                    awayDen = findPiece game (toggle turn, Den)
                    penalty = if turn == whoseTurn game then 0 else 1

    possWin 
        = let wh_sp_home_aways = colourHomeAway White
              bl_sp_home_aways = colourHomeAway Black
          in  any (maybeWin bl_sp_home_aways) wh_sp_home_aways ||
              any (maybeWin wh_sp_home_aways) bl_sp_home_aways
              where
              maybeWin defenders (s, (_, dist)) = 
                   null [def | (def, (homeDist, _)) <- defenders,
                               canCaptureSpecies def s,
                               homeDist <= dist - 2]
                  


-- return who will win any races and how many plies it will take
denRace :: Game -> Maybe (Turn, Depth)
denRace game
    = case (bestUnstoppable White, bestUnstoppable Black) of 
           (Nothing, Nothing) -> Nothing
           (Just a, Nothing)  -> Just (White, a)
           (Nothing, Just a) ->  Just (Black, a)
           (Just a, Just b) -> 
                if (a `mod` 2 == b `mod` 2) 
                    then error "denRace -- congruent mod 2"
                    else if a < b then Just (White, a) else Just (Black, b) 
                
                                  
    where
    bestUnstoppable colour 
        = let adjSquares = adjacentDenSquares (toggle colour)
              minimum' [] = Nothing
              minimum' xs = Just (minimum xs)
          in minimum' [ (fromJust canWin) | 
                        adj <- adjSquares,
                        s <- allMoveableSpecies,
                        let canWin = unstoppable (colour, s) adj,
                        isJust canWin]

    turn = whoseTurn game

    -- Just x means piece can't be stopped and it's x distance from winning
    unstoppable :: Piece -> Location -> Maybe Int
    unstoppable piece@(colour, species) adjacentSq 
        = let opp = toggle colour 
              -- how many plies it takes to get there:
              adjDist piece' = effectiveDistance game piece' adjacentSq
              myDist = adjDist piece + if colour == turn then 0 else 1
              stoppingPieces = 
                  [ p | s <- allMoveableSpecies, 
                        let p = (opp, s),
                        stillOnBoard game p, 
                        canCaptureSpecies s species,
                        adjDist p <= myDist ]
          in if stillOnBoard game piece && null stoppingPieces
             then Just (2 * adjDist piece + if colour == turn then 1 else 2) 
             else Nothing

    adjacentDenSquares colour 
        = [loc | dir <- allDirections, let loc = adjacentSquare denLoc dir, 
                                       inBounds game loc]
        where denLoc = findPiece game (colour, Den)

    {- returns the distance with an extra penalty if there are pieces in
       the way (in theory -- will return how far away a piece is in moves) -}
    effectiveDistance :: Game -> Piece -> Location -> Int
    effectiveDistance _ _ Offboard
        = error "effectiveDistance for Offboard location"
    effectiveDistance game' piece sq2@(Onboard x2 y2)
        = distance sq1 sq2 + 
            if x1 == x2 then 
                length' (piecesInRange [(x1, y) | y <- [(min y1 y2) + 1 
                                                        .. (max y1 y2) - 1]])
            else if y1 == y2 then
                length' (piecesInRange [(x, y1) | x <- [(min x1 x2) + 1 .. 
                                                           (max x1 x2) - 1]])
            else 
                0
        where 
        length' ((Piece (_,Den)):xs) -- can't be moved: need 2 mvs to get around
            = 2 + length' xs
        length' (_:xs) = 1 + length' xs
        length' [] = 0

        sq1@(Onboard x1 y1) = findPiece game' piece
        piecesInRange = filter (/= Empty) . map ((board game')!)
    
