module Types where

import FiniteMap
import Array 
import Char(toUpper, toLower)

class Show a => SmallShow a where
    smallShow :: a -> String
    smallShow = (:[]) . toLower . head . show

instance SmallShow a => SmallShow [a] where
    smallShow = unwords . map smallShow

-- note: explicit instances of classes are generally at the bottom of the file


data Species = Den | Mouse | Wolf | Tiger | Elephant 
               deriving (Eq, Show, Ix, Ord, Enum)

data Direction = U | D | L | R   -- unfortunately Left/Right clash with Either
                 deriving (Eq, Ix, Ord, Enum) 

data Move = Move Species Direction  -- include who was taken
            deriving (Show, Eq, Ord)

type History = [Move]  

data Turn = White | Black        -- white goes first
            deriving (Eq, Show, Ix, Ord, Enum)

type Piece = (Turn, Species)

data Square = Piece Piece | Empty -- a board square
              deriving Eq

type Board = Array (Int, Int) Square  

-- I know this is bad but I want it to be fast:
type PieceLocs = Pair (Quintuple Location)
type Pair a = (a,a)
type Quintuple a = (a,a,a,a,a)

data Location = Offboard | Onboard Int Int
     deriving (Eq, Ord)

type Weight = Int
type Score = Int
type Depth = Int

data Player = Human | CPU CPUPrefs

instance Show Player where
    show Human = "Human"
    show (CPU prefs) = name prefs
                        

data CPUPrefs = CPUPrefs {
                    name :: String,
                    magicNums :: [Weight],  -- presented in this fashion so
                                            -- that we can learn them more easily
                    {-
                    aggressiveness, defensiveness, denCloseness, 
                        greed :: Weight,

                    forceMateTrust :: Weight,
                    -}

                    chooseSearchDepth :: Game -> Int, 

                    speciesValue :: Species -> Score
                } 

data GameInfo = Info {
                       player :: Turn -> Player
                     }

instance Show GameInfo where
    show info = unlines ["Player 1: " ++ show (player info White),
                         "Player 2: " ++ show (player info Black)]
    
                         

data Game = Game {
                    height, width :: Int,
                    board :: Board,
                    numMoves :: Int,
                    whoseTurn :: Turn,
                    pieceLocs :: PieceLocs,
                    history :: History
                 } 

instance Eq Game where
    game1 == game2
        = same height &&
          same width && 
          same whoseTurn &&
          same pieceLocs
        where
        same property = property game1 == property game2

instance Ord Game where
    game1 >= game2 
        = case compare (pieceLocs game1) (pieceLocs game2) of
               GT -> True
               LT -> False
               EQ -> case compare (whoseTurn game1) (whoseTurn game2) of
                          GT -> True
                          LT -> False
                          EQ -> height game1 >= height game2



-- anything that has an involution of order 2
class Toggle a where
    toggle :: a -> a

instance Toggle Turn where
    toggle White = Black
    toggle Black = White

instance Toggle Direction where
    toggle U = D
    toggle D = U
    toggle L = R
    toggle R = L

{- Not really necessary
instance Toggle Move where
    toggle (Move species dir) = Move species (toggle dir)
    -}


{-------------------+
| Computation monad |
+-------------------}
type GameLookup = (Int,Int)      -- how to represent the game in a finite map

data State = State {
                    numPosEvals :: !Int,  -- positions evaluated (keep strict)
                    hashTable   :: FiniteMap GameLookup (Depth, Score, Move)
                   } 

instance Show State where
    show state
        = "Positions Evaluated: " ++ show (numPosEvals state) ++ "\n" ++
          "Hash table: " ++ showHash (hashTable state)
          where
          showHash table
            = "Size: " ++ show (sizeFM table) ++ "\nContents: " ++ show (fmToList table)

newtype Search a = Search (State -> (a, State))

instance Monad Search where
    return x = Search (\state -> (x, state))

    (Search comp) >>= fun
        = Search (\state ->
                   let (res, state') = comp state 
                       Search comp'  = fun res
                   in comp' state'
                 )

runSearch :: Search a -> (a, State)
runSearch (Search comp) 
    = comp initState where initState = State { numPosEvals = 0,
                                               hashTable = emptyFM }




{----------------------------------------+
| Instances and very primitive functions |
+----------------------------------------}
-- these are kept in types because we have information hiding so we can 
-- change the type and we need to change only these functions
findPieceInLocs :: PieceLocs -> Piece -> Location
findPieceInLocs piecelocs (colour, species)   
    = case colour of
           White -> findSpecies species (fst piecelocs)
           Black -> findSpecies species (snd piecelocs)
      where
           findSpecies Den (x,_,_,_,_) = x
           findSpecies Mouse (_,x,_,_,_) = x
           findSpecies Wolf (_,_,x,_,_) = x
           findSpecies Tiger (_,_,_,x,_) = x
           findSpecies Elephant (_,_,_,_,x) = x

setPieceLoc :: Piece -> Location -> PieceLocs -> PieceLocs
setPieceLoc (colour, species) newval piecelocs 
    = case colour of
           White -> (alter species (fst piecelocs), snd piecelocs)
           Black -> (fst piecelocs, alter species (snd piecelocs))
      where
           alter Den (_,b,c,d,e) = (newval,b,c,d,e)
           alter Mouse (a,_,c,d,e) = (a,newval,c,d,e)
           alter Wolf (a,b,_,d,e) = (a,b,newval,d,e)
           alter Tiger (a,b,c,_,e) = (a,b,c,newval,e)
           alter Elephant (a,b,c,d,_) = (a,b,c,d,newval)

readBoardSquare :: Board -> Location -> Square
readBoardSquare brd (Onboard a b) = brd!(a,b)
readBoardSquare _   Offboard = error "Can't readBoardSquare offboard"

writeBoardSquares :: Board -> [(Location, Square)] -> Board
writeBoardSquares board locPieces
    = board // (updates locPieces)
    where
    updates [] = []
    updates ((Offboard,_) : xs) = updates xs
    updates (((Onboard a b), piece) : xs)
        = ((a,b), piece):updates xs

allPieces :: [Piece]
allPieces = [(col, spec) | col <- allTurns, spec <- allSpecies]

allMoveablePieces :: [Piece]
allMoveablePieces = [(col, spec) | col <- allTurns, spec <- allMoveableSpecies]

allTurns :: [Turn]
allTurns = [White, Black]

allSpecies :: [Species]
allSpecies = [Den .. Elephant]

allMoveableSpecies :: [Species]
allMoveableSpecies = tail allSpecies  -- don't include Den

allDirections :: [Direction]
allDirections = [U .. R]

-- includes the illegal ones as well!
allMoves :: [Move]
allMoves = [Move spec dir | spec <- allMoveableSpecies, dir <- allDirections]


instance Show Game where
    show game = 
        "Turn:         " ++ show (whoseTurn game) ++ "\n" ++ 
        "Moves played: " ++ show (numMoves game) ++ "\n" ++
        "\n" ++
        -- showPieceLocs (pieceLocs game) ++ "\n\n\n"
        showBoard (board game)  ++
        ""
        where
        {-
        showPieceLocs :: PieceLocs -> String
        showPieceLocs locs 
            = unwords [show (Piece p) ++ "@" ++ show (findPieceInLocs locs p) |
                         p <- allPieces ]
        -}

        showBoard :: Board -> String
        showBoard board = bigUnlines $ map unwords brd
            where
                ((1,1), (height,width)) = bounds board  -- find out board dims 
                brd = [ [" " ++ show (board!(x,y)) ++ " " | 
                        y <- [1 .. width]] | x <- [1 .. height] ]
                bigUnlines = concatMap (\line -> line ++ "\n\n")


instance SmallShow Species
instance SmallShow Direction
instance SmallShow Turn

instance SmallShow Move where
    smallShow (Move a b) = smallShow a ++ smallShow b

instance SmallShow Game where
    smallShow game =
        "Turn: " ++ smallShow (whoseTurn game) ++ "\n" ++
        smallShowBoard (board game)
        where
            -- display a smaller version of the board
         smallShowBoard :: Board -> String
         smallShowBoard board = unlines $ map unwords brd
             where
                 ((1,1), (height,width)) = bounds board
                 brd = [ [myshow (board!(x,y)) |
                         y <- [1 .. width]] | x <- [1 .. height] ]
                 myshow = (:[]) . (!!1) . show  -- nice obfuscated stuff :)



instance Show Square where
    show (Piece (White,Den)) = "/D\\"
    show (Piece (Black,Den)) = "\\d/"
    show (Piece (turn,species))
        = case turn of 
               White -> ['=', toUpper (char species), '=']
               Black -> ['-', char species, '-']
        where 
        char Mouse = 'm'
        char Wolf  = 'w'
        char Tiger = 't'
        char Elephant = 'e'
        char Den   = error "Handled in the above case for Show Square"
    show Empty     = " . "

instance Show Direction where
    show U = "up"
    show D = "down"
    show L = "left"
    show R = "right"

instance Show Location where
    show Offboard = "off"
    show (Onboard x y) = "(" ++ show x ++ "," ++ show y ++ ")"



