module Basics where

import Types
import FiniteMap 
import Set
import Maybe(fromJust)

defaultEnvironment :: Env
defaultEnvironment
    = Env { {- primDomains = listToFM [("X", map Obj [1..3]),("Y", map Obj [1..4]) ], -}
            identTypes  = 
              listToFM 
               ([(v, Domain "X") | v <- ["x", "y", "z"]] ++
                [(v, binrel (Domain "X")) | v <- ["R", "S", "T"]] ++
                [(v, property (Domain "X")) | v <- ["A","B","P","Q"]] ++
                [(v, func (Domain "X") (Domain "X")) | v <- ["f", "g"]]),
            identVals = 
              listToFM [("X", SetOf (mkSet (map Obj [1..3])))] 
           }
      where
      func x y = FunctionType (x,y)
      binrel dom = func (Product [dom,dom]) Booleans
      property dom = func dom Booleans

-- given an environment and a setting command, return new env
applyNewSetting :: Env -> Setting -> IO Env
applyNewSetting env (Assign vname expr) 
    = do val <- eval env expr
         return $ env { identVals = addToFM (identVals env) vname val }
applyNewSetting env (DefaultType typeSpec) 
    = return $ env { identTypes = addListToFM (identTypes env) 
                          (map (\(x,y) -> (x,fromJust y)) typeSpec) }
       -- ignore any Nothing's (we probably should give an error) 

-- the main function: evaluate an expression in an environment
eval :: Env -> Expr -> IO Object
eval _   (Basic obj)       = return obj
eval env (Var varName)     = return (myLookupFM (identVals env) varName )
eval _   (PrintString str) = putStrLn str >> return (Boolean True)
eval env (Print expr)   = 
    do obj <- eval env expr
       printOut obj
       return (Boolean True)
eval env (MakeSetting setting expr) =
    do env' <- applyNewSetting env setting 
       eval env' expr

eval env (Apply fun arg) 
    = do
      fun' <- eval env fun
      arg' <- eval env arg
      return (applyFun fun' arg')
eval env (Builtin builtin args) = evalBuiltin builtin env args
eval env (Quantifier q typeSpec expr)
    = chooseQuant (\vals ->
                do {eval (env' vals) expr}
              ) (genCartProduct (map (listOf env) objTypes))
    where
    chooseQuant | q == All = myAllIO
                | q == Exists = myAnyIO
                | q == Find = myFindIO
                | q == FindAll = myFindAllIO
                | q == ForEach = myForEachIO
                | otherwise = error "undefined quantifier"

    niceTypeSpec = lookupTypes env typeSpec

    (varNames, objTypes) = unzip niceTypeSpec

    envWithTypes :: Env
    envWithTypes 
        = env { identTypes = addListToFM (identTypes env) niceTypeSpec }

    env' :: [Object] -> Env
    env' vals = envWithTypes { 
        identVals = addListToFM (identVals envWithTypes) (zip varNames vals) }

    myAllIO _ [] = return (Boolean True)
    myAllIO fun (x:xs)
        = do ans <- fun x
             if ans == Boolean False then return (Boolean False)
                                     else myAllIO fun xs

    myAnyIO _ [] = return (Boolean False)
    myAnyIO fun (x:xs)
        = do ans <- fun x
             if ans == Boolean True then return (Boolean True)
                                    else myAnyIO fun xs

    myForEachIO fun xs = mapM fun xs >> return (Boolean True)
    {-
    myForEachIO _ [] = return (Boolean True)
    myForEachIO fun (x:xs)
        = do fun x
             myForEachIO fun xs
             -}


    -- like Any, but prints out first one that it works for
    myFindIO _ [] = return (Boolean False)
    myFindIO fun (x:xs)
        = do ans <- fun x
             if ans == Boolean True then do printOutValn x
                                            return (Boolean True)
                                    else myFindIO fun xs

    myFindAllIO fun vals = myFindAllIO' vals False
        where
        -- acc keeps track of whether we actually found a soln
        myFindAllIO' [] acc = return (Boolean acc)
        myFindAllIO' (x:xs) acc
          = do ans <- fun x
               acc' <- if ans == Boolean True 
                       then printOutValn x >> return True
                       else return acc
               myFindAllIO' xs acc'

    printOutValn :: [Object] -> IO ()
    printOutValn vals = mapM_ printOutVar (zip varNames vals)



evalBuiltin :: Builtin -> Env -> [Expr] -> IO Object
evalBuiltin MkTuple env exprs
    = do exprs' <- mapM (eval env) exprs
         return (case exprs' of [obj] -> obj -- don't allow 1-ary tuples
                                _     -> TupleOf exprs')
evalBuiltin IfThenElse env (ifExpr:thenExpr:rest)
    = do ifExpr' <- eval env ifExpr
         if (ifExpr' == Boolean True)
             then eval env thenExpr
             else if (rest == [])
                      then return (Boolean True) -- just a no-op
                      else eval env (head rest)  -- evaluate else part
evalBuiltin IfThenElse _ _ = error "Need two or three args for IfThenElse"

evalBuiltin Equal env xs
    = do [x',y'] <- mapM (eval env) xs
         return (Boolean (x' == y'))
evalBuiltin Elem env xs
    = do [x', SetOf xs'] <- mapM (eval env) xs
         return (Boolean (elementOf x' xs'))
evalBuiltin Not env xs
    = do [Boolean x'] <- mapM (eval env) xs
         return (Boolean (not x'))
evalBuiltin Iff env xs
    = do [x',y'] <- mapM (eval env) xs
         return (Boolean (x' == y'))
evalBuiltin Compose env xs
    = do xs' <- mapM (eval env) xs
         return (evalCompose xs')
        where   
        evalCompose [Function fInt f, Function gInt g]
          | f_ran == Booleans && g_ran == Booleans
            = case (f_dom, g_dom) of 
              (Product [x,y], Product [_,z]) ->
                 Function (fInt {getType = (Product [x,z], Booleans),
                         getDomain = mkSet (listOf env (Product [x,z]))})
                    (\(TupleOf [obj1,obj2]) ->
                       myAny (\val -> 
                        Boolean (fromBoolean (f (TupleOf [obj1,val])) && 
                             fromBoolean (g (TupleOf [val,obj2])))
                       ) (listOf env y)
                    )
              _ -> error "Can only compose relations if they are both binary"
          | otherwise =
            Function (fInt {getType = (g_dom, f_ran)}) 
                 (\obj -> f (g obj))
            where 
            (f_dom, f_ran) = getType fInt
            (g_dom, g_ran) = getType gInt
        evalCompose _ 
          = error $ "Can only compose binary relations and functions"
-- lazy functions
evalBuiltin And env xs
    = do [x,y] <- return xs
         (Boolean x') <- eval env x
         (Boolean y') <- if x' then eval env y else return (Boolean undefined)
         return (Boolean (x' && y'))

evalBuiltin Or env xs
    = do [x,y] <- return xs
         (Boolean x') <- eval env x
         (Boolean y') <- if x' then return (Boolean undefined) else eval env y 
         return (Boolean (x' || y'))
evalBuiltin Imp env xs 
    = do [x,y] <- return xs
         evalBuiltin Or env [Builtin Not [x], y]

printOut :: Object -> IO ()
printOut obj = putStrLn (niceShow obj)

printOutVar :: (VarName, Object) -> IO ()
printOutVar (name, val) = putStr (name ++ ":\n") >> printOut val

-- approximate the number of steps this calculation will take
-- (obviously this will be an upper bound because of lazy evaluation)
numberSteps :: Env -> Expr -> IO Integer
numberSteps env expr
    = case expr of Basic _ -> return 1
                   Var   _ -> return 1
                   Quantifier q typespec expr -> 
                      do expr' <- num expr
                         return ((cardTypeSpec env typespec) * expr')
                   Apply x y -> 
                      do x' <- num x 
                         y' <- num y
                         return (x' + y')
                   Builtin _ xs -> 
                      do xs' <- mapM num xs
                         return (sum xs') -- this isn't quite right
                                          -- for IfThenElse
                   -- e.g. changing the domain size, which has a huge
                   -- effect, so we apply the new change to the env.
                   MakeSetting setting@(DefaultType _) x -> 
                      do env' <- applyNewSetting env setting
                         numberSteps env' x
                   MakeSetting setting@(Assign _ expr') x ->
                      do a' <- num expr'
                         env' <- applyNewSetting env setting
                         x' <- numberSteps env' x
                         return (a' + x')
                   Print expr -> num expr
                   PrintString _ -> return 1
    where
    num = numberSteps env

cardTypeSpec :: Env -> TypeSpecification -> Integer
cardTypeSpec env typespec
    = product (map (cardType env) (map snd (lookupTypes env typespec)))
    -- all possible combos on the given types

-- given an environment and a type, count how many objects will be in that type
-- (intended semantics: card env = length . listOf env [but much faster])
cardType :: Env -> ObjectType -> Integer
cardType env tp
    = case tp of (FunctionType (a,b)) -> (card b)^(card a)
                 (Power t) -> 2^(card t)
                 (Product ts) -> product (map card ts)
                 Booleans -> defaultVal
                 Domain _ -> defaultVal
    where
    card = cardType env
    defaultVal = toInteger (length (listOf env tp)) 

-- given an environment and a type, lists all objects in that type
listOf :: Env -> ObjectType -> [Object]
listOf _ Booleans = [Boolean False, Boolean True]
listOf env (Domain domName)
    = case lookupFM (identVals env) domName of
             Just (SetOf vals) -> setToList vals
             Just _ -> error $ "Domain " ++ domName ++ " is not a set"
             Nothing -> error $ "Couldn't find " ++ domName ++ " as a domain"
listOf env (FunctionType types@(argType,resultType))
    = let domain = listOf env argType
          range  = listOf env resultType
          domSize = length domain
      in map (pairsToFunction types) $ map (zip domain) (combos range domSize)
listOf env (Power objType)
    = map SetOf (map mkSet (powerset (listOf env objType)))
listOf env (Product domains)  
    = map TupleOf (genCartProduct (map (listOf env) domains))

{-- basic utilities --}

powerset        :: [a] -> [[a]]
powerset []     = [[]]
powerset (x:xs) = let rest = powerset xs in map (x:) rest ++ rest


{- returns all possible combos of values from the domain of length n:
   e.g. combos [0,1] 2 = [[0,0],[0,1],[1,0],[1,1]] -}
combos       :: [a] -> Int -> [[a]]
combos dom n = genCartProduct (replicate n dom)

-- Generalised cartesian product.
-- So genCartProduct [[1,2], [2,4,5], [3]] 
--     ==> [[1,2,3], [1,4,3], [1,5,3], [2,2,3],[2,4,3],[2,5,3]]
genCartProduct :: [[a]] -> [[a]]
genCartProduct [] = [[]]
genCartProduct (dom:doms)
    = let rest = genCartProduct doms in
          concat [map (x:) rest | x <- dom]

myLookupFM :: (Show key, Show elt, Ord key) => FiniteMap key elt -> key -> elt
myLookupFM fm x
    = case lookupFM fm x of 
           Just y -> y
           Nothing -> 
            error $ "Can't find " ++ show x ++ " in " ++ show fm

applyFun :: Object -> Object -> Object
applyFun (Function _ f) x = f x
applyFun f x = error $ "Could not apply " ++ show f ++ " to " ++ show x

fromBoolean :: Object -> Bool
fromBoolean (Boolean x) = x
fromBoolean x = error $ "Can't fromBoolean on " ++ show x

myAny :: (a -> Object) -> [a] -> Object
myAny _ [] = Boolean False
myAny cond (x:xs) = Boolean ((cond x == Boolean True) 
                            || (myAny cond xs == Boolean True))

-- this is where we look up all the Nothing types and convert
-- them to what the environment says
lookupTypes :: Env -> TypeSpecification -> [(VarName, ObjectType)]
lookupTypes env typeSpec = map convert typeSpec
    where
    convert (v, Just x) = (v,x)
    convert (v, Nothing) = (v, myLookupFM (identTypes env) v)
