module Parser(exprParse) where

import MyParseLib
import ExprParseLib
import Types
import Set

exprParse :: String -> Expr
exprParse str
    = case papply parseExpr str of
           [(expr,"")] -> expr
           [] -> error "Couldn't parse any of the input"
           xs -> error $ "Parse error: " ++ show xs

keywords :: [String]
keywords = ["all","findall","find","some","any", "each", "foreach", "for",
            "forall","exists","print",
            "elem","member","in",
            "printstr","AND","and","or","iff","implies","if","not",
            "let", "set", "setting"]

parseExpr :: Parser Expr
parseExpr = parse $  -- get rid of leading junk
    parseSettings $
    -- AND is low precedence so can be used to join lines
    binopl [("AND", \x y -> Builtin And [x,y])] $ 
    quantParser FindAll ["findall", "find all"] $ 
    quantParser Find ["find"] $
    quantParser ForEach ["foreach", "for each", "for"] $ 
    parseIfThenElse $
    englishConnectives $
    quantParser All ["all","forall"] $
    quantParser Exists ["exists","some","any"] $
    symbolConnectives  $
    binopl [(a,\x y -> Builtin Equal [x,y]) | a <- ["==","="]] $
    binopl [(a,\x y -> Builtin Not [Builtin Equal [x,y]]) | 
                a <- ["/=","!=","~="]] $
    binopl [(a,\x y -> Builtin Elem [x,y]) | a <- ["elem","member","<-"]] $
    printStringParse $
    prefix [("print", Print)] $
    binopr [(a, \x y -> Builtin Compose [x,y]) | a <- [".","*"]] $
    parseApplication $
    baseCase
    where
    baseCase = bracketed parseExpr +++ 
               parseTuple +++  -- this one could take over the first's work
               (parseObject >>= return . Basic) +++
               (parseVarName >>= return . Var)

    {- let X = {1,2,3}  or  
       let X = 3 (means 3 elts)  or  
       let a,b:X, c,d:Y -}
    parseSettings p
        = (do (symbol "let" +++ symbol "set" +++ symbol "setting")
              setting <- parseSetting
              (symbol "in" +++ return undefined) -- optional
              --rest <- (parseSettings p +++ p)  -- check also for more settings
              rest <- parseExpr
              return (MakeSetting setting rest)) +++ p
    parseSetting
        = (do var <- parseVarName
              symbol "="
              expr <- parseExpr
              return (Assign var expr))
           +++ (do x <- parseTypeSpec
                   return (DefaultType x))
                   

    parseIfThenElse p
        = (do symbol "if"
              ifCond <- p
              symbol "then"
              thenCond <- p
              (do symbol "else"  
                  elseCond <- p
                  return (Builtin IfThenElse [ifCond, thenCond, elseCond])
               ) +++ return (Builtin IfThenElse [ifCond, thenCond])) +++ p

    englishConnectives p = 
        binopr [("iff", \x y -> Builtin Iff [x,y])] $
        binopl [("implies", \x y -> Builtin Imp [x,y])] $
        binopl [("and", \x y -> Builtin And [x,y])] $
        binopl [("or", \x y -> Builtin Or [x,y])] $ 
        prefix [("not", \x -> Builtin Not [x])] $ p

    symbolConnectives p = 
        binopr [("<->", \x y -> Builtin Iff [x,y])] $
        binopl [("->", \x y -> Builtin Imp [x,y])] $
        binopl [(s, \x y -> Builtin And [x,y]) | s <- ["&&", "&"]] $
        binopl [(s, \x y -> Builtin Or [x,y]) | s <- ["||", "|"]] $
        prefix [(s, \x -> Builtin Not [x]) | s <- ["~", "!"]] $ p
    
    parseApplication :: Parser Expr -> Parser Expr
    parseApplication subParser
        = do fun <- subParser
             (do args <- parseTuple
                 return (Apply fun args)) +++ return fun
    
    parseTuple :: Parser Expr
    parseTuple  
        = do
          args <- bracket (symbol "(") (parseCommaSep parseExpr) (symbol ")")
          let expr = case args of [x] -> x
                                  _   -> Builtin MkTuple args
          return expr

    printStringParse :: Parser Expr -> Parser Expr
    printStringParse restParse =  -- note printString has no Expr children
        token $ 
            (do symbol "printstr"
                str <- token (bracket (symbol "\"") inStr (symbol "\""))
                return (PrintString str)) +++ restParse
        where
        inStr = many (sat (/= '"'))

    quantParser :: Quan -> [String] -> Parser Expr -> Parser Expr
    quantParser quant possNames subParser
        = (do foldl1 (+++) (map symbol possNames) -- look for initial quant name
              typespec <- parseTypeSpec
              body <- (subParser +++ parseExpr)  -- this seems a bit dodgy
              return (Quantifier quant typespec body)
          ) +++ subParser

-- this is for things like [x,y:X, f: X -> X, a, b] etc.
-- although now the brackets are optional
parseTypeSpec :: Parser TypeSpecification
parseTypeSpec = 
    do parsedForm <- possBracketed (parseCommaSep parseOneSpec)
       return (convertTypeSpec parsedForm)
    where
    parseOneSpec = do varNames <- parseCommaSep parseVarName
                      typeGroup <- parsePossType
                      return (varNames, typeGroup)
    parsePossType = (do symbol ":" 
                        tp <- parseObjectType
                        return (Just tp)) +++ return Nothing

    -- convert the parsed type specifications into a simpler form
    convertTypeSpec typespec = concatMap convert' typespec
    convert' (vars, tp) = map (\var -> (var,tp)) vars

parseObjectType :: Parser ObjectType
parseObjectType = 
    binopr [ ("->", \x y -> FunctionType (x,y)) ] $
    parseCartProduct $ 
    parseCartPower $ -- e.g. X^4 = X * X * X * X
    -- binopr [ ("*", \x y -> Product [x,y]) ] $
    prefix [ ("2^", Power), ("Pow", Power), ("Subset", Power) ] $
    prefix [ ("Property", \x -> FunctionType(x,Booleans)) ] $
    prefix [ ("Relation", \x -> FunctionType(Product [x,x], Booleans))] $
    factor
    where 
    factor = bracketed parseObjectType +++ bool +++ domain
    bool = nullary [("Boolean", Booleans), ("Bool", Booleans)]
    domain = do dom <- parseVarName
                return (Domain dom)
    parseCartProduct p
        = do objTypes <- p `sepby` (symbol "*")
             return (case objTypes of [x] -> x
                                      _ -> Product objTypes)

    parseCartPower p
        = do tp <- p
             (do symbol "^"
                 n <- natural
                 return (Product (replicate n tp))
              ) +++ return tp

             

-- don't parse TupleOf's here because they should become MkTuple's
parseObject :: Parser Object
parseObject = parseBool +++ parseInt +++ parseSet
    where
    parseBool = (symbol "T" >> return (Boolean True)) 
            +++ (symbol "F" >> return (Boolean False))
    parseInt  = do x <- natural
                   return (Obj x)
    parseSet  = do xs <- bracket (symbol "{") 
                                 (parseCommaSep parseObject) 
                                 (symbol "}")
                   return (SetOf (mkSet xs))


parseList :: Parser a -> Parser [a]
parseList argParser = bracketed (parseCommaSep argParser)

parseCommaSep :: Parser a -> Parser [a]
parseCommaSep argParser = argParser `sepby` (symbol ",")

bracketed :: Parser a -> Parser a
bracketed parser = bracket (symbol "(") parser (symbol ")") +++ 
                   bracket (symbol "{") parser (symbol "}") +++
                   bracket (symbol "[") parser (symbol "]")

possBracketed :: Parser a -> Parser a
possBracketed p = bracketed p +++ p

parseVarName :: Parser VarName
parseVarName = token $ do x <- letter
                          xs <- many alphanum
                          let str = x:xs
                          ans <- 
                               if str `elem` keywords
                                then sat (const False) >>= return . undefined
                                else return str
                          return ans
                          
