module CPUUtils where

import Random
import Types
import Utilities
import Array
import Numeric


--minimax :: Plies -> CPUTendencies -> Game -> 
-- IO in case CPUTendencies introduce randomness in later programming
minimax :: CPUTendencies -> Game -> IO Column
minimax tendencies game =
    do
    let plies = if ((width game)*(height game) - (startEndGame tendencies) 
                        <= numMoves game)
                then 
                   endGameDepth tendencies -- we're in the end game now!
                else
                   mainDepth tendencies    -- main part of game still.
        --(scoreMove, column) = minimax' plies tendencies game {- (turn game) -}
        alpha' = -10000000000
        beta'  = 10000000000
        (scoreMove, column) 
            = alphabeta game (alpha', beta') plies tendencies (turn game)
    putStrLn $ "Expected score is " ++ show scoreMove

    return column

-- this is highly experimental and for speed -- it's also very ugly!!
alphabeta :: Game -> (Score, Score) -> Plies -> 
                     CPUTendencies -> Turn -> (Score, Column)
alphabeta game (alpha, beta) depth tendencies perspective
    = scores (alpha, beta) 1 legalMoveList
    where
    legalMoveList = [m | m <- [4,3,5,2,6,1,7], columnNotFull game m,
                                               near_last_move m]

    is_max_turn = turn game == perspective
    last_move   = case history game of ((Place _ col):_) -> col
                                       _                 -> 4
    near_last_move col = abs (col - last_move) <= 4 
    -- otherwise don't consider (to keep things fast!)

    scores :: (Score, Score) -> Column -> [Column] -> (Score, Column)
    scores (alpha, beta) best_move [] = if is_max_turn then (alpha, best_move) 
                                                       else (beta, best_move)
    scores (alpha, beta) best_move (col:cols)
        = let game' = move col game 
              inf   = 100000000 
              score = if lastMoveWin game' then 
                         (if is_max_turn then inf else -inf)
                      else
                          if (depth == 1) then
                              positionEvaluation tendencies game' perspective
                          else 
                              fst $ alphabeta game' (alpha, beta) 
                                             (depth - 1) tendencies perspective
              new_alpha = if is_max_turn && score > alpha 
                           then score else alpha
              new_beta = if not is_max_turn && score <= beta
                          then score else beta
              new_best_move = if alpha /= new_alpha || beta /= new_beta 
                               then col else best_move
           in 
           if (alpha >= beta) then
               (if is_max_turn then beta else alpha, col)
           else
               scores (new_alpha, new_beta) new_best_move cols   



-- return the best move and best score from the point of view of the player
-- on turn
minimax' :: Plies -> CPUTendencies -> Game -> (Float, Column)
minimax' plies tendencies game = 
    let 
        -- how good going in column col is for the player
        score col = 
            let game' = move col game 
                perspective = turn game
                infinity = 10000000 in
            if lastMoveWin game' then infinity  -- obviously
            else 
                if (plies == 1) then
                    positionEvaluation tendencies game' perspective
                else let (s,_) = minimax' (plies - 1) tendencies game' 
                     in (-1) * s  -- the opponent's best score

        scores = [ (score m, m) | m <- (legalMoves game) ]
    in
    if null scores then error $ "No legal moves for me to make\n" ++ show game
                   else maximum scores



{- given some tendencies (like desire for 3s in row, defensiveness etc.)
   a game, and a perspective to analyse the game from, return a float
   representing the goodness of the position: Larger numbers -> better pos -}
positionEvaluation :: CPUTendencies -> Game -> Turn -> Float
positionEvaluation tendencies game perspective =  -- perspective is X/O 
    let stats' = allBlocksCalculate game          -- (whichever we are)
        swap (x,y) = (y,x)
        (us, them) = if (perspective == O) then swap stats' else stats'
        evaluateOneSide side = 
            let [lr, vert, diag1, diag2] =  -- fromInt is hugs-specific
                    (map (map fromIntegral) side) :: [[Float]]
            in  sum (zipWith (*) lr (leftRight tendencies)) + 
                sum (zipWith (*) vert (upDown tendencies)) +
                sum (zipWith (*) diag1 (diagonal tendencies)) +
                sum (zipWith (*) diag2 (diagonal tendencies))
    in
        (evaluateOneSide us) -  
            (defensiveness tendencies) * (evaluateOneSide them)
    

{- allBlocksCalculate returns a list of pairs [leftRight, vertical, ...]
   Each pair = (numberXs, numberOs) and each numberXs = [a,b,c,d] where
   a = number of blocks in this direction (leftRight or whatever) with
   exactly 1 X in them, b = ... with 2 Xs, ..., d = number of 4s in this
   direction
-}
allBlocksCalculate :: Game -> ([[Int]], [[Int]])
allBlocksCalculate game = unzip (map counter list)  
    -- todo: make it so we don't need to unzip, it shows the bad design
    where 
        list = [map blockCalculate x | x <- allBlocks game]
        incrementIndex n []     = error "incrementIndex []"
        incrementIndex 0 (x:xs) = (x+1):xs
        incrementIndex n (x:xs) = x:(incrementIndex (n-1) xs)
            

        counter :: [Maybe (Turn, Int)] -> ([Int], [Int])
        counter [] = ([0,0,0,0],[0,0,0,0])
        counter (Nothing:xs) = counter xs
        counter ((Just (turn,cnt)):xs) =
            let (xsRest, osRest) = counter xs in
            if (turn == X) then (incrementIndex (cnt-1) xsRest, osRest)
                else (xsRest, incrementIndex (cnt-1) osRest)
            

{- calculate how many Xs/Os are in a block of four (if they are the only
   ones in it, that is). 
   Examples:
       1. [X,X,-,X] -> Just (X, 3) 
       2. [X,O,X,-] -> Nothing (because X and O are both in it)
       3. [-,-,-,-] -> Nothing (because nobody owns it)
       4. [-,O,-,-] -> Just (O, 1)
-}
blockCalculate :: [Square] -> Maybe (Turn, Int)
blockCalculate blocks = blockCalculate' Empty 0 blocks
    where
    blockCalculate' Empty 0 [] = Nothing  -- there were no pieces in the block
    blockCalculate' Empty 0 (x:xs) =
        case x of Piece turn -> blockCalculate' x 1 xs
                  Empty      -> blockCalculate' x 0 xs
    blockCalculate' (Piece turn) num [] = Just (turn, num)
    blockCalculate' piece@(Piece turn) num (x:xs)
        | x == Piece (toggle turn) = Nothing -- we now have both colours in same block
        | x == piece = blockCalculate' piece (num+1) xs
        | otherwise = blockCalculate' piece num xs



{- returns a whole pile of square-blocks representing all the possible fours
   on the board: see code for more details -}
allBlocks :: Game -> [[[Square]]]
allBlocks game =
    let w = width game
        h = height game
        brd = board game
        leftRight = 
            [ [brd!(row,col + i) | i <- [ 0 .. 3]] | 
            col <- [1 .. w - 3], row <- [1 .. h]]
        vertical = 
            [ [brd!(row + i, col) | i <- [0 .. 3]] |
            col <- [1 .. 7], row <- [ 1 .. h - 3 ] ]
        topRbotL =
            [ [brd!(row + i, col - i) | i <- [0 .. 3]] | 
            col <- [4 .. w], row <- [1 .. h - 3]]
        topLbotR = 
            [ [brd!(row + i, col + i) | i <- [0 .. 3]] | 
            col <- [1 .. w - 3], row <- [1 .. h - 3]]
    in
        [leftRight, vertical, topRbotL, topLbotR] 


{- findForceMate takes a maximum number of plies to look for mate,
   a game and returns Nothing (meaning no forced mate was there)
   or Just (col, p) meaning that a move in column col will lead to
   mate in at most p moves if we play perfectly -}
findForceMate :: Plies -> Game -> Maybe (Column, Plies)
findForceMate plies game = maybeHead winMinPlies
    where
        -- for calculating the number of moves before victory is ours.
        winMinPlies = [(win, minMoves win) | win <- wins] 

        {- given a move, what is the minimum number of plies that 
           making that move will lead to mate in? -}
        minMoves move = head [ p | p <- [1, 3 .. plies], forceMate p game move ]

        -- wins = the moves that will force a mate in plies plies or fewer
        wins        = filter (forceMate plies game) allMoves
        allMoves    = [1 .. (width game)]
        maybeHead []     = Nothing
        maybeHead (x:xs) = Just x


{- True if within p plies from the current game, a move in column
   will allow us to force mate.
     For speed reasons, it doesn't check whether a mate has already occurred,
   so in some senses isn't correct, but who wants correctness if it slows a 
   program down? -}
forceMate :: Plies -> Game -> Column -> Bool
forceMate p     _    _   | p <= 0 = False
forceMate plies game col =
    not (columnFull game col) &&
    (lastMoveWin game' || goOnToWin)
    where
        game' = move col game  -- after move is made
        legalMoves g = [ col | col <- [1 .. (width g)], 
                               not (columnFull g col) ]
        goOnToWin =
            all (\response ->
                let game''          = move response game' 
                    opponentVictory = lastMoveWin game'' 
                in
                {- if the opponent sneaks in with a quick win
                   then our move is really not that good now, is it? -}
                    not (opponentVictory) &&  
                    any (\myResponse ->
                        forceMate (plies - 2) game'' myResponse)
                       (legalMoves game'')
                ) (legalMoves game')


{- TODO: probably this should check that the board isn't full! -}
randMove :: Game -> IO Column
randMove game = 
    do
        move <- rand 1 maxCol
        if (legal move) 
            then return move 
            else randMove game
    where
    legal col = not (columnFull game col)
    maxCol    = width game

-- rand takes a range and returns a random integer in that range
rand :: Int -> Int -> IO Int
rand low high =
    do f <- getStdRandom (randomR (low, high)) :: IO Int
       return f

{- e.g. 4 plies is 2 moves -- I don't know if it should be p/2+1 or p/2-}
pliesToMoves :: Plies -> Int
pliesToMoves p = p `div` 2 + 1



{- return Just x if moving in x either wins or prevents a loss (in that order)
   and Nothing otherwise: suggested that you call this function at the start
   of a computer personality -}
lookForMates :: Game -> Maybe Column
lookForMates game =
    let legalMove col = not (columnFull game col)
        myPossibleWins  = [ col |  
                            col <- [1 .. (width game)], 
                            legalMove col && lastMoveWin (move col game)] 
        opp = toggle (turn game)  -- the computer's opponent
        oppPossibleWins = 
            [ col | col <- [1 .. (width game)],
              legalMove col &&
              lastMoveWin (makeMove (Place opp col) game{strictRules = False}) ]
    in
        if not (null myPossibleWins)
            then Just (head myPossibleWins)
            else if not (null oppPossibleWins)
                 then Just (head oppPossibleWins)
                 else Nothing


