module Interaction where

-- handles what the user sees and plays with (for the main loop primarily)

import Types
import Utilities
import CPU
import Array
import IO
import Char
import List(isPrefixOf)

playGame :: Game -> IO ()
playGame game = 
    do
    if lastMoveWin game  
        then do
        print game
        putStrLn $ 
            "**********\nFour in a row.\n" ++ 
            show (playerOnTurn) ++ 
            " wins\nGame over.\n**********"
        command <- promptHuman
        handleCommand command
        else 
            if drawnGame game 
                then do print game
                        putStrLn "**********\nDraw\nGame over.\n**********"
                        command <- promptHuman
                        handleCommand command
                else do print game
                        command <- getCommand game playerOnTurn 
                        handleCommand command
    where
    handleCommand command = case command of
        MakeMove (Place _ col) -> continueGame game col
        QuitProg               -> do putStrLn $ "Thanks for playing!"
                                     return ()
        NoParse str            -> do putStrLn $ "Pardon? [" ++ str ++ "]"
                                     reprompt
        Help                   -> do putStrLn "Connect 4 by Bryn Humberstone."
                                     reprompt
        NewGame                -> playGame (newGameSameProps game)
        ShowBoard              -> do print game
                                     reprompt
        SaveGame fname         -> do saveGame game fname
                                     reprompt
        LoadGame fname         -> do game' <- loadGame game fname
                                     playGame game'
        _                      -> do putStrLn "I should implement that soon!"
                                     reprompt
    reprompt = playGame game 
    playerOnTurn = if (turn game == X) then playerX game else playerO game
    promptHuman = getCommand game (Human "Human")
    
    -- if the user has typed in a move then we continue the game, else
    -- we do something else (like quit/start a new game)
    continueGame game col =
        do
        putStrLn $ "Move at " ++ show col ++ " " 
                ++ show (gravity game ! col) -- for expectk
        playGame (move col game)
        {-
        -}

{- get a command from the player -}
getCommand :: Game -> Player -> IO Command
getCommand game cpu@(Computer p) = 
    do
        column <- getMove p game
        let t = turn game
        return (MakeMove (Place t column)) 
        -- the computer's only command is to place pieces in columns
getCommand game (Human name) =
    do
        putStr $ "(" ++ name ++ ") Cmd: "
        cmd       <- getLine  -- read in the command, then parse it:
        parsedCmd <- parseCommand game cmd
        return parsedCmd


{----------------------------------------------------------------}
{---- Utilities to help interaction/understanding commands  -----}
{- convert a string entered by a human into a command -}
parseCommand :: Game -> String -> IO Command
parseCommand game cmd = parseCommand' (map toLower cmd)
    where
    parseCommand' "quit"     = return QuitProg
    parseCommand' "exit"     = return QuitProg
    parseCommand' "new"      = return NewGame
    parseCommand' (x:_) | x `elem` "123456789" = 
            let t   = turn game 
                col = read [x] :: Int 
            in if col <= (width game) then 
                  return $ MakeMove (Place t col)
               else return $ NoParse cmd
    parseCommand' cmd
        | isPrefixOf "show" cmd = return ShowBoard
        | isPrefixOf "help" cmd = return Help
        | isPrefixOf "save" cmd = saveGameHandle cmd
        | isPrefixOf "load" cmd = loadGameHandle cmd
    parseCommand' str      = return $ NoParse cmd


{-- lots of cutting and pasting below here --}
loadGameHandle :: String -> IO Command
loadGameHandle ('l':'o':'a':'d':name) =
    let name' = dropWhile isSpace name in
        if null name' then  -- did they provide a filename?
            do fname <- promptFileName
               return (LoadGame fname)
        else 
               return (LoadGame name')

saveGameHandle :: String -> IO Command
saveGameHandle ('s':'a':'v':'e':name) =
    let name' = dropWhile isSpace name in
        if null name' then  -- did they provide a filename?
            do fname <- promptFileName
               return (SaveGame fname)
        else 
               return (SaveGame name')

promptFileName :: IO String
promptFileName = do
                     putStr "File name? "
                     name <- getLine
                     return name


{- saves game to file: code stolen from Gentle Intro to Haskell -}
saveGame :: Game -> String -> IO ()
saveGame game file = 
    do 
       catch (do handle <- openFile file WriteMode
                 hPutStr handle (simpleShowGame game)
                 hClose handle
                 putStrLn "Saved game."
             ) 
             (\_ -> do putStrLn $ "Error writing to file " ++ file
                       return ())


{-  loads game from file: just looks at first line (move list)
    for error handling, we have to give it the existing game to return
    if something goes wrong  -}
loadGame :: Game -> String -> IO Game
loadGame currentGame file =
    do 
        catch (do handle <- openFile file ReadMode
                  str    <- hGetContents handle
                  let g  = simpleReadGame str
{-                  print str
                  hClose handle <-- BUG! I can't have this in without print str, which I don't really want to have!! :( -}
                  return g
              )
              (\_ -> do putStrLn $ "Error reading from file " ++ file
                        return currentGame)

                

    
    
