module Types where

import Array

data Command = MakeMove Move | QuitProg | NewGame  |
               NoParse String | Help  | ShowBoard | SaveGame String | LoadGame String
    deriving (Show, Eq)

{- Turn -}
data Turn = X | O
    deriving (Show, Eq)


{- Move -}
data Move = Place Turn Column |
            Resign Turn
    deriving (Eq)

instance Show Move where
    show (Place turn col) = show turn ++ " in column " ++ show col

{- Square -}
data Square = Piece Turn | Empty
    deriving (Eq)

instance Show Square where
    show (Piece x) = show x
    show (Empty)   = "-"


{- Personality -}
data Personality = 
    Personality {
        name         :: String,
        tendencies   :: CPUTendencies, -- likes and dislikes for player
        moveFunction :: CPUTendencies -> Game -> IO Column 
        -- moveFunction is how player uses his tendencies to decide on 
        -- a column (i.e. the underlying algorithm)
    }

instance Show Personality where
    show pers = name pers ++ ", " ++ show (tendencies pers)

data CPUTendencies =
    CPUTendencies {
        mainDepth    :: Plies, -- how many plies to look ahead
        trailOff     :: [Int],   -- what percentage (NB!!) of moves 
                                 -- to look at at each ply
        forceMateDepth :: Plies, -- how far to look for a forced mate
        startEndGame   :: Int,  -- how far from end go into end game mode
        endGameDepth   :: Plies, -- how far to look ahead during end game

        upDown :: [Float],  -- how interested we are in 1s, 2s, 3s and 4s 
                          -- in a row vertically
        leftRight :: [Float], -- same as upDown but horizontal
        diagonal :: [Float],  -- ditto, but for two diagonals
        defensiveness :: Float -- 1 ==> just as defensive as attacking,
                               -- >1 ==> more defensive
                               -- <1 ==> less defensive
                               -- 0 ==> ignore defense
                               -- <0 ==> idiotic
    } 


instance Show CPUTendencies where
    show t = show (mainDepth t) ++ " plies."

                    
{- Player -}
data Player      = Human String | Computer Personality
instance Show Player where
    show (Human name) = name
    show (Computer p) = "CPU " ++ show p


{- Game -}
data Game = Game { 
                   board       :: Board, 
                   gravity     :: Gravity,    -- where next piece drops to
                   turn        :: Turn,
                   history     :: History,   -- in *reverse* order (for speed)
                   playerX     :: Player,
                   playerO     :: Player,
                   height      :: Int,
                   width       :: Int,
                   strictRules :: Bool,
                   numMoves    :: Int        -- always is length history
                 }

instance Show Game where
    show (game) = 
        boxAround $
            let nl = "\n" in 
                "Xs: "     ++ 
                show (playerX game) ++ nl ++
                "Os: "           ++
                show (playerO game) ++ nl ++ nl ++
                showBoard (board game)    ++ nl ++
                "Turn: " ++ show (turn game)    ++ nl ++
                "Moves played: " ++ show (numMoves game)
    {-            ++ "History: " ++ show (history game) ++ nl ++
                "Gravity: " ++ 
                    show ([gravity game ! i | i <- [1..(width game)]])
    -}

{- this probably should be in Utilities, but Show Game needs it -}
boxAround :: String -> String
boxAround str = horizontal ++ unlines (map sideLine lines') ++ horizontal
    where
    lines'     = lines str
    sideLine s = padRight (width - 1) ("| " ++ s) ++ "|"
    width      = 4 + maximum (map length lines')
    padRight 0 s       = s
    padRight n ""      = replicate n ' ' 
    padRight n (s:str) = s:(padRight (n - 1) str)
    horizontal = "|" ++ replicate (width - 2) '-' ++ "|\n"

{- Board -}
type Board = Array (Int, Int) Square
showBoard :: Board -> String
showBoard board = unlines $ map unwords brd
    where
    ((1,1), (height,width)) = bounds board  -- find out the dimensions of board
    brd = [ [show (board!(x,y)) | y <- [1 .. width]] | x <- [1 .. height] ]

{- Gravity -}
type Gravity = Array Int Int  -- how far a piece can drop in a certain column
    
{- Synonyms -}
type History = [Move]
type Column = Int
type Plies = Int

type Score = Float
