module Utilities where
{- some simple utility functions for Connect 4 -}

import Array
import Types 
import Char
import List(isPrefixOf)


------------------------------------------------------------------
{------------  Making moves and updating the game  --------------}

openWith :: [Column] -> Game
openWith mvs = moves mvs newGame

{- given a list of column numbers, make the moves listed and return new game -}
moves :: [Column] -> Game -> Game
moves cols = foldr1 (flip (.)) (map move cols)

{- just assumes the player on turn is the one moving, then calls makeMove -}
move :: Column -> Game -> Game
move col game = makeMove (Place (turn game) col) game

{- given a history list (remember that they're in reverse order), 
   make the moves in it and return an appropriate game -}
replayHistory :: History -> Game -> Game
replayHistory hist = makeMoves (reverse hist)

{- given a list of moves, make the moves listed and return the new game -}
makeMoves :: [Move] -> Game -> Game
makeMoves mvs = foldr1 (flip (.)) (map makeMove mvs)

{- make a move and return altered game -}
makeMove :: Move -> Game -> Game
makeMove mv@(Place t col) game =
    if   (col > width game || columnFull game col) -- can piece fit here?
    then error $ "Illegal move -- column full or out of bounds." ++
                 "\ncol=" ++ show col ++ "  grav=" ++ show ((gravity game)!col) ++ "\n"

    else 
        if (t /= (turn game)) && strictRules game
        then error $ "It's not the turn of " ++ show t
        else
            game { 
                board   = board', 
                gravity = gravity', 
                turn    = toggle t, -- TODO: Should this be toggle turn?
                numMoves = numMoves',
                history = mv:(history game)
            }
      where
        numMoves' = numMoves game + 1
        row       = gravity game ! col
        board'    = (board game) // [((row, col), Piece t)]
        gravity'  = (gravity game) // [(col, row - 1)]
makeMove (Resign _) _ = error "Sorry -- resignations not handled yet."


columnFull :: Game -> Column -> Bool
columnFull game col = gravity game!col <= 0

columnNotFull :: Game -> Column -> Bool
columnNotFull g = not . columnFull g

legalMoves :: Game -> [Column]
legalMoves g = filter (columnNotFull g) [1 .. (width g)]


------------------------------------------------------------------
{------------             Checking for wins         --------------}
{- takes a game and tells you whether the last move won the game for a player -}
{- geared for speed rather than readability -- sorry! -}
lastMoveWin :: Game -> Bool
lastMoveWin game = 
    not newgame &&   -- see if any moves have been played
    lastMoveWin' (row, col) (board game) t
    where
    newgame = null (history game)
    hist = history game
    (Place t col:_) = hist
    row  = (gravity game)!col + 1  -- for before the move was made
    
    -- IMPORTANT: assumes that board!(row, col) == turn
    lastMoveWin'   :: (Int, Int) -> Board -> Turn -> Bool
    lastMoveWin' (row, col) board turn
        | board!(row,col) /= (Piece turn) = 
            error $ "Piece must be in square already: " ++ show (row,col)
        | otherwise = (look' left + look' right) >= 3 || 
                      (look' down) >= 3               || 
                      (look' topl + look' botr)  >= 3 ||
                      (look' botl + look' topr)  >= 3
        where
        staySame   x = repeat x
        increasing x = [x + 1 .. ]
        decreasing x = [x - 1, x - 2 .. ]
    
        {- directions to look in -}
        -- straightforward ones
        left   = zip (staySame row) (decreasing col)
        right  = zip (staySame row) (increasing col)
        down   = zip (increasing row) (staySame col)
    
        -- top left, bottom right, bottom left and top right diagonals
        topl   = zip (decreasing row) (decreasing col)
        botr   = zip (increasing row) (increasing col)
        botl   = zip (increasing row) (decreasing col)
        topr   = zip (decreasing row) (increasing col)
    
        {- counting consecutive occurrences of (Piece turn) on board -}
        look' indices = look (takeWhile (inBounds board) indices) 
        look indices = 
            length $ takeWhile (\sq -> board!sq == Piece turn) indices

{- are all the columns full? -}
drawnGame :: Game -> Bool
drawnGame game = all (columnFull game) [1 .. (width game)]

-- takes a board and piece coordinates and returns whether coords are in bounds
inBounds :: Board -> (Int, Int) -> Bool
inBounds board (row, col) = 
    let (w, h) = snd (bounds board) in
        row >= 1 && col >= 1 && 
        row <= w && col <= h


-- returns a new game of standard size
newGame  :: Game
newGame  = newGameDims 6 7

-- takes height and width of board and makes a new game
newGameDims :: Int -> Int -> Game
newGameDims h w =
        Game { 
                 height = h,
                 width  = w,
                 gravity = array (1,w) (zip [1..w] $ replicate w h), 
                 board = newBoard (h,w), 
                 turn  = O,  -- arbitrary decision that O goes first
                 numMoves = 0,
                 history = [],
                 playerO = Human "Bryn",
                 playerX = Human "Rose",
                 strictRules = True
                }

{- start a new game but with the same properties (like players, 
   board size etc. -}
newGameSameProps :: Game -> Game
newGameSameProps game = 
    let h = height game
        w = width game in
    game {
          gravity = array (1,w) (zip [1..w] $ replicate w h),
          board   = newBoard (h,w),
          turn    = O,
          history = [],
          numMoves = 0
    }


-- returns empty board of height h and width w
newBoard :: (Int, Int) -> Board
newBoard (h, w) = 
    array ((1,1), (h,w)) [((x,y),Empty) | x <- [1..h], y <- [1..w]]

-- change the turn between X and O
toggle   :: Turn -> Turn
toggle X = O
toggle O = X

{- have a computer personality make a move in the current game -}
getMove :: Personality -> Game -> IO Column
getMove Personality { tendencies = tend, moveFunction = moveFun } game =
    moveFun tend game



{- take a simpleShowGame result and reads it in 
   just reads in the history at the moment -- sorry! -}
simpleReadGame :: String -> Game
simpleReadGame str = openWith . readHist . head . lines $ str
    where
    readHist  = map readInt . words 
    readInt x = read x :: Int 

{- writes the game out in a simple to parse form -}
simpleShowGame :: Game -> String
simpleShowGame game = 
    let showHist             = unwords . map showMove . reverse
        showMove (Place _ c) = show c
        showMove (Resign _)    = "resign"
    in
    unlines [showHist (history game),
             show (playerX game),
             show (playerO game), 
             showBoard (board game)]

