
\section{Colour}

\subsection{Header}

This module implements Colour as an ADT, just like Point.  It also
defines some useful operations. The original Pancito had a whole pile
of Colour functions that I never used; this one doesn't.  Other
changes include HSV support and pre-multiplied alpha.

\begin{code}
module Colour (
  Colour, rgba, hsva, optimizeHsva, optimizeRgba, tiny,
  r, g, b, a, h, s, v,
  Tint, red, green, blue, black, white,
  opacity, transparent, interp, lighten, darken,
  cyan, magenta, yellow, grey,
  rotateHue, saturate, brighten,
  overlay, combine, add, sub, average, cMap
) where

default ()
\end{code}

\subsection{Constructor, Classes}

Again, by using two different representations we can avoid unnecessary
conversions (if the user is explicit in writing conversions) but still
support both representations at any point.  We can also check that
components are within accepted value ranges.

However, there are some disadvantages to using two different
representations: some operations depend on the underlying type.  In
particular, adding two colours gives different results for HSVA and
RGBA encoded values.  This is ugly, but I'm not sure what the best
solution is.  At the moment, you are free to force the type using the
two optimize functions.  An alternative would be to provide different
functions for the two operations.

\begin{code}
data Colour = RGBA Double Double Double Double
            | HSVA Double Double Double Double

instance Eq Colour where
  p1 == p2 = equalColour p1 p2

equalColour (RGBA r1 g1 b1 a1) (RGBA r2 g2 b2 a2) = 
  r1 == r2 && g1 == g2 && b1 == b2 && a1 == a2
equalColour (HSVA h1 s1 v1 a1) (HSVA h2 s2 v2 a2) = 
  h1 == h2 && s1 == s2 && v1 == v2 && a1 == a2
equalColour c1 c2 = equalColour c1' c2'
  where
    c1' = optimizeRgba c1
    c2' = optimizeRgba c2

instance Show Colour where
  show p = showColour p

showColour (RGBA r g b a) = 
  "rgb(" ++ show r ++ "," ++ show g ++ "," ++ 
    show b ++ "," ++ show a ++ ")"
showColour (HSVA h s v a) = 
  "hsv(" ++ show h ++ "," ++ show s ++ "," ++ 
    show v ++ "," ++ show a ++ ")"
\end{code}

\subsection{Proxy Constructors}

The constructor is hidden --- these should be used instead.  They
enforce value ranges (including pre--multiplied alpha) which are
assumed when values are accessed later.

\begin{code}
rgba, hsva :: Double -> Double -> Double -> Double -> Colour
rgba r g b a = RGBA (clip0 a' r) (clip0 a' g) (clip0 a' b) a'
  where a' = (clip01 a)
hsva h s v a = HSVA (roll2pi h) (clip01 s) (clip0 a' v) a'
  where a' = (clip01 a)

clip0 :: Double -> Double -> Double
clip0 max x | x > max   = max
            | x < 0     = 0
            | otherwise = x

clip01, roll2pi :: Double -> Double
clip01 = clip0 1
roll2pi x | x >= twoPi = roll2pi (x - twoPi)
          | x < 0      = roll2pi (x + twoPi)
          | otherwise  = x
  where twoPi = 2 * pi
\end{code}

\subsection{Conversion}

Again, note that explicit conversion is optional, but improves
efficiency.

The conversions here are lifted from several posts on the internet
(search for ``RGB HSV convert''), translated from C or pseudocode to
Haskell.  I hope an optimizing compiler can simplify the logic in the
comparisons (I could do it by hand, but then it would even less
readable).

\begin{code}
tiny :: Double
tiny = 0.000001

mkHSVA max min n x y a = 
    hsva (rad * (n + xy)) s max a
  where
    d = max - min
    s = if max > tiny then d / max else 0
    xy = if d > tiny then (x - y) / d else 0
    rad = pi / 3.0

rgbaToHsva (RGBA r g b a)
  | abs (r - g) < tiny && abs (g - b) < tiny && 
    abs (b - r) < tiny = hsva 0 0 r a
  | order b g r = mkHSVA r b 0 g b a
  | order g b r = mkHSVA r g 0 g b a
  | order r b g = mkHSVA g r 2 b r a
  | order b r g = mkHSVA g b 2 b r a
  | order g r b = mkHSVA b g 4 r g a
  | order r g b = mkHSVA b r 4 r g a
  where
    order x y z = x <= y && y <= z

optimizeHsva :: Colour -> Colour
optimizeHsva c = case c of
                   (RGBA r g b a) -> rgbaToHsva c
                   (HSVA h s v a) -> c

mkRgba i p q t v a 
  | i == 0    = rgba v t p a
  | i == 1    = rgba q v p a
  | i == 2    = rgba p v t a
  | i == 3    = rgba p q v a
  | i == 4    = rgba t p v a
  | i == 5    = rgba v p q a
  | otherwise = error ("case " ++ show i ++ " for " ++ show p ++ ", " 
                       ++ show q ++ ", " ++ show t ++ ", "
                       ++ show v ++ ", " ++ show a ++ " in mkRgba")

hsvaToRgba (HSVA h s v a)
  | s == 0.0  = rgba v v v a
  | otherwise = mkRgba i p q t v a
  where
    h' :: Double
    h' = h / rad
    rad :: Double
    rad = pi / 3.0
    i :: Integer 
    i = floor h'
    f :: Double
    f = h' - (fromIntegral i)
    p :: Double
    p = v * (1 - s)
    q :: Double
    q = v * (1 - s * f)
    t :: Double
    t = v * (1 - s * (1 - f))

optimizeRgba :: Colour -> Colour
optimizeRgba c = case c of
                   (RGBA r g b a) -> c
                   (HSVA h s v a) -> hsvaToRgba c
\end{code}

\subsection{Access}

The following operators pull values out from the ADT.

\begin{code}
r, g, b, a, h, s, v :: Colour -> Double
r c = case c of
        (RGBA r' g b a) -> r'
	(HSVA h s v a)  -> r (hsvaToRgba c)
g c = case c of
        (RGBA r g' b a) -> g'
	(HSVA h s v a)  -> g (hsvaToRgba c)
b c = case c of
        (RGBA r g b' a) -> b'
	(HSVA h s v a)  -> b (hsvaToRgba c)
a c = case c of
        (RGBA r g b a') -> a'
	(HSVA h s v a') -> a'
h c = case c of
        (RGBA r g b a)  -> h (rgbaToHsva c)
	(HSVA h' s v a) -> h'
s c = case c of
        (RGBA r g b a)  -> s (rgbaToHsva c)
	(HSVA h s' v a) -> s'
v c = case c of
        (RGBA r g b a)  -> v (rgbaToHsva c)
	(HSVA h s v' a) -> v'
\end{code}

\subsection{Utilities}

First, a few colours and simple tools.  I've not provided anything to
force a single component to a value (but see combining colours later).
The Tint type will be discussed later when bring everything together
into a pipeline.

Note that, unlike Pancito 1, alpha is pre-multiplied (thanks to Conal
Elliott for point me to Alvy Ray Smith's
work\footnote{http://www.alvyray.com --- Memo ``Image Compositing
Fundamentals''.}).

\begin{code}
type Tint = Colour -> Colour

red, green, blue, black, white :: Colour
red = rgba 1 0 0 1
green = rgba 0 1 0 1
blue = rgba 0 0 1 1
white = rgba 1 1 1 1
black = rgba 0 0 0 1

opacity :: Double -> Tint
opacity x (RGBA r g b a) = rgba r' g' b' x
  where
    r' = rescale r a x
    g' = rescale g a x
    b' = rescale b a x
opacity x (HSVA h s v a) = hsva h s' v' x
  where
    s' = rescale s a x
    v' = rescale v a x

rescale x old new | old < tiny  = 0  -- transparent is black?
                  | otherwise   = x * new / old

transparent = opacity 0 black
\end{code}

Colours can be interpolated (this preserves alpha from the {\em
second} colour, allowing functions like lighten and darken that leave
transparency unchanged to be defined using currying):

\begin{code}
interp :: Double -> Colour -> Tint
interp frac c1 c2 =
    rgba (intp r) (intp g) (intp b) a2
  where 
    intp f = (\x -> x + frac * ((f c2') - x)) (f c1')
    a2 = a c2
    c1' = opacity a2 $ optimizeRgba c1
    c2' = optimizeRgba c2

lighten, darken :: Double -> Tint
lighten frac = interp (1 - frac) white
darken frac = interp (1 - frac) black

cyan, magenta, yellow, grey :: Colour
cyan = interp 0.5 blue green
magenta = interp 0.5 red blue
yellow = interp 0.5 green red  --- 0.6 looks better on my screen
grey = interp 0.5 black white
\end{code}

The equivalent of interpolation when thinking is HSVA space is a bunch
of separate actions:

\begin{code}
rotateHue, saturate, brighten :: Double -> Tint
rotateHue theta c = hsva (h c' + theta) (s c') (v c') (a c)
  where 
    c' = optimizeHsva c
saturate x c = hsva (h c') (s c' + x) (v c') (a c)
  where 
    c' = optimizeHsva c
brighten x c = hsva (h c') (s c') (v c' + x) (a c)
  where 
    c' = optimizeHsva c
\end{code}

\subsection{Combining Colours}

The canonical way to combine colours is by overlaying and letting the
alpha channel do its work.  In Pancito 1, ``underlay'' had reversed
arguments for use with foldl.  I now understand foldr is more commonly
used, so am back with ``overlay'' (the first argument is laid over the
second).

\begin{code}
overlay :: Colour -> Colour -> Colour
overlay c1 c2 = rgba (ov r) (ov g) (ov b) (ov a)
  where
    c1' = optimizeRgba c1
    c2' = optimizeRgba c2
    beta = a c1
    ov clr = (clr c1') + (1 - beta) * (clr c2')
\end{code}

If this isn't what you want (it often isn't when you're dealing with
pure colour images, as it loses saturation) the following might help
(alpha is set to 1 to avoid limiting when pre--multiplication is
enforced).

\begin{code}
combine :: (Double -> Double -> Double) -> 
             Colour -> Colour -> Colour
combine f (HSVA h1 s1 v1 a1) (HSVA h2 s2 v2 a2) =
    hsva (f h1 h2) (f s1 s2) (f v1 v2) 1
combine f (RGBA r1 g1 b1 a1) (RGBA r2 g2 b2 a2) =
    rgba (f r1 r2) (f g1 g2) (f b1 b2) 1
combine f c1 c2 = combine f c1' c2'
  where
    c1' = optimizeRgba c1
    c2' = optimizeRgba c2

add, sub :: Colour -> Colour -> Colour
add = combine (+)
sub = combine (-)
\end{code}

An average (RGBA) of a list of colours is useful sometimes.

\begin{code}
average :: [Colour] -> Colour
average cs = avg cs black 0

avg [] clr _   = clr
avg (c:cs) clr n = avg cs clr' n'
  where
    n' = n + 1
    clr' = rgba (av r) (av g) (av b) (av a)
    c1 = optimizeRgba clr
    c2 = optimizeRgba c
    k1 = n / n'
    k2 = 1 / n'
    av f = (k1 * f c1) + (k2 * f c2)
\end{code}

It can also be convenient to apply a function to each component of a
colour.

\begin{code}
cMap :: (Double -> Double) -> Colour -> Colour
cMap f (HSVA h1 s1 v1 a1) = hsva (f h1) (f s1) (f v1) 1
cMap f (RGBA r1 g1 b1 a1) = rgba (f r1) (f g1) (f b1) 1
\end{code}
 
