
\section{Pancito2}

\subsection{Header}

This defines the basic Pancito module.

\begin{code}
module Pancito2 (
  Image, flat, Merge, (<<), (<<<),
  VTint, o, Filter, mkFilter,
  writePpmBase, writePpmAlias, ppm, ppmAlias,
  tranAlias
) where

import Point
import Colour
import IO
import Monad
\end{code}

\subsection{Image}

Pancito considers images to be functions.  Give an image a point and
it gives you a colour:

\begin{code}
type Image = Point -> Colour

flat :: Colour -> Image
flat c _ = c
\end{code}

\subsection{Simple Image Pipelines}

It's useful to think of a functional image as a pipeline (a flow from
right to left matches Haskell's syntax when using function application
or composition).  To find the colour at a particular point in an image
we pop the Point into the right of the pipeline.  The Point can then
pass through Transforms (defined in Point --- these can alter the
Point coordinates, distorting the image) before entering an Image.
The Image converts the Point to a Colour.  The Colour then continues
down the pipe to the left, possibly passing through Tints that alter
it further, before finally appearing to us.

Simple Tints can be made from functions defined in Colour (brighten or
saturate, for example); Point has several Transforms.

The description above is a very simple pipeline.  Now we will extend
them to include several pipes in parallel and Tints that can vary with
position.

\subsection{Parallel Images}

A list of images can be generated in parallel and combind in some way.
The simplest example of combination is by overlaying.

\begin{code}
type Merge = [Image] -> Image

(<<) :: ([Colour] -> Colour) -> Merge
(<<) f ims p = f $ map (\x -> x p) ims

(<<<) :: Colour -> Merge
(<<<) bg = (foldr overlay bg <<)
\end{code}

That code is fairly dense, so some examples will probably help clear
things up:

\begin{code}
{-
im1, im2, imR, imG :: Image
imR = flat red
imG = flat green
im1 = white <<< [imR, imG]
im2 = (foldr add black) << [imR, imG]
-}
\end{code}

Here im1 is white overlaid with a red and then a green image (and
since green is opaque, the result will be green).  Im2 is formed by
adding the individual colour components and so will be yellow.

\subsection{Varying Tints}

The type system described so far doesn't support Tints that vary with
position (within a pipeline defined purely using functional
composition a Tint does not receive a Point).

We can always work round this:

\begin{code}
{-
mkTint :: Point -> Tint

im1, im2 :: Image
im1 p = tint . im2 p
  where tint = mkTint p
-}
\end{code}

but it's pretty ugly.  Instead, we can replace the ``.'' of functional
composition with a different operator, ``o'' that pulls a Point from
earlier in the pipeline.

\begin{code}
type VTint = Point -> Tint

o :: VTint -> Image -> Image
o mkTint im p = mkTint p $ im p
infixr 8 `o`
\end{code}

For example,

\begin{code}
{-
brightPos :: VTint
brightPos p = if (x p > 0) then brighten 0.5 else brighten (-0.5)

im1, im2, imR :: Image
imR = flat red
im1 = brightPos `o` imR . shift 1 2
im2 = (brightPos `o` imR) . shift 1 2
-}
\end{code}

Note that because ``o'' binds less strongly that ``.'' the coordinate
received by the tint in im1 above will not be shifted.  This behaviour
can be altered using parentheses, as in im2.

\subsection{Filter}

A still more general operation is Filter.

\begin{code}
type Filter = Image -> Image
\end{code}

While this could be any code, a simple way of building filters is by
assembling a Tint and a Transform.

\begin{code}
mkFilter :: Tint -> Transform -> Filter
mkFilter tint trans im = tint . im . trans
\end{code}

\section{Output}

This is similar to the code in the original Pan version, except that
aliasing is treated slightly differently.

\subsection{ppm Format}

The ppm image format is simple, text based, verbose, and has no
support for transparent images, but it does support arbitrary colour
depth and can be converted to a wide variety of other formats using
the netpbm and pbmplus packages (for CMYK encoded tiff files, see my
own pnmtocmyktiff).

A line in a ppm file can contain a maximum of 70 characters...

\begin{code}
lineLimit :: Int
lineLimit = 69 -- allow for \n

putChunk :: Handle -> Int -> String -> IO Int
putChunk h soFar str 
    | tot >= lineLimit = do hPutChar h '\n'
                            hPutStr h str
                            return len
    | otherwise        = do hPutChar h ' '
                            hPutStr h str
                            return tot
  where len = length str + 1
        tot = soFar + len 

-- this starts with a newline (see below)
putChunks :: Handle -> [String] -> IO ()
putChunks h l = do last <- foldM (putChunk h) lineLimit l
                   hPutStr h "\n"
\end{code}

...and begins with the following header:

\begin{code}
-- this ends needing a new line (see above)
headerPpm :: Handle -> (Int, Int) -> Int -> IO ()
headerPpm h (x, y) n = do hPutStrLn h "P3"
                          hPutStrLn h (show x)
                          hPutStrLn h (show y)
                          hPutStr h (show n)
\end{code}

\subsection{Generating Images}

To print an image it must be generated: the image function must be
called at each pixel and the colour converted to the correct (integer)
representation.  

\begin{code}
pointStream :: Window -> (Int, Int) -> [Point]
pointStream w dim = 
  map (pixelToPoint w dim) $ pixelStream dim

pixelToPoint :: Window -> (Int, Int) -> Transform
pixelToPoint w (nx, ny) = mapWindow w w'
  where 
    w' = (cartesian 1 1, cartesian nx' ny')
    nx' = fromIntegral nx 
    ny' = fromIntegral ny

pixelStream :: (Int, Int) -> [Point]
pixelStream (nx, ny) = 
  [cartesian (fromIntegral x) (fromIntegral y) 
    | y <- [ny,ny-1..1], x <- [1..nx]]
\end{code}

Converting Colour to a series of integers is all that remains before
tying everything together.  Note that here, n is a direct scale factor
and we drop the transparency (not supported by ppm).

\begin{code}
colourToInts :: Int -> Colour -> [String]
colourToInts n c = [flr r, flr g, flr b]
  where
    flr f = show $ max 0 (min n (floor (realToFrac n * f c')))
    c' = optimizeRgba c

groupRgb :: [String] -> String
groupRgb s = foldr addsp "" s
  where addsp x y = x ++ " " ++ y

showRgb :: Int -> Colour -> String
showRgb n = groupRgb . colourToInts n
\end{code}

Generating an image is now fairly simple:

\begin{code}
writePpmBase :: Int -> Window -> (Int, Int) -> 
                  Handle -> Image -> IO ()
writePpmBase b w dim h im =
    do headerPpm h dim n
       putChunks h $ map (showRgb n . im) $ pointStream w dim
  where n = 2^b - 1
\end{code}

\subsection{Transparency}

Because ppm does not support transparency the opacity will be ignored
--- this is equivalent to overlaying a transparent image on a black
background.  If a different background is required then it is trivial
to do this using overlay.

\begin{code}
-- opaqueOnWhite = overlay transparentImage white
\end{code}

\subsection{Aliasing}

Anti--aliasing sub--samples the image.  To do this correctly requires
information about the range and number of points that will be
generated.  This means that it must be integrated with the output
routines (if the correct parameters are to be automatically carried
across).

It's simplest to generate the aliasing points before converting from
integers to points within the coordinate system used by the image (ie
inbetween pixelStream and pixelToPoint in the output routine above).
So the code below generates a list of transforms that work on pixel
numbers and then applies the correct transform afterwards.

\begin{code}
tranAlias :: Int -> Window -> (Int, Int) -> [Transform]
tranAlias n w dim = 
    map mkShift [(x, y) | x <- [1..n], y <- [1..n]]
  where
    mkShift (x, y) = zoom . shift (f x) (f y)
    f z = (fromIntegral z - 0.5) / (fromIntegral n) - 0.5
    zoom = pixelToPoint w dim
\end{code}

This can then be used in an output routine.  Apart from using
pixelStream rather than pointStream (see comments above) we reproduce
the earlier code, but change the image into a list of parallel
pipelines, one for each subsampled point, averaging the result.

It's worth looking at this code (and comparing it with writePpmBase)
in some detail --- once it's clear what's happening you'll have
understood how to use parallel images (I hope!).

\begin{code}
writePpmAlias :: Int -> Int -> Window -> (Int, Int) -> 
                   Handle -> Image -> IO ()
writePpmAlias b x w dim h im =
    do headerPpm h dim n
       putChunks h $ map (showRgb n . im') $ pixelStream dim
  where 
    n = 2^b - 1
    im' = average << map (im .) (tranAlias x w dim) 
\end{code}

\subsection{Simple Output}

These are simplified versions of the above with some standard
parameters.

\begin{code}
ppm :: Window -> (Int, Int) -> String -> Image -> IO ()
ppm w dim name im = do h <- openFile name WriteMode
                       writePpmBase 8 w dim h im

ppmAlias :: Window -> (Int, Int) -> String -> Image -> IO ()
ppmAlias w dim name im = do h <- openFile name WriteMode
                            writePpmAlias 8 3 w dim h im
\end{code}
