
\section{Point}

\subsection{Header}

This module implements Point as an ADT.  This has some disadvantages
--- you can't pattern match to get the individual components --- but
it provides a safe way of switching between polar and rectangular
coordinates and it lets me introduce checks for value boundaries.

\begin{code}
module Point (
  Point, cartesian, polar, optimizePolar, optimizeCartesian, 
  x, y, d, t,
  Transform, shift, scale, expand, rotate,
  Window, origin, unit, nUnit, square01, square11, mapWindow,
  Region, contains, winRegion,
  rot90, rot180, rot270, flipX, flipY, pixelate11
) where

default ()
\end{code}

\subsection{Constructor, Classes}

The code is based on an example posted to the Haskell mailing list by
Tom Pledger.  By using two different representations we can avoid
unnecessary conversions (if the user is explicit in writing
conversions) but still support both representations at any point.

\begin{code}
data Point = Cartesian Double Double
           | Polar     Double Double

instance Eq Point where
  p1 == p2 = equalPoint p1 p2

equalPoint (Cartesian x1 y1) (Cartesian x2 y2) = x1 == x2 && y1 == y2
equalPoint (Polar d1 t1) (Polar d2 t2) = d1 == d2 && t1 == t2
equalPoint p1 p2 = equalPoint p1' p2'
  where
    p1' = optimizeCartesian p1
    p2' = optimizeCartesian p2

instance Show Point where
  show p = showPoint p

showPoint (Cartesian x y) = "(" ++ show x ++ "," ++ show y ++ ")"
showPoint (Polar d t) = "p(" ++ show d ++ "," ++ show t ++ ")"
\end{code}

\subsection{Proxy Constructors}

The constructor is hidden --- these should be used instead.  Note that
explicit conversion is optional (it's provided to allow code to be
optimized, hence the name) and isn't as important as for Colours
because conversino is no less expensive than using d and t once each
(so optimizing is only worthwhile if you will repeatedly use polar
values from the same Point).

\begin{code}
cartesian, polar :: Double -> Double -> Point
cartesian x y = Cartesian x y
polar d t     = Polar d (roll2pi t)

roll2pi :: Double -> Double
roll2pi x | x >= twoPi = roll2pi (x - twoPi)
          | x < 0      = roll2pi (x + twoPi)
          | otherwise  = x
  where twoPi = 2 * pi

optimizeCartesian, optimizePolar :: Point -> Point
optimizeCartesian p = case p of 
                        (Polar d t)     -> cartesian (x p) (y p)
                        (Cartesian x y) -> p 
optimizePolar p  = case p of 
                     (Polar d t)     -> p
                     (Cartesian x y) -> polar (d p) (t p)
\end{code}

\subsection{Access}

The following operators pull values out from the ADT.  I've used d
rather than r for the polar radial distance so that r can be used for
the red component of colours.

\begin{code}
x, y, d, t :: Point -> Double
x (Cartesian x y) = x
x (Polar d t)     = d * cos t
y (Cartesian x y) = y
y (Polar d t)     = d * sin t
d (Cartesian x y) = sqrt (x*x + y*y)
d (Polar d t)     = d
t (Cartesian x y) = atan2 y x
t (Polar d t)     = t
\end{code}

\subsection{Modifying Points}

Remember that these alter coordinates before being passed to the
image, and so are counterintuitive.  For example, scaling the
coordinates expands to include more of the image --- more of the image
is displayed in the same area, so it appears "smaller".

\begin{code}
type Transform = Point -> Point

shift, scale :: Double -> Double -> Transform
shift dx dy p = cartesian (x p + dx) (y p + dy)
scale fx fy p = cartesian (fx * x p) (fy * y p)

expand, rotate :: Double -> Transform
expand f p = case p of
               (Cartesian x y) -> scale f f p
               (Polar d t)     -> polar (f * d) t
rotate theta p = polar (d p) (t p + theta)
\end{code}

\subsection{Windows}

Often it is necessary to convert from one ractangular region to
another.

\begin{code}
type Window = (Point, Point)

origin, unit, nUnit :: Point
origin = cartesian 0.0 0.0
unit = cartesian 1.0 1.0
nUnit = cartesian (-1.0) (-1.0)

square01, square11 :: Window
square01 = (origin, unit)
square11 = (nUnit, unit)

mapWindow :: Window -> Window -> Transform
mapWindow (fromBL, fromTR) (toBL, toTR) =
    shift dx dy . scale fx fy
  where
    fx = (x fromTR - x fromBL) / (x toTR - x toBL)
    fy = (y fromTR - y fromBL) / (y toTR - y toBL)
    dx = x fromBL - x toBL * fx
    dy = y fromBL - y toBL * fy
\end{code}

Windows are the simplest example of a more general idea --- describing
a specific region of an image.  The Window type is convenient for
ractangles, but for more complex shapes we need functional Regions.

\begin{code}
type Region = Point -> Bool

contains :: Region -> Point -> Bool
contains r p = r p  --- syntactic sugar

winRegion :: Window -> Region
winRegion (bl, tr) p =
  (x bl - x p) * (x p - x tr) >= 0 &&
    (y bl - y p) * (y p - y tr) >= 0
\end{code}

\subsection{Various Tools}

Some simple flips and rotations.

\begin{code}
rot90, rot180, rot270, flipX, flipY :: Transform
rot90 (Cartesian x y) = cartesian (-y) x
rot90 (Polar d t) = polar d (t + pi / 2)
rot180 (Cartesian x y) = cartesian y x
rot180 (Polar d t) = polar d (t + pi)
rot270 (Cartesian x y) = cartesian y (-x)
rot270 (Polar d t) = polar d (t - pi / 2)
flipX p = cartesian (x p') (-1 * y p')
  where p' = optimizeCartesian p  -- lazy, could do polar
flipY p = cartesian (-1 * x p') (y p')
  where p' = optimizeCartesian p
\end{code}

Transform the window ((n,m) (n+1,m+1)) to ((-1,-1) (1,1)).  So each
unit square is mapped to the same square11.

\begin{code}
pixelate11 :: Transform
-- pixelate11 p = cartesian (2 * f x) (2 * f y)
pixelate11 p = cartesian ((2::Double) * f x) ((2::Double) * f y)
  where 
    p' :: Point 
    p' = optimizeCartesian p
    f :: (Point -> Double) -> Double
    f xy = (xy p') - fromIntegral ((round (xy p')):: Integer)
\end{code}
 
