
\section{Reprocess}

\subsection{Header}

This module allows you to read images back into the system (and
reprocess them).  Only ASCII ppm images can be read --- check out
pnmtoplainpnm from the netpbm package (this is the same format that
Pancito uses for output).

\begin{code}
module Reprocess (
  readPpm, wrapArray
) where

import Point
import Colour
import Pancito2
import Char
import IO
import Monad
import Array
\end{code}

\subsection{Arrays}

The image is read into an array.  Functions here allow that array to
be used as an Image.  This function might be useful in its own reight
one day.

\begin{code}
quant :: Point -> Window -> ((Int, Int), (Int, Int)) -> (Int, Int)
quant p (lo,hi) ((ilo,jlo),(ihi,jhi)) = (i, j)
  where
    x' = x p
    y' = y p
    xlo = x lo
    ylo = y lo
    xhi = x hi
    yhi = y hi
    dx = (x' - xlo) / (xhi - xlo)
    dy = (y' - ylo) / (yhi - ylo)
    ni = ihi - ilo + 1
    nj = jhi - jlo + 1
    i = min (ilo + floor(dx * fromIntegral ni)) ihi
    j = min (jlo + floor(dy * fromIntegral nj)) jhi

wrapArray :: Window -> Colour -> Array (Int, Int) Colour -> Image
wrapArray box c a p = if (winRegion box) p
     	                then a!(quant p box (bounds a))
                        else c
\end{code}

\subsection{Parsing the File}

Simple utilities to tokenize the input.  Note that slurp removes both
preceding and following whitespace, making life simpler when reading
and processing data.

\begin{code}
dropSpace :: String -> String
dropSpace [] = []
dropSpace (c:s) | isSpace c = dropSpace s
                | otherwise = c:s

slurp :: String -> (String, String)
slurp [] = ([], [])
slurp (c:s) | isSpace c = ([], dropSpace s)
            | otherwise = (c:x, y) where (x, y) = slurp s

tok :: String -> (String, String)
tok = slurp . dropSpace

itok :: String -> (Int, String)
itok s = (i, s'')
  where
    (s', s'') = tok s
    i = read s'

rtok :: Double -> String -> (Double, String)
rtok mx s = (d, s')
  where
    (i, s') = itok s
    d = fromIntegral i / mx

ctok :: Double -> String -> (Colour, String)
ctok mx [] = error "no more data"
ctok mx s = (rgba r g b 1.0, s''')
  where
    (r, s') = rtok mx s
    (g, s'') = rtok mx s'
    (b, s''') = rtok mx s''
\end{code}

\subsection{Read the Data}

Put everything together.

\begin{code}
dropPpm :: String -> String
dropPpm s = let (hdr, s') = slurp s
             in if hdr == "P3"
                then s'
                else error "incorrect file format"

buildColours :: Double -> String -> [Colour]
buildColours mx s = c:(buildColours mx s') where (c, s') = ctok mx s

buildImage :: String -> Window -> Colour -> Image
buildImage s w c = wrapArray w c a
  where
    (nx, s') = itok s
    (ny, s'') = itok s'
    (mx, s''') = itok s''
    bnds = ((1,1),(nx,ny))
    xy = map (\(y,x) -> (x, ny-y+1)) (range ((1,1),(ny,nx)))
    a = array bnds (zip xy (buildColours (fromIntegral mx) s'''))

readPpm :: String -> Window -> Colour -> IO Image
readPpm name w c = do
		     s <- readFile name
		     return $ buildImage (dropPpm s) w c
\end{code}

\subsection{Example}

This is fairly obvious, I hope.

\begin{code}
{-
module Main where
main :: IO ()
main = do
         img <- readPpm "test-in.ppm" square01 white
         ppm square11 (20, 20) "test-out.ppm" img
-}
\end{code}
 