
\section{Test}

This section describes a ``typical'' image.

\subsection{Modules}

As well as the Pancito modules, we use random numbers.

\begin{code}
module Main where

import Random
import Pancito2
import Colour
import Point

default ()
\end{code}

\subsection{Random Squares}

This code is a bit clunky --- at one point it had an extra predicate
in the comprehension that generated the list of colours (the idea was
to test for suitable colour combinations).  Without that test the
number of random numbers required is fixed and the code could probably
be simplified (ie there's no backtracking).

\begin{code}
mkImages :: Int -> Int -> [Double] -> [Bool] -> [Image]
mkImages nx ny ran flip =
  map mkSquare $ zip [(x, y) | x <- [(-1)*nx..nx], y <- [(-1)*ny..ny]]
                     (map mkBase $ zip [c | c <- threeColours ran]
                                       (double flip))

double :: [a] -> [(a,a)]
double ran = (r1, r2) : double ran'
  where r1:r2:ran' = ran

triple :: [a] -> [(a,a,a)]
triple ran = (r1, r2, r3) : triple ran'
  where r1:r2:r3:ran' = ran

mkColour :: (Double, Double, Double) -> Colour
mkColour (r1, r2, r3) = hsva (2 * pi * r1) r2 r3 1

threeColours :: [Double] -> [(Colour, Colour, Colour)]
threeColours ran = triple $ map mkColour $ triple ran
\end{code}

This is the function that varies the colour across a square.  There is
an implicit asumption that the colours are in HSVA format (addition in
RGB would look different).  This is a gotcha I discuss earlier.  I
could use optimizeHsva, but since I know that mkColour is generating
HSVA values, there's no real need.

\begin{code}
mkBase :: ((Colour, Colour, Colour), (Bool, Bool)) -> Image
mkBase ((c1, c2, c3), (b1, b2)) p = 
    (cmb b2 (cmb b1 c1 c2 (x p)) c3 (y p))
  where
    cmb b c1 c2 z = (if b then add else sub) c1 (cMap ((z * 0.2) *) c2)

mkSquare :: ((Int, Int), Image) -> Image
mkSquare ((x', y'), im) = (square im) . shift x'' y''
  where
    x'' = 2 * fromIntegral x'
    y'' = 2 * fromIntegral y'

box :: Region
box p = abs (x p) < 0.7 && abs (y p) < 0.7

square :: Image -> Image
square im p = if contains box p 
                then im p
                else transparent
\end{code}

The arcPixel function quantizes the image in small arcs, giving the
impression that the screen is made from pixels in a circular pattern.
At least, that was the original intentions (and it does work on images
which change colour fairly rapidly).  Here, however, it simply
crinkles the edges of the gem-like squares.

\begin{code}
arcPixel :: Transform
arcPixel p = polar qd qt
  where
    d' = d p
    t' = t p
    qd = quant n d'
    qt = if qd < tiny then 0 else (quant n (qd * (t' + z))) / qd - z
    n = 40
    z = 0.1

quant :: Double -> Double -> Double
quant n x = (fromIntegral ((round (n * x))::Integer)) / n
\end{code}

Finally, combine everything.

\begin{code}
im :: Image
im = (black <<< mkImages n n ran flip) . expand (2 * (1 + fromIntegral n)) . arcPixel
  where
    n = 2
    ran = randoms $ mkStdGen 1
    flip = randoms $ mkStdGen 2
    -- ran = [1..]
    -- flip = cycle [False , True]

main :: IO ()
-- main = ppmAlias square11 (200, 200) "test.ppm" im
main = ppmAlias square11 (50, 50) "test.ppm" im
\end{code}

To run this code, type (exact details depend on your compiler etc.):
\begin{verbatim}
ghc -O -c Colour.lhs 
ghc -O -c Point.lhs
ghc -O -c Pancito2.lhs 
ghc -O -o Test Test.lhs Pancito2.o Colour.o Point.o
./Test
\end{verbatim}

To generate the image (200x200 pixels, anti-aliased) takes about 5
minutes on my laptop (200MHz pentium running Linux; without the -O
flag it takes 10 minutes).  Without anti-aliasing it should take just
over 30 seconds (1/9 the time).

I use xloadimage and NetPBM to view and manipulate the result; NetPBM
is also available for Win32 and I understand that Paint Shop Pro also
reads the ppm format.

 
