-- Parser.hs
--
-- Dieser Modul stellt Parsergeneratoren fuer einige gaengige EBNF-Konstrukte
-- zur Verfuegung. Ein CParser ist ein abstrakter Datentyp.

module Parser(CParser, detCParser, lit, pToken, epsilon, (<|>), (<*>), 
              star, oneOrMore, (<@)) where


-- Eine guenstige Wahl von Assoziativitaet und Prioritaet erspart viele 
-- Klammern.

infixr 6 <*>
infixr 5 <@
infixr 4 <|>


-- Ein Parser fuer eine Sprache mit deterministischer Grammatik 
-- akzeptiert, wenn die Eingabe restlos verbraucht wird.
-- Dieser Parser ist also eine schlichte Funktion von Token zu einem
-- Syntaxbaum.
-- `detCParser' wandelt einen solchen Parser entsprechend um.

detCParser :: CParser token syntaxtree -> ([token] -> syntaxtree)

detCParser (CParser parser) tokensequence = check (parser tokensequence)
  where
  check [([], syntaxtree)] = syntaxtree
  check [] 		   = error "Parsing terminated due to failure."
  check _	           = error "More input than expected."


-- Ein Parser bildet eine Tokenfolge auf einen Syntaxbaum und eine restliche
-- Tokenfolge ab.
-- Fuer eine mehrdeutige Grammatik kann ein Parser auch mehrere verschiedene 
-- Syntaxbaeume erzeugen.
-- Im Falle einer fehlerhaften Tokenfolge wird gar kein Syntaxbaum erzeugt.
-- Daher produziert ein Parser eine Liste von Ergebnissen.

newtype CParser token syntaxtree 
  = CParser ([token] -> [([token], syntaxtree)])


-- Fuer (parameterlose) Literale gibt es einen einfachen Parsergenerator

lit :: Eq token => token -> CParser token token

lit literal = CParser lit'
  where
  lit' [] = []
  lit' (token:tokenrest)
    | (literal == token) = [(tokenrest, literal)]
    | otherwise	         = []


-- Zur Generierung eines Parsers fuer Token mit Parametern wird eine
-- Funktion benoetigt, die das Token erkennt und sogar eventuell direkt
-- modifiziert.

pToken :: (token -> Maybe modifiedToken) -> CParser token modifiedToken

pToken recognizeAndModify = CParser token
  where
  token [] = []
  token (token:tokenrest)
    = check (recognizeAndModify token)
    where 
    check Nothing 	       = []
    check (Just modifiedToken) = [(tokenrest, modifiedToken)]


-- Mitunter nuetzlich ist ein Parser, der die leere Tokenfolge epsilon parst.
-- Der produzierte Syntaxbaum ist immer das leere Tupel.

epsilon :: CParser token ()

epsilon = CParser ( \tokensequence -> [(tokensequence, ())] )


-- Rechte Regelseiten koennen aus Alternativen bestehen.
-- Aus zwei alternativ anwendbaren Parsern wird ein neuer gebaut.
-- Beide Parser muessen den gleichen Typ von Syntaxbaum liefern,
-- da der Syntaxbaum des Gesamtparsers von jedem der beiden Parser
-- produziert worden sein kann.

(<|>) :: CParser token stree -> CParser token stree -> CParser token stree

CParser parser1 <|> CParser parser2
  = CParser (\tokensequence -> parser1 tokensequence ++ parser2 tokensequence)


-- Eine rechte Regelseite kann eine Sequenz von Symbolen sein.
-- Auch hier entsteht aus zwei Parsern einer, der Syntaxbaum setzt
-- sich jedoch aus den zwei Teilsyntaxbaeumen zusammen.

(<*>) :: CParser token stree1 -> CParser token stree2 -> 
         CParser token (stree1, stree2)

CParser parser1 <*> (CParser parser2)
  = CParser (\tokensequence ->
      concatMap next (parser1 tokensequence))
    where
    -- (CParser parser2) = cp2  -- solves strictness problem of Redex Tracer
    next (tokenrest1,syntaxtree1) = map combine (parser2 tokenrest1)
      where
      combine (tokenrest2, syntaxtree2) = 
        (tokenrest2, (syntaxtree1, syntaxtree2))

-- Eine spezielle Variante der Sequenz ist der Kleene'sche Stern.
-- Diese Funktion verwendet die vordefinierte Datenstruktur Liste.
-- Zu beachten: es wird nicht nur der longest possible match
-- geparst, sondern alle Moeglichkeiten; der longest possible match
-- jedoch zuerst, z.B:
-- star (lit 'a') "aa" => [([],"aa"), ("a","a"), ("aa",[])]

star :: CParser token stree -> CParser token [stree]

star parser = parser <*> star parser <@ uncurry (:) <|>
              epsilon <@ const []
	

-- `oneOrMore' ist eine Variante von `star', die die leere Sequenz
-- nicht zulaesst.

oneOrMore :: CParser token stree -> CParser token [stree]

oneOrMore parser = parser <*> star parser <@ uncurry (:)


-- Die von den Parsern erzeugten Syntaxbaeume bestehen nur aus
-- Token, modifizierten Token, leeren Tupeln, Zweiertupeln und
-- Listen.
-- Diese Darstellung muss ein Parser fuer eine konkrete Sprache an vielen 
-- Stellen noch in die jeweilige Datenstruktur fuer die abstrakte Syntax
-- konvertieren.
-- Dies erfolgt mit dem folgenden Operator.

(<@) :: CParser token syntaxtree -> (syntaxtree -> modifiedSyntaxtree) -> 
        CParser token modifiedSyntaxtree

CParser parser <@ modifier 
  = CParser (\tokensequence -> 
      map (\(toks,syntree) -> (toks, modifier syntree)) (parser tokensequence))

-- Ende




