-- PsaAnalyser.hs
--
-- Dieser Modul stellt eine Funktion `analyse' zur Verfuegung, die zu dem
-- Syntaxbaum eines PSA-Programms die Symboltabelle aufbaut und anhand
-- dieser Symboltabelle die kontextsensitiven Nebenbedingungen der PSA-Syntax
-- ueberprueft.

module PsaAnalyser(analyse, module Symboltable) where

import PsaAbstractSyntax
import Symboltable
--import Observe

-- `analyse' liefert als explizites Ergebnis nur die Symboltabelle; die
-- syntaktische Korrektheit wird implizit in dem Sinne ueberprueft, dass
-- `analyse' bei einem nicht korrekt gebildeten Programm mit einer 
-- Fehlermeldung aussteigt.
-- Als Speicherplaetze fuer die einzutragenden Bezeichner werden 
-- fortlaufende Zahlen beginnend mit der 1 verwendet.

analyse	:: Program -> Symboltable

analyse (Program declarations statements)
  | (correctStatementSequence symboltable statements) = symboltable
  | otherwise	                     = error "Variable not declared."
  where
  symboltable = makeSymboltable (flattenDecl declarations)


-- `flatten' wandelt eine Liste von Deklarationen in eine einfache Liste 
-- von Variablenbezeichnern um.

flattenDecl :: [Declaration] -> [Identifier]

flattenDecl 
  = foldr appendDecl [] 
  where
  appendDecl :: Declaration -> [Identifier] -> [Identifier]
  appendDecl (Decl identifiers1) identifiers2 = identifiers1 ++ identifiers2  


-- Erstellen der Symboltabelle aus der Liste alle Variablennamen

makeSymboltable :: [String] -> Symboltable

makeSymboltable identifiers
  = fst (foldl insertIdentifier (tableEmpty, 1) identifiers)
  where
  insertIdentifier :: (Symboltable, Int) -> String -> (Symboltable, Int)
  insertIdentifier (symboltable, memoryAddress) identifier
    | (declared identifier symboltable) 
      = error "Variable declared twice."
    | otherwise	
      = (tableInsert identifier memoryAddress symboltable, memoryAddress+1)


-- `declared' testet, ob ein Variablenbezeichner in der Symboltabelle
-- eingetragen ist.

declared :: String -> Symboltable -> Bool

declared identifier symboltable = (tableRead identifier symboltable) /= Nothing


-- `correct' testet, ob alle verwendeten Variablenbezeichner auch
-- deklariert sind.
-- Wir verwenden eine Klasse, um die Einheitlichkeit der Funktion fuer
-- die unterschiedlichen Syntaxelemente (Ausdruecke und Anweisungen)
-- zu betonen.

correctStatementSequence :: Symboltable -> StatementSequence -> Bool

correctStatementSequence symboltable (StmtSeq statements)
  = and (map (correctStatement symboltable) statements)


correctStatement :: Symboltable -> Statement -> Bool

correctStatement symboltable (Assignment identifier expression)
  = declared identifier symboltable && correctExpression symboltable expression
correctStatement symboltable (If bExpression statements1 statements2)
  = correctBExpression symboltable bExpression &&
    correctStatementSequence symboltable statements1 &&
    correctStatementSequence symboltable statements2
correctStatement symboltable (While bExpression statements)
  = correctBExpression symboltable bExpression && 
    correctStatementSequence symboltable statements


correctExpression :: Symboltable -> Expression -> Bool  

correctExpression symboltable (Variable identifier) 
  = declared identifier symboltable
correctExpression symboltable (Number _)
  = True
correctExpression symboltable (Compound expression1 _ expression2)
  = correctExpression symboltable expression1 && 
    correctExpression symboltable expression2


correctBExpression :: Symboltable -> BExpression -> Bool

correctBExpression symboltable (BExpression expression1 _ expression2)
  = correctExpression symboltable expression1 && 
    correctExpression symboltable expression2

-- Ende










