-- PsaCodeGenerator.hs
--
-- Uebersetzung eines Syntaxbaumes und der zugehoerigen Symboltabelle
-- in Befehle fuer die Abstrakte Maschine.


module PsaCodeGenerator(generateCode, 
                        Program, Symboltable, Table,
                        MachineProgram, MachineInstruction) where

import PsaAbstractSyntax
import Symboltable(Symboltable, Table, tableRead)
import MachineInstructions(MachineProgram, MachineInstruction(..))

-- import Observe

-- Die Uebersetzung erfolgt durch `generateCode'.

generateCode :: Program -> Symboltable -> MachineProgram

generateCode (Program _ statementSequence) symboltable 
  = {- [Halt] ++ -} programCode ++ [Halt]
  where
  (programCode, _) = translateStatementSequence symboltable 0 statementSequence


-- Es folgen die Spezifikationen der eigentlichen Uebersetzungsfunktion(en).
--
-- Fuer jeden Typ von Sprachkonstrukten (Anweisungsfolgen, Anweisungen,
-- arithmetische und boolesche Ausdruecke) gibt es eine eigene
-- Uebersetzungsfunktion. 

-- Da zur Bestimmung der Address von Sprungzielen das Codelaengenverfahren
-- eingesetzt wird, ergibt die Uebersetzung eines Teils des Syntaxbaumes
-- ein Maschinenprogramm plus dessen Laenge.

type IntermediateResult = (MachineProgram, Int)


-- Alle Uebersetzungsfunktionen benoetigen die Symboltabelle aus der semantischen
-- Analyse. Die Uebersetzungsfunktionen, die Sprungbefehle erzeugen, benoetigen
-- noch einen weiteren Parameter: die Anfangsadresse des von ihnen zu
-- erzeugenden Codes.
-- Wir verwenden eine Klasse um die Einheitlichkeit der Uebersetzungsfunktionen
-- zu betonen. 
-- (Freilich erhalten dadurch  alle Uebersetzungsfunktionen eine Anfangsadresse
-- als Parameter, obwohl diese zur Uebersetzung  arithmetischer und boolescher 
-- Ausdruecke nicht benoetigt wird. Im Hinblick auf eine denkbare Erweiterung 
-- der Sprache PSA mag dies jedoch auch nuetzlich sein.)


translateStatementSequence :: 
  Symboltable -> Int -> StatementSequence -> IntermediateResult

translateStatementSequence symboltable startAddress (StmtSeq stmtlist)
    = foldl combineTranslate ([], 0) stmtlist
    where
    combineTranslate :: IntermediateResult -> Statement -> IntermediateResult
    combineTranslate (code, codeLength) stmt
      = (code ++ stmtCode, codeLength + stmtCodeLength)
      where
      (stmtCode, stmtCodeLength) 
	= translateStatement symboltable (startAddress + codeLength) stmt



translateStatement symboltable startAddress (Assignment identifier exp)
    = (expCode ++ [Sto varAddress], expCodeLength+1)
    where
    (expCode, expCodeLength) = translateExpression symboltable startAddress exp
    Just varAddress = tableRead identifier symboltable

--translateStatement symboltable startAddress (If bExp stmts stmts2)
translateStatement symboltable startAddress (While bExp stmts)
    = (bExpCode ++ [Jmc endWhileCode] ++ stmtsCode ++ [Jmp startAddress], 
      whileCodeLength)
    where
    (bExpCode, bExpCodeLength)   = 
      translateBExpression symboltable startAddress bExp
    (stmtsCode, stmtsCodeLength) = 
      translateStatementSequence symboltable beginStmtsCode stmts
    beginStmtsCode  = startAddress + bExpCodeLength + 1
    endWhileCode    = startAddress + whileCodeLength
    whileCodeLength = bExpCodeLength + stmtsCodeLength + 2

translateStatement symboltable startAddress (If bExp stmts1 stmts2)
    = (bExpCode ++ [Jmc beginStmts2Code] ++ stmts1Code ++ [Jmp endIfCode] ++ 
      stmts2Code, ifCodeLength)
    where
    (bExpCode, bExpCodeLength)     = 
      translateBExpression symboltable startAddress bExp
    (stmts1Code, stmts1CodeLength) = 
      translateStatementSequence symboltable beginStmts1Code stmts1
    (stmts2Code, stmts2CodeLength) = 
      translateStatementSequence symboltable beginStmts2Code stmts2
    beginStmts1Code = startAddress + bExpCodeLength + 1
    beginStmts2Code = beginStmts1Code + 1 + stmts1CodeLength
    endIfCode       = beginStmts2Code + stmts2CodeLength
    ifCodeLength    = endIfCode - startAddress



translateExpression symboltable startAddress (Variable identifier)
    = ([Lod varAddress], 1)
    where
    Just varAddress = tableRead identifier symboltable

translateExpression symboltable startAddress (Number number) = 
  ([Lit number], 1)

translateExpression symboltable startAddress (Compound exp1 arithOp exp2)
    = (exp1Code ++ exp2Code ++ [machineOperation arithOp], compoundCodeLength)
    where
    (exp1Code, exp1CodeLength) = 
      translateExpression symboltable startAddress exp1
    (exp2Code, exp2CodeLength) = 
      translateExpression symboltable beginExp2Code exp2
    beginExp2Code      = startAddress + exp1CodeLength
    compoundCodeLength = exp1CodeLength + exp2CodeLength + 1
    machineOperation Plus  = Ad  -- ein Array zur Uebersetzung waere effizienter
    machineOperation Minus = Sb
    machineOperation Times = Ml
    machineOperation Div   = Dv


translateBExpression symboltable startAddress (BExpression exp1 relOp exp2)
    = (exp1Code ++ exp2Code ++ [machineOperation relOp], bExpressionCodeLength)
    where
    (exp1Code, exp1CodeLength) = 
      translateExpression symboltable startAddress exp1
    (exp2Code, exp2CodeLength) = 
      translateExpression symboltable beginExp2Code exp2
    beginExp2Code         = startAddress + exp1CodeLength
    bExpressionCodeLength = exp1CodeLength + exp2CodeLength + 1
    machineOperation Equal          = Eq    
    machineOperation NotEqual       = Ne    
    machineOperation GreaterThen    = Gt    
    machineOperation GreaterOrEqual = Ge    
    machineOperation LessThen       = Lt    
    machineOperation LessOrEqual    = Le    

-- Ende
