-- PsaParser.hs
--
-- Der Parser fuer PSA.

-- Als Eingabe erhaelt der Parser Token aus `PsaToken'.
-- Als Ausgabe produziert er den Syntaxbaum gemaess `PsaAbstractSyntax'.
-- Er stuetzt sich auf das Modul `Parser'.

module PsaParser(parsePsa) where

import Parser
import PsaToken
import PsaAbstractSyntax


-- Der Parser fuer die Sprache PSA

parsePsa :: [Token] -> Program

parsePsa = detCParser programParser


-- Es folgt die Grammatik fuer PSA.
-- Zu jeder Regel der Grammatik wird der zugehoerige Teilparser angegeben.

-- Program ::= Declaration* Statement-Sequence .

programParser :: CParser Token Program

programParser 
  = star declarationParser <*> statementSequenceParser <*> lit FullStop <@ 
    makeProgram
  where
  makeProgram (decls, (stmtSeq, fullstop)) = Program decls stmtSeq


-- Declaration = var Variable+ ;

declarationParser :: CParser Token Declaration

declarationParser 
  = lit VarSym <*> oneOrMore identifierParser <*> lit Semicolon <@ makeDeclaration
  where
  recognizeAndModifyVariable :: Token -> Maybe Identifier
  recognizeAndModifyVariable (Ident variableName) = Just variableName
  recognizeAndModifyVariable _			  = Nothing
  makeDeclaration (var, (identifiers, semicolon)) = Decl identifiers  


identifierParser :: CParser Token Identifier

identifierParser = pToken recognizeAndModifyIdentifier
  where
  recognizeAndModifyIdentifier :: Token -> Maybe Identifier
  recognizeAndModifyIdentifier (Ident identifierName) = Just identifierName
  recognizeAndModifyIdentifier _		      = Nothing


-- Statement-Sequence = Statement*

statementSequenceParser :: CParser Token StatementSequence

statementSequenceParser = star statementParser <@ StmtSeq


-- Statement ::= Variable := Expression 
--           | if BExpression then Statement-Sequence else Statement-Sequence end 
--           | while BExpression do Statement-Sequence end 

statementParser :: CParser Token Statement

statementParser = assignmentParser <|> ifParser <|> whileParser


assignmentParser, ifParser, whileParser :: CParser Token Statement

assignmentParser 
  = identifierParser <*> lit AssignOp <*> expressionParser <@ makeAssignment
  where
  makeAssignment (identifier, (assignOp, expression)) 
    = Assignment identifier expression


ifParser 
  = lit IfSym <*> bExpressionParser <*> lit ThenSym <*> 
    statementSequenceParser <*> lit ElseSym <*> statementSequenceParser <*> 
    lit EndSym <@ makeIf
  where
  makeIf (ifSym, (bexpression, (thenSym, (stmts1, (elseSym, (stmts2, endSym))))))
    = If bexpression stmts1 stmts2

whileParser
  = lit WhileSym <*> bExpressionParser <*> lit DoSym <*> 
    statementSequenceParser <*> lit EndSym <@ makeWhile
  where
  makeWhile (whileSym, (bexpression, (doSym, (stmts, endSym))))
    = While bexpression stmts


-- Expression ::= Variable 
--              |  Number 
--              |  (Expression ArithOp Expression)

expressionParser :: CParser Token Expression

expressionParser = identifierParser <@ Variable <|> 
		   numberParser <@ Number <|> 
		   compoundParser


numberParser :: CParser Token Int

numberParser = pToken recognizeAndModifyNumber
  where
  recognizeAndModifyNumber :: Token -> Maybe Int
  recognizeAndModifyNumber (Num number) = Just number
  recognizeAndModifyNumber _	        = Nothing


compoundParser :: CParser Token Expression

compoundParser 
  = lit OpenPar <*> expressionParser <*> arithOpParser <*> expressionParser <*>
    lit ClosePar <@ makeCompound
  where
  makeCompound (openPar, ( expr1, (arithOp, (expr2, closePar))))
    = Compound expr1 arithOp expr2


arithOpParser :: CParser Token ArithOp

arithOpParser = pToken recognizeAndModifyArithOp
  where
  recognizeAndModifyArithOp :: Token -> Maybe ArithOp
  recognizeAndModifyArithOp (ArithOp arithOp) = Just arithOp
  recognizeAndModifyArithOp _	              = Nothing


-- BExpression ::= Expression RelOp Expression

bExpressionParser :: CParser Token BExpression

bExpressionParser = 
  expressionParser <*> relOpParser <*> expressionParser <@ makeBExpression
  where
  makeBExpression (expr1, (relOp, expr2)) = BExpression expr1 relOp expr2


relOpParser :: CParser Token RelOp

relOpParser = pToken recognizeAndModifyRelOp
  where
  recognizeAndModifyRelOp :: Token -> Maybe RelOp
  recognizeAndModifyRelOp (RelOp relOp) = Just relOp
  recognizeAndModifyRelOp _	        = Nothing

-- Ende
