{-------------------------------------------------------------------------------

	module:		Object.hs

	author:		Bernie Pope

	date:		September 1998

	notes:		implements the object module for the raytracer.

-------------------------------------------------------------------------------}

module Object where

import Data
import Vector 

default ()


intersectLineObject :: LineSegment -> Object -> [Coordinate3D] 

intersectLineObject (p1, p2) (Sphere sc radius _ _ _ _)
   | behaviour < 0 = [] 
   | coord1 == coord2 = [coord1]
   | otherwise = [coord1, coord2]
   where
   coord1 = Coord3D ((x2 - x1) * intersect1) 
                    ((y2 - y1) * intersect1)
                    ((z2 - z1) * intersect1)
   coord2 = Coord3D ((x2 - x1) * intersect2)
                    ((y2 - y1) * intersect2)
                    ((z2 - z1) * intersect2) 
   intersect1 :: Double
   intersect1 = ((-b) + (sqrt behaviour)) / (2*a)
   intersect2 :: Double
   intersect2 = ((-b) - (sqrt behaviour)) / (2*a)
   behaviour :: Double
   behaviour = (b * b) - (4 * a * c)
   (x1, y1, z1) = (cx p1, cy p1, cz p1)
   (x2, y2, z2) = (cx p2, cy p2, cz p2)
   (x3, y3, z3) = (cx sc, cy sc, cz sc)
   a :: Double
   a = (x2 - x1)^^(2::Int) + (y2 - y1)^^(2::Int) + (z2 - z1)^^(2::Int)
   b :: Double
   b = 2 * (((x2 - x1) * (x1 - x3)) +
            ((y2 - y1) * (y1 - y3)) +
            ((z2 - z1) * (z1 - z3)))
   c :: Double
   c = x3^^(2::Int) + y3^^(2::Int) + z3^^(2::Int) +
       x1^^(2::Int) + y1^^(2::Int) + z1^^(2::Int) - 
       ((2::Double) * ((x3 * x1) + (y3 * y1) + (z3 * z1))) -
       (radius^^(2::Int))



intersectLineObject (p1, p2) (Polygon pts _ _ _ _ (Plane3D a b c d))
   | denominator == 0 = []
   | insidePoly planeIntersect pts = [planeIntersect]
   | otherwise = []
   where
   planeIntersect = Coord3D ((x2 - x1) * u)
                            ((y2 - y1) * u)
			    ((z2 - z1) * u)
   x1 = cx p1
   y1 = cy p1
   z1 = cz p1
   x2 = cx p2
   y2 = cy p2
   z2 = cz p2
   u = numerator / denominator 
   numerator = a * x1 + b * y1 + c * z1 + d 
   denominator = a * (x1 - x2) + b * (y1 - y2) + c * (z1 - z2)


insidePoly :: Coordinate3D -> [Coordinate3D] -> Bool

insidePoly p1 pts
   = allEqual (map (sideOf p1) (listPairs pts))
   where
   listPairs xs = zip xs (tail xs)

-- not quite right, need to cope with points on the line

sideOf :: Coordinate3D -> LineSegment -> Bool 

sideOf p1 (p2, p3)
   | mag <= 0 = True 
   | mag > 0 = False 
   where
   mag = ((y1 - y2) * (x3 - x2)) - ((x1 - x2) * (y3 - y2))
   x1 = cx p1
   y1 = cy p1
   x2 = cx p2
   y2 = cy p2
   x3 = cx p3
   y3 = cy p3 

allEqual :: Eq a => [a] -> Bool

allEqual [] = True 
allEqual (x:xs) = and [x == y | y <- xs] 

objectNormal :: Object -> Coordinate3D -> Vector3D

objectNormal (Sphere (Coord3D sx sy sz) radius _ _ _ _) (Coord3D px py pz)
   = Vect3D normx normy normz
   where
   normx = (px - sx) / radius
   normy = (py - sy) / radius
   normz = (pz - sz) / radius

-- Polygon normal can be computed from the coeffs of its plane

objectNormal (Polygon _ _ _ _ _ (Plane3D a b c d)) _
   = normalise (Vect3D a b c)

specularConstant :: Object -> Double
specularConstant (Sphere _ _ _ _ ks _)  = ks 
specularConstant (Polygon _ _ _ ks _ _) = ks 

lambertianConstant :: Object -> Double
lambertianConstant (Sphere _ _ _ kd _ _)  = kd
lambertianConstant (Polygon _ _ kd _ _ _) = kd

ambientConstant :: Object -> Double
ambientConstant (Sphere _ _ ka _ _ _)  = ka 
ambientConstant (Polygon _ ka _ _ _ _) = ka 

specularExponent :: Object -> Double
specularExponent (Sphere _ _ _ _ _ se)  = se 
specularExponent (Polygon _ _ _ _ se _) = se 

objectName :: Object -> String
objectName (Sphere _ _ _ _ _ _)  = "sphere"
objectName (Polygon _ _ _ _ _ _) = "polygon"
