{-------------------------------------------------------------------------------

        module:         Parser.hs

        author:         Bernie Pope

        date:           September 1998

        notes:         	An input parser for the raytracer. 

-------------------------------------------------------------------------------}

module Parser where

import ParseLib
import Data
import Char
import Geometry

data Finish = 
   Finish { 
	    fka::Double,
	    fkd::Double,
	    fks::Double,
	    fn::Double
          }

type State = (Finish, Scene)

defaultState
   = (defaultFinish, ([], []))
   where
   defaultFinish = Finish 0.5 0.5 0.5 10

myParse :: [FilePath] -> IO Scene

myParse filenames
   = do
        input <- mapM readFile filenames
        return (checkParseResult (papply (topParse defaultState) (concat input ++ "EOF")))

checkParseResult :: [(State, String)] -> Scene

checkParseResult [] = ([], [])

checkParseResult (((_, scene), _):_) = scene


{-

	Grammar for the input.

	ManyLines  -> OneLine "\n" ManyLines | OneLine EOF
	OneLine    -> Sphere | Light | Finish
	Sphere     -> Sx Sy Sz Rx Ry Rz Tx Ty Tz
	Light      -> Intensity X Y Z
	Finish     -> Ka Kd Ks N

-}

topParse :: State -> Parser State

topParse oldState
   = do { maybespaces; manyLines oldState }

manyLines :: State -> Parser State 

manyLines oldState 
   = do { newState <- oneLine oldState; maybespaces; 
          string "EOF"; return newState} 
     `mplus` 
     do { newState <- oneLine oldState; maybespaces; manyLines newState } 

oneLine :: State -> Parser State 

oneLine oldState@(oldFinish, oldScene) 
   = do { s1 <- sphere oldFinish; return (oldFinish, addObjects s1 oldScene)}
     `mplus` 
     do { l1 <- light; return (oldFinish, addLight l1 oldScene) }
     `mplus` 
     do { newFinish <- finish; return (newFinish, oldScene) }

sphere :: Finish -> Parser [Object] 

sphere oldFinish@(Finish ka kd ks n) 
   = do { symbol "sphere"; 
          sx <- double; sy <- double; sz <- double;
	  rx <- double; ry <- double; rz <- double;
	  tx <- double; ty <- double; tz <- double;
	  return [(Sphere (Coord3D tx ty tz) sx ka kd ks n)]
        }

light :: Parser Light

light 
   = do { symbol "light"; 
          intensity <- double; x <- double; y <- double; z <- double;
          return (Light (Coord3D x y z) intensity)
        }

finish :: Parser Finish

finish
   = do { symbol "finish"; 
          ka <- double; kd <- double; ks <- double; n <- double;
          return (Finish ka kd ks n)
        }

double :: Parser Double

double
   =
     do { maybespaces; 
          s <- sign;
          prefix <- many digit; dot <- char '.'; suffix <- many digit;
          return (s * (read (prefix ++ "." ++ suffix))) }
     `mplus`
     do { maybespaces; 
          s <- sign;
          prefix <- many digit; return (s * (read prefix)) }

sign :: Parser Double 

sign 
   = do { char '-'; return (-1.0) }
     `mplus` 
     do { return 1.0 }

addObjects :: [Object] -> Scene -> Scene

addObjects objs (objects, lights) = (objs++objects, lights)

addLight :: Light -> Scene -> Scene

addLight l1 (objects, lights) = (objects, l1:lights)

maybespaces :: Parser ()

maybespaces = do {many (sat isSpace); return ()}

list :: Parser a -> Parser [a]

list p
   = do {char '[';
         maybespaces;
         elements 
	    <- p `sepby` do {maybespaces; (char ','); maybespaces; return ()};
         maybespaces;
         char ']';
         return elements}
