{-------------------------------------------------------------------------------

	module:		RayTrace.hs

	author:		Bernie Pope

	date:		September 1998

	notes:		implements the actual raytracing and rendering
			functions

-------------------------------------------------------------------------------}

module RayTrace where

import Data
import Object
import Vector

rayTrace :: Scene -> Int -> Int -> Double -> Image

rayTrace scene width height focal_length
   = [render (coord x y) scene | y <- yrange, x <- xrange]  
   where
   coord x y = Coord3D (fromIntegral x) (fromIntegral y) (-focal_length)
   xrange = [xlo .. xlo + width - 1]
   yrange = reverse [ylo .. ylo + height - 1]
   xlo = -(width `div` 2)
   ylo = -(height `div` 2)


render :: Coordinate3D -> Scene -> Int

render point (objects, lights) 
   = case (getIntersect (Coord3D 0 0 0, point) objects) of
        Nothing       -> 0
	Just (p1, o1) -> toPixel (getTotalIntensity p1 o1 lights)

getTotalIntensity :: Coordinate3D -> Object -> [Light] -> Intensity 

getTotalIntensity p1 _ [] = 0 

getTotalIntensity p1 object (l:lights)
   = (shade p1 object l) + (getTotalIntensity p1 object lights)
	
shade :: Coordinate3D -> Object -> Light -> Intensity 

shade p1@(Coord3D px py pz) object (Light (Coord3D lx ly lz) intensity)
   = lamb + spec 
   where
      lightVector = normalise (Vect3D (lx - px) (ly - py) (lz - pz))
      normalVector = objectNormal object p1 
      lamb = lambertian lambConst lightVector normalVector intensity
      spec = 0
      lambConst = lambertianConstant object


getIntersect :: LineSegment -> [Object] -> Maybe (Coordinate3D, Object)

getIntersect line objs
   = foldl filterClosest Nothing pairs 
   where
   pairs = [(i, o) | o <- objs, i <- intersectLineObject line o]


filterClosest :: Maybe (Coordinate3D, Object) -> 
		 (Coordinate3D, Object) -> Maybe (Coordinate3D, Object)

filterClosest Nothing new = Just new
filterClosest prev@(Just (Coord3D _ _ prevz, _)) 
	      new@(newp@(Coord3D _ _ newz),newo) 
   = if (abs prevz < abs newz) then prev else Just new

lambertian ::  Double -> Vector3D -> Vector3D -> Intensity -> Intensity 

lambertian kd lightVector normalVector intensity 
   | product < 0 = 0
   | otherwise = intensity * kd * product
   where
   product = dotProduct normalVector lightVector

toPixel :: Double -> Int

toPixel i
   | pixel < minPixel = minPixel 
   | pixel > maxPixel = maxPixel 
   | otherwise = pixel
   where
   pixel = round (pixelScale * i)
   maxPixel = 255
   minPixel = 0
   pixelScale = 255
