\documentclass[a4paper,10pt]{article}
% We define the code environment to use the verbatim environment.
\usepackage{verbatim}
\newenvironment{code}{\verbatim}{\endverbatim}
% Uncomment the next line if you want to include eps graphics
% \usepackage[dvips]{graphicx}
% The following three lines are a typical way of including an eps graphic:
% \begin{center}
% \includegraphics[width=0.5\textwidth,height=0.2\textheight]{/home/dan/latex/epsGraphics/fileName}
% \end{center}
% Uncomment the next line if you want to use pdf hyperlinks:
% \usepackage[colorlinks=true]{hyperref}
% Comment out the next two lines if you do want paragraph indentation:
\setlength{\parindent}{0pt}
\setlength{\parskip}{1ex plus 0.5ex minus 0.2ex}

\begin{document}
\title{Quine-McCluskey Minimization algorithm}
\author{Daniel David Parry}
\maketitle
\tableofcontents

\section{The Program, implemented in Haskell}

\subsection{The module declaration and main function}

Note that the List module is imported to gain a little extra
functionality.

\begin{code}
module Main where
import List

main :: IO()
main = do putStr "-------------Quine McCluskey-------------\n"
          putStr "Use / to denote not, i.e. /A = not A.\n"
          putStr "Enter the expression to be minimized:\n"
          input <- getLine
          putStr "The minimized expression is:\n"
          putStr (quineMcCluskey input ++ "\n")
          putStr "-----------------------------------------\n"
\end{code}

\subsection{My own Types}

I declare the following types, mainly for cosmetic reasons:

\begin{code}
type Term        = [Int]
type MarkedTerm  = (Term,Bool)
type PI          = (Term,[Bool])
type Parse       = ([String],String)
\end{code}

\begin{itemize}

\item Term represents the basic Int list datatype.

\item A MarkedTerm is made up of a Term, and an additional Bool
denoting whether it is a Prime Implicant or not.

\item PI represents prime implicants. The Boolean list denotes
the function minterms that the prime implicant covers.

\item Parse represents the datatype created when the program's
input is parsed.

\end{itemize}

\subsection{Parsing the program's input}

notNegated returns False if the negated form of a character
appears in a String, i.e. notNegated `A' ``/A'' will return
False:

\begin{code}
notNegated :: Char -> String -> Bool -> Bool
notNegated _        []  _ = True
notNegated x ('/':b:bs) t = (not t || x/=b) && notNegated x bs t
notNegated x     (b:bs) t = (    t || x/=b) && notNegated x bs t
\end{code}

canBeTrue returns False if a String contains a boolean
expression that always evaluates to False, such as ``A/A'':

\begin{code}
canBeTrue :: String -> Bool
canBeTrue ('/':x:xs) = notNegated x xs False && canBeTrue xs
canBeTrue     (x:xs) = notNegated x xs True  && canBeTrue xs
canBeTrue       _    = True
\end{code}

parse returns a tuple of type Parse by doing the following:

\begin{enumerate}

\item splits a list of characters separated by the plus
character, '+', into separate lists, e.g. ``AB+AC'' [ ]
$\rightarrow$ [``AB'',``AC''],

\item makes a list of unique characters from a String, e.g.
``CB+AC'' $\rightarrow$ ``ABC'',

\end{enumerate}

\begin{code}
parse :: String -> String -> [String] -> String -> Parse
parse      []  ls ms vs = (filter canBeTrue (ls:ms),  vs)
parse (' ':xs) ls ms vs = parse xs  ls       ms       vs
parse ('+':xs) ls ms vs = parse xs  []   (ls:ms)      vs
parse ('/':xs) ls ms vs = parse xs (ls++"/") ms       vs
parse (  x:xs) ls ms vs = parse xs (ls++[x]) ms (nub (vs++[x]))
\end{code}

\subsection{Converting the characters to a binary
representation}

updateTerm finds the position of a character in the variable
list, and changes the value of the element in the equivalent
position in a Term to a specified Int value, e.g. updateTerm 1
[0,1,-1] 'C' "ABC" $\rightarrow$ [0,1,1]:

\begin{code}
updateTerm :: Int -> Term -> Char -> String -> Term
updateTerm i ns c vs = zipWith (\ n t -> if t then i else n)
                         ns (map (c ==) vs)
\end{code}

convert changes a String to a Term. It considers the positions
of the characters in the String to be converted with respect to
the variable list, and also considers whether they are negated
(by the '/' character). e.g. convert [-1,-1,-1] ``ABC'' ``/BA''
$\rightarrow$ [1,0,-1]:

\begin{code}
convert :: Term -> String -> String -> Term
convert t _         []  = t
convert t vs ('/':x:xs) = convert (updateTerm 0 t x vs) vs xs
convert t vs (    x:xs) = convert (updateTerm 1 t x vs) vs xs
\end{code}

charsToInts maps the convert function onto a list of Strings:

\begin{code}
charsToInts :: Parse -> [Term]
charsToInts (xs,vs) = map (convert (map (\ v -> -1) vs) vs) xs
\end{code}

\subsection{Expanding the Don't care Values}

expandDontCares applies expand to all the terms repeatedly until
all the -1 values are replaced with 0s or 1s; expand takes a
list of Terms and either: replaces the first occurrence of -1 in
each of the Terms with 0 and 1 forming two new terms or just
returns the Term unchanged if it contains no -1s:

\begin{code}
expandDontCares :: [Term] -> [Term]
expandDontCares xs = if (xs==es) then xs else expandDontCares es
  where es = concatMap (expand []) xs
        expand ns     []  = [ns]
        expand ns (-1:ys) = (ns++[1]++ys) : [ns++[0]++ys]
        expand ns ( y:ys) = expand (ns++[y]) ys
\end{code}

\subsection{Grouping the Terms by number of bits set to 1}

addTermInfo takes a list of Terms and returns a list of tuples
of the form: ((Term,True), number of ones in Term). The Boolean
part denotes whether Terms are prime implicants. Knowing the
number of ones allows us to group the Terms, so that Terms from
adjacent groups can be compared and combined:

\begin{code}
addTermInfo :: [Term] -> [(MarkedTerm,Int)]
addTermInfo xs = map (\ x -> ( (x, True), foldl (+) 0 x) ) xs
\end{code}

groupMinterms puts the MarkedTerms into lists, based on the
number of ones in the term. The length of a Term is l's initial
value. All Terms with `l' lots of 1's are added to a list, which
is concatenated onto the recursive call to groupMinterms. The
recursive call's arguments are the MarkedTerms whose number of
ones did not equal l, with one added to their number of ones;
and l:

\begin{code}
groupMinterms :: [(MarkedTerm,Int)] -> [[MarkedTerm]]
groupMinterms [] = []
groupMinterms ls = [(xs,t)   | ((xs,t),0)<-ls] :
 groupMinterms [((xs,t),n-1) | ((xs,t),n)<-ls, n/=0]
\end{code}

\subsection{Generate next set of Terms}

makeTerm has a tuple of Terms as its argument. It returns the
Term that contains the elements that matched exactly in terms of
value and position in the input Terms, and where the elements
did not match exactly their value is substituted with that of
-1, e.g. makeTerm ([1,1,0,1],[1,0,0,1]) $\rightarrow$
[1,-1,0,1]:

\begin{code}
makeTerm :: (Term,Term) -> (Term,Bool)
makeTerm (xs,ys) = (zipWith compare xs ys, True)
  where compare x y = if (x == y) then x else -1
\end{code}

makeTerms takes two lists of MarkedTerms and returns in the form
of a tuple:

\begin{enumerate}

\item The prime implicants in the first list that can not be
combined with any elements in the second (xsReduced);

\item The new terms made by combining elements in the first list
with those that differed by one element in the second (xsMade);

\item The second list with those elements used in forming new
elements marked as not being prime implicants:

\end{enumerate}

\begin{code}
makeTerms :: [MarkedTerm] -> [MarkedTerm] ->
                     (([Term],[MarkedTerm]),[MarkedTerm])
makeTerms xs ys = ((xsReduced, xsMade), ysTicked)
  where
    differBy1s = [(x,y)|(x,_)<-xs, (y,_)<-ys,
      1 == (length $ filter (False ==) $ zipWith (==) x y)]
    ysUsed     = map snd differBy1s
    xsMade     = nub (map makeTerm differBy1s)
    xsReduced  = map fst (filter snd xs) \\ map fst differBy1s
    ysTicked   = map (\ (y,t) -> (y, y `notElem` ysUsed)) ys
\end{code}

takePIs returns the list of all Terms marked as prime
implicants:

\begin{code}
takePIs :: [MarkedTerm] -> [Term]
takePIs xs = map fst (filter snd xs)
\end{code}

createTerms returns a list of prime implicant Terms from a list
of MarkedTerms:

\begin{code}
createTerms :: [[MarkedTerm]] -> [[MarkedTerm]] -> [Term]
createTerms ns     []     =
  if concat ns /= [] then createTerms [] ns else []
createTerms _  (xs:[])    = takePIs xs
createTerms ns (xs:ys:ls) 
  | ls == []  = takePIs (snd newTerms) ++ ((fst.fst) newTerms)
                ++ createTerms (((snd.fst) newTerms):ns) []
  | otherwise = (fst.fst) newTerms ++ createTerms
                 (((snd.fst) newTerms):ns) ((snd newTerms):ls)
  where newTerms = makeTerms xs ys
\end{code}

\subsection{Select smallest set of Terms to cover Minterms}

testCoverage assigns each Term a Boolean list as the second
argument of a tuple representing the Terms that they cover;
testCovers returns True if wherever an element in the first Term
does not have a value of -1, it instead has the same value as
the corresponding element in the second Term:

\begin{code}
testCoverage ::[Term] -> [Term] -> [PI]
testCoverage xs ys = map (\ y -> (y, map (testCovers y) xs) ) ys
  where testCovers xs ys = 0 == length (filter (False ==)
          (zipWith (\ a b -> a == -1 || a == b ) xs ys))
\end{code}

reducePIs compares all the PIs with each other and removes those
not uniquely covering any minterms\footnote{Note:
$\backslash\backslash$ removes the first appearance of an
element from a list.}; 

\begin{code}
reducePIs :: [PI] -> [PI]
reducePIs xs = filter (\ x -> or $ foldl (zipWith
  (\ a b -> not (a&&b) && a)) (snd x) (map snd (xs\\[x]))) xs
\end{code}

\subsection{Converting the binary representation back}

outputs maps the function output onto a list of Terms; output
converts the Terms to Strings, adding the negation character,
'/', where appropriate.  For example, output [1,0,1] ``ABC''
$\rightarrow$ ``A/BC''.

\begin{code}
outputs :: String -> [Term] -> [String]
outputs vs xs = map (output vs) xs
  where output as bs = concat $ zipWith (\ a b ->
          if (b==1) then     [a] else
          if (b==0) then ['/',a] else []) as bs
\end{code}

outputString concatenates `` + '' between Strings, i.e.
[``A'',``B''] $\rightarrow$ ``A + B''.

\begin{code}
outputString :: [String] -> String
outputString  []  = "False"
outputString [""] = "True"
outputString  xs  = unwords $ intersperse "+" xs
\end{code}

\subsection{Bringing it all together}

quineMcCluskey defines the order in which the above functions
are called:

\begin{code}
quineMcCluskey :: String -> String
quineMcCluskey [] = "No Expression Entered!"
quineMcCluskey xs = outputString $ outputs (snd parseList) $
  map fst $ reducePIs $ testCoverage expandSet $ createTerms []
  $ groupMinterms $ addTermInfo expandSet
  where parseList = parse xs [] [] []
        expandSet = (nub.expandDontCares.charsToInts) parseList
\end{code}

\subsection{Examples of the code in operation}

An example of where simplification can be helpful, followed by
an example taken from the part 1B ECAD lecture course:

\begin{verbatim}
-------------Quine McCluskey-------------
Enter the expression to be minimized.
Use / to denote not, i.e. /A = not A.
a/b + abc + abd/c + a/cb/de + a/c/db
The minimized expression is:
a
-----------------------------------------

-------------Quine McCluskey-------------
Use / to denote not, i.e. /A = not A.
Enter the expression to be minimized:
/a/b/c/d + /acd + ab/c + ab/d + bcd
The minimized expression is:
/a/b/c/d + /acd + ab
-----------------------------------------
\end{verbatim}

\end{document}
