-- simple arithmetic langauge
-- redundant bracketing not allowed
-- a term on its own is not allowed (code must have an operator)

module Main where

import Char (isSpace, isDigit)

data Expr
   = Plus Term Term |
     Minus Term Term  |
     Mult Term Term |
     Div Term Term
     deriving (Eq, Show)

data Term
   = Val Int |
     Brack Expr
     deriving (Eq, Show)

-- evaluate an expr

calcExpr :: Expr -> Int

calcExpr (Plus term1 term2)
   = (calcTerm term1) + (calcTerm term2)

calcExpr (Minus term1 term2)
   = (calcTerm term1) - (calcTerm term2)

calcExpr (Mult term1 term2)
   = (calcTerm term1) * (calcTerm term2)

calcExpr (Div term1 term2)
   = (calcTerm term1) `div` (calcTerm term2)

-- evaluate a terminal

calcTerm :: Term -> Int

calcTerm (Val i)
   = i

calcTerm (Brack expr)
   = calcExpr expr

-- token datatype

data Token 
   = LeftBracket |
     RightBracket |
     PlusSign |
     MultSign |
     MinusSign |
     DivSign |
     Number Int |
     Unknown
     deriving (Eq, Show)

-- the lexer

lexer :: String -> [Token]

lexer xs = lexer' (dropWhile isSpace xs)

lexer' :: String -> [Token]

lexer' [] = []

lexer' (x:xs)
   | x == '(' = LeftBracket : lexer xs
   | x == ')' = RightBracket : lexer xs
   | x == '+' = PlusSign : lexer xs
   | x == '*' = MultSign : lexer xs
   | x == '-' = MinusSign : lexer xs
   | x == '/' = DivSign : lexer xs
   | isDigit x = (Number getNumber) : lexer rest 
   | otherwise = Unknown : lexer xs
   where
   getNumber = read (takeWhile isDigit (x:xs))
   rest = dropWhile isDigit (x:xs) 

-- the parser

parser :: [Token] -> Expr

parser tokens = fst (parseExpr tokens)

parseExpr :: [Token] -> (Expr, [Token])

parseExpr tokens 
   | symbol == PlusSign  = (Plus t1 t2, tokens'')
   | symbol == MinusSign = (Minus t1 t2, tokens'')
   | symbol == MultSign  = (Mult t1 t2, tokens'')
   | symbol == DivSign   = (Div t1 t2, tokens'')
   where
   (t1, symbol:tokens') = parseTerm tokens
   (t2, tokens'')       = parseTerm tokens'

parseTerm :: [Token] -> (Term, [Token])

parseTerm (t1:tokens)
   | t1 == LeftBracket = parseBracket tokens
   | otherwise = parseNumber (t1:tokens)

parseBracket :: [Token] -> (Term, [Token])

parseBracket tokens
   = (Brack expr, tokens')
   where
   (expr, rightBracket:tokens') = parseExpr tokens 

parseNumber :: [Token] -> (Term, [Token])

parseNumber (t:tokens)
   = (Val (getVal t), tokens)
   where
   getVal (Number n) = n

main = print $ calcExpr $ parser $ lexer "((3+4)-2)*(11+1)"
