{- fun with lists - implementing lists as partial functions from index
                    to value -}

module Main where

import Prelude hiding 
               ( take, map, reverse, head, tail, length, drop,
                 init, last, foldl, foldr, repeat, filter, replicate,
                 zip, iterate
               )

import Monad (liftM)
import Maybe (fromJust)

main = print $ length $ mkList "abc"

newtype List a = L (Integer -> Maybe a)

cons :: a -> List a -> List a
cons x list
   = L $ \index -> if index == 0 
                      then Just x 
                      else list `at` (index - 1)

empty :: List a
empty = L $ \index -> Nothing

isEmpty :: List a -> Bool
isEmpty list
   = case list `at` 0 of
        Nothing -> True
        _       -> False

at :: List a -> Integer -> Maybe a
at (L list) n = list n 

head :: List a -> a
head list
   = case list `at` 0 of
        Nothing -> error "empty list"
        Just a  -> a 

tail :: List a -> List a
tail list = L (\index -> list `at` (index + 1)) 

length :: List a -> Integer
length list 
   = if isEmpty list 
        then 0
        else 1 + length (tail list)

append :: List a -> List a -> List a
append list1 list2
   = case list1 `at` 0 of
        Nothing -> list2
        Just x  -> cons x (append (tail list1) list2)

nrev :: List a -> List a
nrev list
   = case list `at` 0 of
        Nothing -> empty
        Just x  -> append (nrev (tail list)) (cons x empty)

reverse :: List a -> List a
reverse list
   = L $ \index -> list `at` (len - index)
   where
   len = length list - 1

take :: Integer -> List a -> List a
take n list 
   = L $ \index -> if index >= n then Nothing else list `at` index

init :: List a -> List a
init list = take (length list - 1) list

last :: List a -> a
last list = fromJust $ list `at` (length list - 1)

drop :: Integer -> List a -> List a
drop n list = L $ \index -> list `at` (index + n)

map :: (a -> b) -> List a -> List b
map f list = L $ \index -> liftM f (list `at` index) 

instance Show a => Show (List a) where
   showsPrec _ list
      = showList list
      where
      showList list 
         = case list `at` 0 of
              Nothing -> showString "<>"
              Just x  -> showChar '<' . shows x . showL (tail list)
                         where
                         showL list 
                            = case list `at` 0 of
                                 Nothing -> showChar '>' 
                                 Just x  -> showChar ',' . shows x . showL (tail list) 

mkList :: [a] -> List a
mkList [] = empty
mkList (x:xs) = cons x (mkList xs)

repeat :: a -> List a
repeat x = L $ \index -> Just x

replicate :: Integer -> a -> List a
replicate num x
   = L $ \index -> if index < num then Just x else Nothing 

zip :: List a -> List b -> List (a,b)
zip list1 list2
   = L $ \index -> do v1 <- list1 `at` index
                      v2 <- list2 `at` index
                      return (v1, v2)

delete :: Integer -> List a -> List a
delete i list
   = L $ \index -> if index < i
                      then list `at` index 
                      else list `at` (index + 1)

filter :: (a -> Bool) -> List a -> List a
filter p list
   = case list `at` 0 of
        Nothing -> empty
        Just x  -> if p x 
                      then cons x (filter p (tail list))
                      else filter p (tail list)

iterate :: (a -> a) -> a -> List a 
iterate f x 
   = let list 
            = L $ \index -> if index == 0 
                               then Just $ f x
                               else liftM f (list `at` (index - 1))
     in list
