{-------------------------------------------------------------------------------


	File:		StackMachine.hs

	Author:		Bernie Pope

	Description:

	The code in this file implements a simple stack machine in Haskell.
	The instructions that the machine has are fairly primitive.

	We illustrate how to implement multiplication over natural
	numbers at the bottom of the file.

-------------------------------------------------------------------------------}

module Main where

-- Possible instructions for the machine

data Instruction 
   = Push Int |
     Pop      |
     Jump Int |
     Swap     |
     Add      |
     Sub      |
     Halt
     deriving (Eq, Show)

-- The stack data type

type Stack a = [a]

push :: a -> Stack a -> Stack a
push x stack = x:stack

pop :: Stack a -> Stack a
pop [] = []
pop (s:ss) = ss

top :: Stack a -> a
top [] = error "no top element"
top (x:xs) = x

swap :: Stack a -> Stack a
swap [] = []
swap [x] = [x]
swap (x1:x2:xs) = (x2:x1:xs)

add :: Stack Int -> Stack Int
add [] = []
add [x] = [x]
add (x1:x2:xs) = (x1 + x2):xs

sub :: Stack Int -> Stack Int
sub [] = []
sub [x] = [x]
sub (x1:x2:xs) = (x2 - x1):xs

-- running a stack program

runProgram :: [Instruction] -> Int 
runProgram [] = error "empty instructions" 
runProgram instrs
   = top (eval [] instrs (instrs !! 0) 0)

-- evaluate the current instruction and then determine the 
-- next instruction or halt.

eval :: Stack Int -> [Instruction] -> Instruction -> Int -> Stack Int 

eval stack instrs Halt index
   = stack

eval stack instrs (Push n) index 
   = eval newStack instrs nextInstr nextIndex
   where
   newStack = push n stack 
   nextInstr = instrs !! nextIndex
   nextIndex = index + 1

eval stack instrs Pop index 
   = eval newStack instrs nextInstr nextIndex
   where
   newStack = pop stack 
   nextInstr = instrs !! nextIndex
   nextIndex = index + 1

eval stack instrs Swap index 
   = eval newStack instrs nextInstr nextIndex
   where
   newStack = swap stack 
   nextInstr = instrs !! nextIndex
   nextIndex = index + 1

eval stack instrs Add index 
   = eval newStack instrs nextInstr nextIndex
   where
   newStack = add stack 
   nextInstr = instrs !! nextIndex
   nextIndex = index + 1

eval stack instrs Sub index 
   = eval newStack instrs nextInstr nextIndex
   where
   newStack = sub stack 
   nextInstr = instrs !! nextIndex
   nextIndex = index + 1

eval stack instrs (Jump n) index 
   | top stack == 0 
      = eval newStackJmp instrs nextInstrJmp nextIndexJmp
   | otherwise
      = eval newStackCont instrs nextInstrCont nextIndexCont
   where
   newStackJmp   = pop stack 
   nextInstrJmp  = instrs !! n 
   nextIndexJmp  = n 
   newStackCont  = stack 
   nextInstrCont = instrs !! nextIndexCont 
   nextIndexCont = index + 1 


-- mult is a program that multiplies two natural numbers
-- have to be careful with zeros.

-- also works if the second argument is negative and the first is positive.

mult x y 
   =    [ Push 0,       -- 0
          Push x,       -- 1
          Jump 9,       -- 2
          Push y,       -- 3
          Swap,         -- 4
	  Push 1,       -- 5
	  Sub,          -- 6
	  Push 0,       -- 7
	  Jump 2,       -- 8
	  Swap,         -- 9
	  Jump 14,      -- 10
	  Add,          -- 11
	  Push 0,       -- 12
	  Jump 9,       -- 13
	  Halt          -- 14
        ]

main = print $ runProgram $ mult 3 4
