{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 CLI 

        Description:            Command line interface to buddha.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{- 
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module CLI 
   ( cli
   , displayDerivation 
   , inputs
   ) where

import Data
   ( DebugState (..) 
   , Config (..)
   , Command (..)
   , Judgement (..)
   , WhatToDraw (..)
   , Input (..)
   , InputSpec (..)
   , EDT
   , Derivation (..)
   , Response (..)
   , readGlobalState
   , readGlobalStateConfig
   , updateGlobalState 
   , emptyJudgeDatabase
   )

import IOUtils

import Monad 
   ( when 
   , sequence_
   , unless
   , liftM
   )

import EDT

import Advice
   ( advice )

import Help
   ( help
   , info
   , kidsInfo
   , parentsInfo
   , setInfo
   , helpInfo
   , refreshInfo
   , adviceInfo
   , dumpInfo
   , infoInfo
   , correctInfo
   , erroneousInfo
   , inadmissInfo
   , uknownInfo
   , postponeInfo
   , quitInfo
   , drawInfo
   , forgetInfo
   , sizeInfo
   , depthInfo
   , restartInfo
   , observeInfo
   , jumpInfo
   , backInfo
   )

import Settings 
   ( set
   , getSetting
   )

import List 
   ( intersperse 
   , null
   , nub
   )

import IO

import Graph 
   ( Graph )

import Meta 
   ( Val (V) )

import GraphToExp 
   ( graphToExp )

import GraphToDotGraph
   ( graphToDotGraph )

import EDTtoDotGraph
   ( edtToDotGraph )

import DotGraph
   ( prettyDotGraph )

import PrettyExp 
   ( renderExp )

import Text.PrettyPrint as P hiding (Mode)

import Char 
   ( isSpace )

import System.Console.Readline
   ( readline 
   , addHistory
   )

import Control.Exception as E
   ( try )

import Buddha
   ( callTable 
   , funTable 
   , Record (..)
   , FunRecord (..)
   )   

import Data.IORef
   ( readIORef )

import ReifyHs 
   ( reify )

import Data.PackedString
   ( unpackPS 
   , packString
   , PackedString
   )

import Data.Array
   ( elems 
   )

import ColourString
   ( promptColour
   , valueIOLn
   , valueIO
   , decorIO
   , nameIO
   , alertIOLn
   )

--------------------------------------------------------------------------------

-- maybe show the derivation before the prompt
-- get user input and act upon it
cli :: Bool -> EDT -> IO Response
cli showNode node
   = do when showNode 
           $ do truncate <- readGlobalStateConfig config_truncate 
                newline
                displayDerivation truncate node 
                newline
        input <- getInput 
        handleInput node input 

-- do something with the user's input
handleInput :: EDT -> Input -> IO Response
handleInput node input
   = do case input of
          Cmd command     -> performCommand node command  
          Judge judgement -> return $ Left judgement

-- some commands get handled here, others get passed back to the debugger
performCommand :: EDT -> Command -> IO Response
performCommand node command 
   = case command of
        Help things -> do {help inputs things; cli False node}
        Refresh     -> cli True node
        Advice      -> do {advice; cli False node}
        Dump thing  -> do {dump thing; cli False node}
        Forget      -> do forgetJudgements 
                          cli False node 
        Size        -> do {showSize node; cli False node}
        Depth       -> do {showDepth node; cli False node}
        Set things  -> do opt <- getSetting things
                          case opt of
                             Nothing -> cli False node
                             Just settings  -> do set settings 
                                                  cli False node
        Jump index -> return $ Right $ Jump index
        ShowKids
           -> do kids <- children node
                 if null kids 
                    then tabIO $ alertIOLn "this node has no children"
                    else do newline 
                            putStrLn "Children of the current derivation:"
                            newline
                            width <- readGlobalStateConfig config_screenWidth
                            showListOfNodes width $ nub kids
                            newline
                 cli False node 
        ShowParents
           -> do ps <- parents node
                 if null ps 
                    then tabIO $ alertIOLn "this node has no parents"
                    else do newline
                            putStrLn "Parents of the current node:" 
                            newline
                            width <- readGlobalStateConfig config_screenWidth
                            showListOfNodes width (nub ps)
                            newline
                 cli False node
        Info -> do {info; cli False node} 
        Draw { whatToDraw  = thingToDraw 
             , drawFile    = file
             , drawDepth   = depth }
             -> do draw node thingToDraw file depth
                   cli False node 
        Quit -> return $ Right Quit
        Postpone -> return $ Right Postpone 
        Restart -> return $ Right Restart
        Observe name -> do { observe name; cli False node }
        GoBack -> return $ Right GoBack

forgetJudgements :: IO ()
forgetJudgements
    = updateGlobalState 
         (\s -> s { state_judgeDatabase = emptyJudgeDatabase })

-- draw an argument, result or the edt
draw :: EDT -> WhatToDraw -> FilePath -> Int -> IO ()
draw node thingToDraw file depth 
   = do case thingToDraw of
           Tree -> drawEDT node file depth 
           argOrResult -> drawArgOrResult node thingToDraw file depth

-- draw an argument or result
drawArgOrResult :: EDT -> WhatToDraw -> FilePath -> Int -> IO () 
drawArgOrResult node thingToDraw file depth
   = do deriv <- derivation node 
        toDraw <- graphToDraw deriv thingToDraw 
        case toDraw of
           Nothing -> return ()
           Just graph 
              -> do fileToDrawTo 
                       <- if null file 
                             then readGlobalStateConfig config_drawFile 
                             else return file
                    depthToDrawTo
                       <- if depth < 0
                             then readGlobalStateConfig config_drawDepth
                             else return depth
                    val <- E.try $ writeFile fileToDrawTo $ render $ prettyDotGraph $
                              graphToDotGraph depthToDrawTo graph 
                    -- catch any failed file writes here
                    case val of
                       Left exception 
                          -> tabIO $ print exception
                       Right result -> return () 
   where
   graphToDraw :: Derivation -> WhatToDraw -> IO (Maybe Graph)
   graphToDraw deriv Result 
      = return $ Just $ deriv_result deriv 
   graphToDraw deriv (Arg n)
      = do let arguments = deriv_args deriv 
           let len = length arguments
           if len < n 
              then do tabIO $ alertIOLn "argument position too large"
                      tabIO $ alertIOLn $ "must be <= " ++ show len
                      return Nothing
              else return $ Just (arguments !! (n - 1))
      

-- draw the edt from this node down
drawEDT :: EDT -> FilePath -> Int -> IO ()
drawEDT node file depth
   = do fileToDrawTo
           <- if null file
                 then readGlobalStateConfig config_drawFile 
                 else return file
        depthToDrawTo
           <- if depth < 0
                 then readGlobalStateConfig config_drawDepth 
                 else return depth
        val <- E.try $ writeFile fileToDrawTo $ render $ prettyDotGraph $
                                        edtToDotGraph depthToDrawTo [node]
        -- catch any failed file writes here
        case val of
           Left exception
              -> tabIO $ print exception
           Right result
              -> return ()

showSize :: EDT -> IO ()
showSize node
   = do treeSize <- size node
        tabIO $ putStrLn $ "tree size from this node = " ++ show treeSize 

showDepth :: EDT -> IO ()
showDepth node
   = do treeDepth <- depth node 
        tabIO $ putStrLn $ "maximum tree depth from this node = " ++ show treeDepth 

dump :: String -> IO ()
dump thing
   = case thing of
        "calls" -> dumpCallTable
        "funs"  -> dumpFunTable 
        "jumpStack" -> dumpJumpStack
        other   -> tabIO $ alertIOLn "only these things can be dumped: calls, funs or jumpStack"
   where
   dumpJumpStack :: IO ()
   dumpJumpStack
      = do newline
           tabIO $ putStrLn "--- The jump stack ---"
           newline
           stack <- readGlobalState state_jumpStack
           mapM_ dumpJumpStackEntry stack
           newline
   dumpJumpStackEntry (diagnosis, children)
      = do tabIO $ putStr $ show diagnosis
           IOUtils.space
           print children        
   dumpCallTable :: IO ()
   dumpCallTable
      = do newline 
           tabIO $ putStrLn "--- The call table ---"
           newline
           table <- readIORef callTable
           mapM_ dumpCallTableEntry table
           newline
   dumpFunTable:: IO ()
   dumpFunTable
      = do newline 
           tabIO $ putStrLn "--- The function table ---"
           newline
           table <- readIORef funTable
           mapM_ dumpFunTableEntry table
           newline
   dumpCallTableEntry :: Record -> IO ()
   dumpCallTableEntry (Rec parent child childName args result line modName)
      = do tabIO $ putStr $ unwords [show parent, show child, unpackPS childName, show line, unpackPS modName]
           IOUtils.space
           case length args of
              0 -> return ()
              1 -> do mapM_ display args
                      putStr " -> "
              n -> do putChar '('
                      sequence_ $ intersperse (putChar ',' >> IOUtils.space) $ map display args
                      putStr ") -> "
           display result
           newline
   dumpCallTableEntry (Constant ident name value line modName)
      = do tabIO $ putStr $ unwords [show ident, unpackPS name, show line, unpackPS modName]
           putStr " -> "
           display value
           newline
   dumpCallTableEntry (Ref parent child)
      = tabIO $ putStrLn $ unwords [show parent, show child]
   dumpFunTableEntry :: FunRecord -> IO ()
   dumpFunTableEntry (FunRec number arg result)
      = do tabIO $ putStr $ show number
           IOUtils.space
           display arg
           putStr " -> "
           display result
           putStr "\n"
   display :: Val -> IO ()
   display (V x)
      = do graph <- reify x
           funTable <- readGlobalState state_funTable
           let exp = graphToExp False funTable graph
           putStr $ renderExp exp

displayDerivation :: Int -> EDT -> IO ()
displayDerivation truncate node 
   = do decorIO $ "[" ++ show node ++ "]" 
        IOUtils.space
        deriv <- derivation node 
        displayDerivationHead deriv
        let arguments = deriv_args deriv
            numArgs   = length arguments
        newline
        displayArgs truncate arguments 
        displayResult truncate deriv
        newline

displayDerivationHead :: Derivation -> IO () 
displayDerivationHead d
   = do decorIO $ unpackPS $ deriv_module d
        IOUtils.space
        decorIO $ show $ deriv_sloc d
        IOUtils.space
        nameIO $ unpackPS $ deriv_name d

displayResult :: Int -> Derivation -> IO () 
displayResult truncate deriv 
   = do decorIO resultStr 
        displayGraph truncate (deriv_result deriv) 

resultStr :: String
resultStr = "   result = "

-- this tells us how far to indent the lines that display arguments and
-- results of derivations
leftwidth :: Int
leftwidth = length resultStr 

displayArgs :: Int -> [Graph] -> IO () 
displayArgs truncate args 
   = do displayArgs' 1 args 
   where
   displayArgs' :: Int -> [Graph] -> IO () 
   displayArgs' n [] = return ()
   displayArgs' n (g:gs)
      = do decorIO $ "   arg " ++ show n ++ "  = "
           displayGraph truncate g 
           newline
           displayArgs' (n+1) gs

displayGraph :: Int -> Graph -> IO () 
displayGraph truncate graph 
   = do showCycles <- readGlobalStateConfig config_showCycles 
        funTable <- readGlobalState state_funTable 
        let exp = graphToExp showCycles funTable graph
        let graphStr = renderExp exp 
        portray truncate graphStr 

portray :: Int -> String -> IO ()
portray truncate str
   = do let truncStr = truncateString truncate str
        sizeToChunk <- chunkSize 
        let chunks = chunk sizeToChunk truncStr
        unlinesIO $ indent $ chunks 
   where
   -- indent all lines except the first
   indent :: [String] -> [String]
   indent [] = []
   indent (x:ys) 
      = x : indent' ys
      where 
      indent' list = [leftSpacing ++ str | str <- list]
      leftSpacing = replicate leftwidth ' '
   chunk :: Int -> String -> [String]
   chunk _ [] = []
   chunk width str@(s:ss) 
      = top : chunk width bottom
      where
      (top, bottom) = splitAt width str
   unlinesIO :: [String] -> IO ()
   unlinesIO [] = return ()
   unlinesIO [line] = valueIO line
   unlinesIO (l1:l2:ls) 
      = do valueIOLn l1
           unlinesIO (l2:ls)

-- XXX dirty hack
chunkSize :: IO Int
chunkSize 
   = do width <- readGlobalStateConfig config_screenWidth
        let size = width - leftwidth
        if size <= 0
           then return defaultChunkSize
           else return size
   where
   defaultChunkSize = 60

-- possibly shorten a string to a given length and put
-- ellipses at the cut off point
truncateString :: Int -> String -> String
truncateString width s
   | length s <= width = s
   | otherwise = take amountToTake s ++ ellipses
   where
   amountToTake = (width - leftwidth) - length ellipses
   ellipses = " ..."

prettySrcLoc :: Int -> Doc
prettySrcLoc loc = int loc

unrecognisedInput :: String -> String
unrecognisedInput input
   = "input \"" ++  input ++ "\" not recognised (h for help, q for quit)"

-- read the input from the user
getInput :: IO Input 
getInput 
   = do promptRawText <- readGlobalStateConfig config_prompt
        promptColourText <- promptColour promptRawText 
        line <- readline $ promptColourText ++ " "
        case line of
           Nothing -> return $ Cmd $ Quit 
           Just someText
              | isEmptyLine someText -> getInput 
              | otherwise 
                   -> do addHistory someText
                         expandedInput <- expandInput someText
                         case expandedInput of
                            Nothing    -> getInput 
                            Just input -> return input 
   where
   isEmptyLine :: String -> Bool
   isEmptyLine s = all isSpace s

showListOfNodes :: Int -> [EDT] -> IO ()
showListOfNodes width kids 
   = mapM_ (displayDerivation width) kids 

--------------------------------------------------------------------------------

expandInput :: String -> IO (Maybe Input)
expandInput response
   | null $ responseWords = return $ Nothing
   | otherwise = getThisInput (head responseWords) inputs
   where
   responseWords :: [String]
   responseWords = words response
   getThisInput str [] 
      = do tabIO $ alertIOLn $ unrecognisedInput str
           return Nothing
   getThisInput str (entry:rest)
      | str `elem` in_names entry
           = (in_makeInput entry) (tail responseWords)
      | otherwise = getThisInput str rest

simpleCommand :: Input -> [String] -> IO (Maybe Input)
simpleCommand i [] = return $ Just i
simpleCommand i (_:_) 
   = do tabIO $ alertIOLn $ "this command does not have any arguments"
        return Nothing

inputs :: [ InputSpec ]
   = [ In 
       { in_names     = ["c", "correct"]
       , in_infos     = correctInfo
       , in_makeInput = simpleCommand $ Judge Correct 
       }
     , In 
       { in_names     = ["e", "erroneous"]
       , in_infos     = erroneousInfo
       , in_makeInput = simpleCommand $ Judge Erroneous
       }
     , In 
       { in_names     = ["i", "inadmissible"]
       , in_infos     = inadmissInfo
       , in_makeInput = simpleCommand $ Judge Inadmissible
       }      
     , In 
       { in_names     = ["u", "unknown"]
       , in_infos     = uknownInfo
       , in_makeInput = simpleCommand $ Judge Unknown
       }
     , In 
       { in_names     = ["d", "defer"]
       , in_infos     = postponeInfo
       , in_makeInput = simpleCommand $ Cmd Postpone
       }
     , In 
       { in_names     = ["q", "quit"]
       , in_infos     = quitInfo
       , in_makeInput = simpleCommand $ Cmd Quit
       }
     ,
       In 
       { in_names     = ["k", "kids"] 
       , in_infos     = kidsInfo 
       , in_makeInput = simpleCommand $ Cmd ShowKids
       }
     ,
       In 
       { in_names     = ["p", "parents"] 
       , in_infos     = parentsInfo 
       , in_makeInput = simpleCommand $ Cmd ShowParents
       }
     , In 
       { in_names     = ["set"]
       , in_infos     = setInfo
       , in_makeInput = getSet
       }
     , In 
       { in_names     = ["h", "?", "help"]
       , in_infos     = helpInfo
       , in_makeInput = getHelp 
       }
     , In 
       { in_names     = ["r", "refresh"]
       , in_infos     = refreshInfo
       , in_makeInput = simpleCommand $ Cmd Refresh
       }
     , In 
       { in_names     = ["advice"]
       , in_infos     = adviceInfo
       , in_makeInput = simpleCommand $ Cmd Advice
       }
     , In 
       { in_names     = ["dump"]
       , in_infos     = dumpInfo
       , in_makeInput = getDump 
       }
     , In 
       { in_names = ["info"]
       , in_infos = infoInfo
       , in_makeInput = simpleCommand $ Cmd Info
       }
     , In
       { in_names = ["draw"]
       , in_infos = drawInfo
       , in_makeInput = getDraw
       }
     , In
       { in_names = ["forget"]
       , in_infos = forgetInfo
       , in_makeInput = simpleCommand $ Cmd Forget 
       }
     , In
       { in_names = ["size"]
       , in_infos = sizeInfo
       , in_makeInput = simpleCommand $ Cmd Size 
       }
     , In
       { in_names = ["depth"]
       , in_infos = depthInfo
       , in_makeInput = simpleCommand $ Cmd Depth 
       }
     , In
       { in_names = ["restart"]
       , in_infos = restartInfo
       , in_makeInput = simpleCommand $ Cmd Restart 
       }
     , In
       { in_names = ["observe"]
       , in_infos = observeInfo
       , in_makeInput = getObserve 
       }  
     , In
       { in_names = ["j", "jump"]
       , in_infos = jumpInfo
       , in_makeInput = getJump
       }  
     , In
       { in_names     = ["b", "back"]
       , in_infos     = backInfo 
       , in_makeInput = simpleCommand $ Cmd GoBack 
       }
     ]

getHelp :: [String] -> IO (Maybe Input)
getHelp strings = return $ Just $ Cmd (Help strings) 

getObserve :: [String] -> IO (Maybe Input)
getObserve [s] = return $ Just $ Cmd (Observe s)
getObserve [] 
   = do tabIO $ alertIOLn "too few arguments to observe"
        tabIO $ alertIOLn "you must specify a function name"
        return Nothing 
getObserve other
   = do tabIO $ alertIOLn "too many arguments to observe"
        tabIO $ alertIOLn "you must specify a function name"
        return Nothing 

getSet :: [String] -> IO (Maybe Input)
getSet strings = return $ Just $ Cmd (Set strings)

getDump :: [String] -> IO (Maybe Input) 
getDump [s] = return $ Just $ Cmd $ Dump s
getDump [] 
   = do tabIO $ alertIOLn "too few arguments to dump"
        tabIO $ alertIOLn "only these things can be dumped: calls, funs or jumpStack"
        return Nothing 
getDump other
   = do tabIO $ alertIOLn "too many arguments to dump"
        tabIO $ alertIOLn "only these things can be dumped: calls, funs or jumpStack"
        return Nothing 

getJump :: [String] -> IO (Maybe Input)
getJump [index]
   = case safeReadNat index (Just 1) Nothing of
        Nothing -> do tabIO $ alertIOLn "the argument to jump must be >= 1"
                      return Nothing
        Just i -> return $ Just $ Cmd $ Jump i
getJump other
   = do tabIO $ alertIOLn "jump needs a single positive integer argument"
        return Nothing 

getDraw :: [String] -> IO (Maybe Input)
getDraw ("edt":args)
   = do dArgs <- getDrawArgs args 
        case dArgs of
           Nothing -> return Nothing
           Just (file, depth)
              -> return $ Just $ Cmd $ Draw 
                                         { whatToDraw = Tree
                                         , drawFile   = file
                                         , drawDepth = depth 
                                         } 

getDraw ("arg":pos:args)
   = case safeReadNat pos (Just 1) Nothing of
        Nothing 
           -> do tabIO $ alertIOLn "the argument position must be >= 1"
                 return Nothing
        Just i 
           -> do dArgs <- getDrawArgs args
                 case dArgs of
                    Nothing -> return Nothing
                    Just (file, depth)
                       -> return $ Just $ Cmd $ Draw
                                                    { whatToDraw = Arg i 
                                                    , drawFile   = file 
                                                    , drawDepth  = depth 
                                                    }
getDraw ("result":args)
   = do dArgs <- getDrawArgs args
        case dArgs of
           Nothing -> return Nothing
           Just (file, depth)
              -> return $ Just $ Cmd $ Draw
                                           { whatToDraw = Result 
                                           , drawFile   = file 
                                           , drawDepth  = depth 
                                           }
getDraw [] 
   = do tabIO $ alertIOLn "insufficient arguments for draw"
        tabIO $ alertIOLn "only these things can be drawn: edt, arg n, result"
        return Nothing

getDraw other
   = do tabIO $ alertIOLn "only these things can be drawn: edt, arg n, or result"
        return Nothing

getDrawArgs :: [String] -> IO (Maybe (FilePath, Int))
getDrawArgs args
   = getDrawArgs' args ([], -1)
   where
   getDrawArgs' :: [String] -> (FilePath, Int) -> IO (Maybe (FilePath, Int))
   getDrawArgs' [] pathDepth = return $ Just pathDepth 
   getDrawArgs' ("-f":fileName:rest) (_,depth) 
      = getDrawArgs' rest (fileName, depth) 
   getDrawArgs' ("-d":depth:rest) (file,_) 
     = case safeReadNat depth (Just 0) Nothing of
           Nothing -> do tabIO $ alertIOLn $ "invalid draw depth (" ++ depth ++ ")"
                         tabIO $ alertIOLn "draw depth must be an integer >= zero"
                         return Nothing 
           Just d  -> getDrawArgs' rest (file, d) 
   getDrawArgs' other cmd
      = do tabIO $ alertIOLn $ "bad argument(s) for the draw command: " ++ unwords other
           tabIO $ alertIOLn "valid arguments are \"-d depth\" and \"-f filepath\""
           return Nothing

observe :: String -> IO ()
observe name
   = do records <- liftM elems $ readGlobalState state_recordArray 
        let edts = observeCalls (packString name) records
        unless (null edts) $
           do newline
              putStrLn $ "Observation of " ++ name ++ ":"
              newline
              truncate <- readGlobalStateConfig config_truncate
              showListOfNodes truncate edts 
              newline

observeCalls :: PackedString -> [Record] -> [EDT] 
observeCalls seek [] = []
observeCalls seek (record : recs)
   | nameMatch seek record = index record : observeCalls seek recs
   | otherwise = observeCalls seek recs 
   where
   nameMatch :: PackedString -> Record -> Bool
   nameMatch seek (Rec _parent _index name _args _result _line _modName) 
      = seek == name
   nameMatch seek (Constant _index name _result _line _modName)
      = seek == name
   nameMatch seek other = False
   index :: Record -> Int
   index (Rec _parent i _name _args _result _line _modName) = i 
   index (Constant i _name _result _line _modName) = i
