{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Config 

        Description:            read configuration files 

        Primary Authors:        Bernie Pope

        Notes:                  Don't use anything in this module that reads
                                from the global state because it is not
                                initialised yet. (Pure functional programmers
                                please look the other way.)

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Config 
   ( config ) where

import IORef
   ( readIORef ) 

import Data 
   ( Config (..) 
   , dataDirectory
   )

import Control.Exception as E
   ( try )

import Defaults

import Text.ParserCombinators.Parsec 
   hiding (token,eof)

import qualified Text.ParserCombinators.Parsec as P
   (token)

import Text.ParserCombinators.Parsec.Pos
   ( newPos )

import Char
   ( isAlpha
   , isDigit
   , isPrint
   )

import Platform
   ( directorySep 
   , home
   )

import System.Environment
   ( getEnv )

import AnsiColour
   ( Colour (..) )

--------------------------------------------------------------------------------

defaultConfig :: Config
defaultConfig 
    = Config
     { config_screenWidth     = defaultScreenWidth     
     , config_rememberAnswers = True 
     , config_showCycles      = False 
     , config_trace           = False 
     , config_drawFile        = defaultDrawFile
     , config_drawDepth       = defaultDrawDepth
     , config_truncate        = defaultTruncate 
     , config_prompt          = defaultPrompt 
     , config_colours         = False -- conservative, should be overridden in config file
     , config_promptColour    = defaultPromptColour 
     , config_nameColour      = defaultNameColour 
     , config_decorColour     = defaultDecorColour 
     , config_valueColour     = defaultValueColour 
     , config_alertColour     = defaultAlertColour 
     }

{- 1. read global config
   2. read local config
   settings in 2 override those in 1
-}

config :: IO Config
config 
   = do configGlobal <- configRead defaultConfig globalConfigFile 
        configRead configGlobal localConfigFile 

configRead :: Config -> IO (Maybe (String, String)) -> IO Config
configRead configIn reader
   = do tryReadConfig <- reader
        case tryReadConfig of
           Nothing -> return configIn
           Just (name, contents)
              -> parseConfig configIn name contents 

configFileName :: String
configFileName = "buddha.conf"

localConfigFile :: IO (Maybe (String, String))
localConfigFile 
   = do tryHomeDir <- E.try $ getEnv home
        case tryHomeDir of
           Left e -> return Nothing
           Right homeDir 
              -> do let file = homeDir ++ directorySep ++ "." ++ configFileName
                    tryContents <- E.try $ readFile file
                    case tryContents of
                       Left e -> return Nothing
                       Right contents -> return $ Just (file, contents)

globalConfigFile :: IO (Maybe (String, String))
globalConfigFile 
   = do dataDir <- readIORef dataDirectory 
        let file = dataDir ++ directorySep ++ configFileName
        tryContents <- E.try $ readFile file 
        case tryContents of
           Left e -> return Nothing
           Right contents -> return $ Just (file, contents)

--------------------------------------------------------------------------------
-- parsing the config file

parseConfig :: Config -> String -> String -> IO Config 
parseConfig initConfig filename input
   = case runParser topParser initConfig filename (lexer filename input) of
        Left err 
           -> do putStrLn $ "There were errors when reading the configuration file:" 
                 print err 
                 return initConfig 
        Right finalConfig 
           -> return finalConfig
   where
   topParser 
      = do file 
           finalConfig <- getState
           return finalConfig

type Token = (SourcePos, Tok)

data Tok
   = Word String
   | Num Int
   | Equals
   | Newline
   | Eof
   | Comment
   | Junk Char

instance Show Tok where
   show (Word s) = "term: " ++ s
   show (Num n)  = "number: " ++ show n
   show Equals   = "equals sign"
   show Newline  = "newline"
   show Eof      = "end of file"
   show Comment  = "comment"
   show (Junk c) = "character: " ++ show c

lexer :: String -> String -> [Token]
lexer filename input = lexWork (newPos filename 1 1) input
lexWork pos [] = [(pos, Eof)]
lexWork pos (x:xs)
   | x == '='  = (pos, Equals)  : lexWork (nextCol pos) xs
   | x == '\n' = (pos, Newline) : lexWork (nextLine pos) xs
   -- don't need faithful pos info inside a comment
   | x == '#'  = (pos, Comment) : lexWork pos (dropWhile (/= '\n') xs)
   | isDigit x = let (num, rest) = span isDigit (x:xs)
                     nextPos = incSourceColumn pos (length num)
                 in (pos, Num $ read num) : lexWork nextPos rest                   
   -- require words to start with an alphabetic
   -- in their body they can have other funky characters but
   -- no spaces
   | isAlpha x 
        = let (restWord, rest) = span isWordChar xs
              word = x:restWord
              nextPos = incSourceColumn pos (length word) 
          in (pos, Word word) : lexWork nextPos rest
   | isWhiteSpace x = lexWork (nextCol pos) xs
   | otherwise = (pos, Junk x) : lexWork (nextCol pos) xs

isWhiteSpace :: Char -> Bool
isWhiteSpace c = c `elem` " \t\r\f\v\xA0"

isWordChar :: Char -> Bool
isWordChar c = isPrint c && not (isWhiteSpace c)

nextLine :: SourcePos -> SourcePos
nextLine pos = incSourceLine (setSourceColumn pos 1) 1

nextCol :: SourcePos -> SourcePos
nextCol pos = incSourceColumn pos 1

type ConfigParser a = GenParser Token Config a

token :: (Tok -> Maybe a) -> ConfigParser a
token test
   = P.token showToken posToken testToken
   where
   showToken (pos,tok) = show tok
   posToken  (pos,tok) = pos 
   testToken (pos,tok) = test tok 

file :: ConfigParser ()
file = do many line 
          eof

line :: ConfigParser ()
line = setting <|> comment <|> eol

setting :: ConfigParser ()
setting = width        <|> 
          remember     <|> 
          cycles       <|>
          trace        <|>
          trunc        <|>
          drawFile     <|>
          drawDepth    <|>
          prompt       <|>
          colours      <|>
          promptColour <|>
          nameColour   <|>
          decorColour  <|> 
          valueColour  <|>
          alertColour 

set :: String -> ConfigParser a -> (a -> Config -> Config) -> ConfigParser ()
set item valueParser updater
   = do word item
        equals
        value <- valueParser
        updateState $ updater value

width :: ConfigParser ()
width = set "width" num $ \v s -> s {config_screenWidth = v}

remember :: ConfigParser ()
remember = set "remember" bool $ \v s -> s {config_rememberAnswers = v}

cycles :: ConfigParser ()
cycles =  set "cycles" bool $ \v s -> s {config_showCycles = v}

trace :: ConfigParser ()
trace =  set "trace" bool $ \v s -> s {config_trace = v}

trunc :: ConfigParser ()
trunc = set "truncate" num $ \v s -> s {config_truncate = v}

drawFile :: ConfigParser ()
drawFile = set "drawFile" anyWord $ \v s -> s {config_drawFile = v}

drawDepth :: ConfigParser ()
drawDepth = set "drawDepth" num $ \v s -> s {config_drawDepth = v}

prompt :: ConfigParser ()
prompt = set "prompt" anyWord $ \v s -> s {config_prompt = v}

colours :: ConfigParser ()
colours = set "colours" bool $ \v s -> s {config_colours = v}

promptColour :: ConfigParser ()
promptColour = set "promptColour" colour $ \v s -> s {config_promptColour = v}

nameColour :: ConfigParser ()
nameColour = set "nameColour" colour $ \v s -> s {config_nameColour = v}

decorColour :: ConfigParser ()
decorColour = set "decorColour" colour $ \v s -> s {config_decorColour = v}

valueColour :: ConfigParser ()
valueColour = set "valueColour" colour (\v s -> s {config_valueColour = v})

alertColour :: ConfigParser ()
alertColour = set "alertColour" colour $ \v s -> s {config_alertColour = v}
           
word :: String -> ConfigParser String 
word str
   = token $ 
        \tok -> case tok of
                   Word w -> if w == str then Just str else Nothing
                   other  -> Nothing 

anyWord :: ConfigParser String
anyWord
   = token $ 
        \tok -> case tok of
                   Word w -> Just w 
                   other  -> Nothing 

bool :: ConfigParser Bool
bool 
   = getBool "True"  True  <|>
     getBool "False" False <|>
     getBool "true"  True  <|>
     getBool "false" False 
   where
   getBool :: String -> Bool -> ConfigParser Bool
   getBool item constructor
      = do word item
           return constructor  

num :: ConfigParser Int 
num
   = token $ 
        \tok -> case tok of
                   Num n -> Just n
                   other -> Nothing 

eol :: ConfigParser ()
eol = token $
        \tok -> case tok of
                   Newline -> Just ()
                   other   -> Nothing

equals :: ConfigParser ()
equals 
   = token $
        \tok -> case tok of
                   Equals -> Just ()
                   other  -> Nothing

comment :: ConfigParser ()
comment
   = token $
        \tok -> case tok of
                   Comment -> Just ()
                   other  -> Nothing

eof :: ConfigParser ()
eof = token $
        \tok -> case tok of
                   Eof -> Just ()
                   other   -> Nothing

colour :: ConfigParser Colour 
colour 
   = col "black"   Black   <|>
     col "red"     Red     <|>
     col "green"   Green   <|>
     col "yellow"  Yellow  <|>
     col "blue"    Blue    <|>
     col "magenta" Magenta <|>
     col "cyan"    Cyan    <|>
     col "white"   White   <|>
     col "default" Default
   where
   col :: String -> Colour -> ConfigParser Colour
   col name constructor
      = do word name
           return constructor
