{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 DD 

        Description:            Declarative debugging algorithm and support
                                code.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module DD 
   ( dd 
   , dataDirectory
   ) 
   where

import IO 
   ( stderr
   , hPutStr
   )

import Control.Exception     
   (  Exception (..) )

import EDT 
   ( children
   , mkRecordArray
   , mkNodeMap
   , rootNodes
   , validEDT
   )

import Oracle 
   ( askOracle )
 
import IOUtils  
   ( newline
   , dashline 
   , tabIO
   )

import Data 
   ( DebugState (..)
   , Config (..)
   , Command (..)
   , Judgement (..)
   , Diagnosis (..)
   , emptyJudgeDatabase
   , FunTable
   , EDT
   , initGlobalState
   , readGlobalState
   , readGlobalStateConfig
   , updateGlobalState
   , dataDirectory
   )

import CLI 
   ( displayDerivation )

import List 
   ( nub )

import Terminal
   ( initTerminal
   , endTerminal
   )

import Defaults
   ( defaultScreenWidth )

import Data.IORef
   ( readIORef )

import Buddha 
   ( FunRecord 
   , funTable
   , callTable
   , FunRecord (FunRec)
   )

import Data.FiniteMap
   ( emptyFM
   , addToFM_C
   )

import ColourString
   ( alertIOLn )

import Config
   ( config )

--------------------------------------------------------------------------------


debug :: Bool -> Diagnosis -> [EDT] -> IO Diagnosis
debug _showNode diagnosis [] = return diagnosis
debug showNode diagnosis edts@(node:siblings) 
   = do reponse <- askOracle showNode node
        processResponse reponse
   where
   processResponse :: Either Judgement Command -> IO Diagnosis
   processResponse (Left judgement)
      = case judgement of
           Correct      -> debug True diagnosis siblings
           Inadmissible -> debug True diagnosis siblings
           Unknown      -> debug True (uncertain diagnosis node) siblings
           Erroneous    -> do kids <- children node
                              debug True (Wrong node) kids
   processResponse (Right command)
      = case command of
           Postpone -> do if null siblings 
                             then do tabIO $ alertIOLn $ "you can't defer this node"
                                     debug False diagnosis edts
                             else debug True diagnosis (siblings ++ [node])
           Quit     -> return Break
           Jump index 
              -- don't jump to the node that you are already at 
              | index == node 
                   -> do tabIO $ alertIOLn $ "you are already at that node"
                         debug False diagnosis edts 
              | otherwise
                   -> do validIndex <- validEDT index 
                         if validIndex
                            then do pushJump diagnosis edts
                                    debug True NoBugs [index]
                            else do tabIO $ alertIOLn $ 
                                            "you requested to jump to an invalid node number"
                                    debug False diagnosis edts
           GoBack 
              -> do tryJump <- popJump
                    case tryJump of
                       Nothing 
                          -> do tabIO $ alertIOLn $ "you can't go back any further"
                                debug False diagnosis edts
                       Just (oldDiagnosis, oldEdts) 
                          -> debug True oldDiagnosis oldEdts 
           -- restart debugging from the top of the tree again
           -- forget judgements and set the jump stack to empty
           Restart
              -> do nodeMap <- readGlobalState state_nodeMap
                    updateGlobalState (\s -> s {state_jumpStack = []})
                    updateGlobalState (\s -> s {state_judgeDatabase = emptyJudgeDatabase})
                    debug True NoBugs (rootNodes nodeMap)

-- when a bug diagnosis depends on one or more uncertain children
uncertain :: Diagnosis -> EDT -> Diagnosis
uncertain (Wrong parent) node 
   = MaybeWrong parent [node]
uncertain (MaybeWrong parent children) node
   = MaybeWrong parent (node:children)
uncertain other node = other

pushJump :: Diagnosis -> [EDT] -> IO ()
pushJump diagnosis edts
   = do oldStack <- readGlobalState state_jumpStack
        updateGlobalState (updater oldStack)
   where
   updater :: [(Diagnosis, [EDT])] -> DebugState -> DebugState
   updater oldStack oldState
      = oldState { state_jumpStack = (diagnosis, edts) : oldStack }

popJump :: IO (Maybe (Diagnosis, [EDT]))
   = do oldStack <- readGlobalState state_jumpStack
        case oldStack of
           [] -> return Nothing
           (top:rest) -> do updateGlobalState (\s -> s {state_jumpStack = rest}) 
                            return $ Just top

-- this is printed when debugging starts
welcomeMsg :: IO () 
welcomeMsg 
   = do let width = defaultScreenWidth 
        newline 
        newline 
        dashline width 
        newline 
        newline 
        putStrLn $ "Welcome to buddha, version " ++ VERSION
        putStrLn "A declarative debugger for Haskell"
        putStrLn "Copyright (C) 2004, Bernie Pope" 
        putStrLn "http://www.cs.mu.oz.au/~bjpop/buddha" 
        newline 
        putStrLn "Type h for help, q to quit" 
        newline 
        dashline width
        newline

-- first thing to do is initialise the terminal
-- BUDDHA_TERMINAL_FILE is set at configuration time 
-- it should be something equivalent to /dev/tty
dd :: IO ()
dd = do termOk <- Terminal.initTerminal BUDDHA_TERMINAL_FILE 
        if termOk 
           then do welcomeMsg 
                   ddInit
                   Terminal.endTerminal
           else hPutStr stderr "failed to open terminal"      

ddInit :: IO ()
ddInit 
   = do dataDir <- readIORef dataDirectory
        functionRecords <- readIORef funTable
        callRecords <- readIORef callTable
        let reverseCalls = reverse callRecords
        let numCalls = length reverseCalls
        recordArray <- mkRecordArray reverseCalls
        let nodeMap = mkNodeMap reverseCalls
        configuration <- config
        let state = DebugState
                          { state_wasException    = False
                          , state_judgeDatabase   = emptyJudgeDatabase
                          , state_dataDir         = dataDir
                          , state_funTable        = mkFunTable emptyFM functionRecords
                          , state_recordArray     = recordArray
                          , state_nodeMap         = nodeMap
                          , state_jumpStack       = []
                          , state_config          = configuration
                          }        
        initGlobalState state
        ddTop

-- start debugging
ddTop :: IO ()
ddTop 
   = do nodeMap <- readGlobalState state_nodeMap 
        updateGlobalState (\s -> s {state_judgeDatabase = emptyJudgeDatabase})
        ddLoop NoBugs $ rootNodes nodeMap 

ddLoop :: Diagnosis -> [EDT] -> IO ()
ddLoop initDiagnosis []
   = putStrLn "[Empty EDT]"
ddLoop initDiagnosis edts@(_:_)
   = do diagnosis <- debug True initDiagnosis edts
        width <- readGlobalStateConfig config_screenWidth 
        case diagnosis of
           NoBugs -> ddEnd
           -- user asked to quit
           Break -> putStrLn "[Leaving buddha]"
           Wrong node 
              -> do newline 
                    putStrLn $ "Found a bug:"
                    displayDerivation width node 
                    newline
           MaybeWrong node unknowns
              -> do newline 
                    putStrLn $ "Possible bug:" 
                    displayDerivation width node  
                    newline 
                    putStrLn $ "however these were unknown:"
                    mapM_ (displayDerivation width) (nub unknowns)
                    newline

ddEnd :: IO ()
ddEnd
   = do maybeJump <- popJump 
        -- check if the jump stack is empty
        -- if not pop and debug 
        case maybeJump of
           Nothing -> do tabIO $ alertIOLn $ "No bugs found"
                         tabIO $ alertIOLn $ "Going back to the start"
                         ddTop 
           Just (oldDiagnosis, oldEdts) -> ddLoop oldDiagnosis oldEdts

-- XXX: this is fragile, it depends on GHC's representation of
-- exception values, we use this because GHC puts some file location
-- information in the some exceptions which refer to the transformed program
prettyException :: Exception -> String
prettyException (PatternMatchFail _s) = "Pattern match failure"
prettyException (RecConError _s)      = "Record constructtion error"
prettyException (RecSelError _s)      = "Record selection error"
prettyException (RecUpdError _s)      = "Record update error" 
prettyException (NoMethodError _s)    = "No method error" 
prettyException other = show other

-- construct a searchable version of the function table
-- this is needed when we come to print the function out
mkFunTable :: FunTable -> [FunRecord] -> FunTable
mkFunTable table [] = table 
mkFunTable table (FunRec ident arg result : rest)
   = mkFunTable newTable rest
   where
   newTable = addToFM_C combiner table ident [(arg, result)]
   combiner :: [a] -> [a] -> [a]
   combiner old [new] = new:old
