{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Data 

        Description:            Data types used throughout the debugger.
                                Mostly hoisted here to avoid mutual dependencies
                                in modules.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{- 
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Data 
   ( EDT
   , DebugState (..)
   , Config (..)
   , Command (..)
   , Judgement (..)
   , JudgeDatabase 
   , FunTable
   , Response 
   , Diagnosis (..)
   , emptyJudgeDatabase
   , databaseSize
   , Derivation (..)
   , InputSpec (..)
   , Input (..)
   , InputInfo (..)
   , Setting (..)
   , WhatToDraw (..)
   , Record (..)
   , FunRecord (..)
   , NodeMap
   , RecordArray
   , isUnevaluatedVal
   , globalState
   , initGlobalState
   , updateGlobalState
   , updateGlobalStateConfig
   , readGlobalState
   , readGlobalStateConfig
   , getGlobalState
   , dataDirectory
   ) where

import Data.FiniteMap

import Data.PackedString
   ( PackedString )

import Graph
   ( Graph )

import Meta 
   ( Val (..) )

import Data.Array

import IsException
   ( isException )

import IsThunk
   ( isThunk )

import AnsiColour
   ( Colour )

import IORef
   ( IORef
   , newIORef
   , readIORef
   , writeIORef
   , modifyIORef
   )

import System.IO.Unsafe
   ( unsafePerformIO )

--------------------------------------------------------------------------------

data Diagnosis
   = Wrong EDT                -- a definite bug
   | MaybeWrong EDT [EDT]     -- possibly a bug, but relying on a don't know answer
   | Break                    -- quit without coming to a diagnosis
   | NoBugs                   -- no bugs in a (sub) tree
   deriving (Eq, Show)

data Setting 
   = Width Int
   | Remember Bool
   | ClearMemory
   | Trace Bool
   | Cycle Bool
   | ShowSetting
   | Truncate Int
   | DrawFile FilePath
   | DrawDepth Int
   | Prompt String 
   | Colour Bool
   | PromptColour Colour
   | NameColour Colour
   | DecorColour Colour
   | ValueColour Colour
   | AlertColour Colour
   deriving (Eq, Show)

data WhatToDraw 
   = Arg Int 
   | Result
   | Tree
   deriving (Eq, Show)

data Command 
   = Set  [String] 
   | Help [String]
   | Refresh 
   | Advice
   | Dump String
   | ShowKids
   | ShowParents
   | Info
   | Draw
        { whatToDraw :: WhatToDraw 
        , drawFile   :: FilePath
        , drawDepth  :: Int 
        }
   | Quit 
   | Postpone
   | Forget
   | Size
   | Depth
   | Restart
   | Observe String
   | Jump Int
   | GoBack
   deriving (Eq, Show) 

data Input 
   = Cmd Command 
   | Judge Judgement 

data InputSpec
   = In 
     { in_names     :: [String]
     , in_infos     :: InputInfo 
     , in_makeInput :: [String] -> IO (Maybe Input)
     }

data InputInfo 
   = InInfo
     { shortInfo :: String
     , longInfo  :: [String]
     }

data DebugState
   = DebugState
     { state_wasException    :: Bool      -- was an exception thrown?
     , state_judgeDatabase   :: JudgeDatabase  -- previous judgements 
     , state_dataDir         :: FilePath  -- where to find the data files (config dependent)
     , state_funTable        :: FunTable  -- a record of all the function applications
     , state_recordArray     :: RecordArray -- array of function calls and constants
     , state_nodeMap         :: NodeMap              -- parent-child relationships 
     , state_jumpStack       :: [(Diagnosis, [EDT])] -- for backtracking over browsing jumps
     , state_config          :: Config
     }

data Config
   = Config
     { config_screenWidth     :: Int       -- with of the display
     , config_rememberAnswers :: Bool      -- should buddha remember the user's answers? 
     , config_showCycles      :: Bool      -- show cycles in graphs properly?
     , config_trace           :: Bool      -- trace the steps of the oracle? 
     , config_drawFile        :: FilePath  -- the default filepath to draw to
     , config_drawDepth       :: Int       -- the default depth to draw to
     , config_truncate        :: Int       -- maximum printed length of a value
     , config_prompt          :: String    -- the command line prompt
     , config_colours         :: Bool      -- display colours?
     , config_promptColour    :: Colour    -- colour of the prompt
     , config_nameColour      :: Colour    -- colour of the function name
     , config_decorColour     :: Colour    -- decoration colour
     , config_valueColour     :: Colour    -- args and results colour
     , config_alertColour     :: Colour    -- colour for warnings etc 
     }
     deriving Show

{-# NOINLINE globalState #-}
globalState :: IORef DebugState
globalState
   = unsafePerformIO $ newIORef undefined

initGlobalState :: DebugState -> IO ()
initGlobalState state
   = writeIORef globalState state

updateGlobalState :: (DebugState -> DebugState) -> IO ()
updateGlobalState f
   = modifyIORef globalState f 

updateGlobalStateConfig :: (Config -> Config) -> IO ()
updateGlobalStateConfig f
   = modifyIORef globalState modifier
   where
   modifier :: DebugState -> DebugState
   modifier oldState
      = oldState { state_config = f oldConfig }
      where
      oldConfig = state_config oldState

readGlobalState :: (DebugState -> a) -> IO a
readGlobalState f
   = do state <- readIORef globalState 
        return $ f state

readGlobalStateConfig :: (Config -> a) -> IO a
readGlobalStateConfig f
   = do state <- readIORef globalState
        return $ f $ state_config state

getGlobalState :: IO DebugState
getGlobalState
   = readIORef globalState

type EDT         = Int
type RecordArray = Array Int Record
type NodeMap     = FiniteMap Int [Int]

-- a record of a function call
data Record 
   = Rec
        Int          -- parent id
        Int          -- child id
        PackedString -- the name of this function XXX maybe a packed string?
        [Val]        -- the arguments of this function 
        Val          -- the result of this function
        Int          -- line number
        PackedString -- module
   | Constant
        Int          -- id
        PackedString -- name of the constant
        Val          -- the value of the constant
        Int          -- line number
        PackedString -- module
   | Ref
        Int          -- the parent
        Int          -- the ide of the constant
   deriving Show

-- a record of the application of a higher order function 
-- functions are curried so we only need to record one arg
data FunRecord
   = FunRec
         Int   -- the unique identity of this function
         Val   -- its argument 
         Val   -- its result

data Derivation
   = Derivation
     { deriv_name   :: PackedString -- name of the function
     , deriv_args   :: [Graph]      -- arguments
     , deriv_result :: Graph        -- result
     , deriv_sloc   :: Int          -- possible src loc (not available if exception thrown)
     , deriv_module :: PackedString -- src module
     }
   deriving (Eq, Show)

type JudgeDatabase = FiniteMap PackedString [(Derivation, Judgement)]

databaseSize :: JudgeDatabase -> Int
databaseSize db = sum [length entries | (_rep, entries) <- fmToList db]

emptyJudgeDatabase :: JudgeDatabase
emptyJudgeDatabase = emptyFM

data Judgement
   = Correct 
   | Erroneous 
   | Inadmissible 
   | Unknown 
   deriving (Eq, Show)

type Response = Either Judgement Command

type FunTable = FiniteMap Int [(Val, Val)]

isUnevaluatedVal :: Val -> IO Bool
isUnevaluatedVal (V x)
   = do thunk <- isThunk x
        exception <- isException x
        return (thunk && (not exception))

-- this must be set in Main.hs (in the debugged program)
{-# NOINLINE dataDirectory #-}
dataDirectory :: IORef FilePath
dataDirectory = unsafePerformIO $ newIORef []

