{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 DotGraph 

        Description:            An abstract syntax and pretty printer for
                                graphs in the Dot language. 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{- 
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module DotGraph where

import Text.PrettyPrint hiding (Style)

--------------------------------------------------------------------------------

{-
-- an example graph

digraph test123 {
               a -> b -> c;
               a -> {x y};
               b [shape=box];
               c [label="hello\nworld",color=blue,fontsize=24,
                    fontname="Palatino-Italic",fontcolor=red,style=filled];
               a -> z [label="hi", weight=100];
               x -> z [label="multi-line\nlabel"];
               edge [style=dashed,color=red];
               b -> x;
               {rank=same; b x}
       }
-}

-- below is similar to the graph above, but not exactly the same as
-- we don't implement all the features of the dot language
exampleDotGraph :: DotGraph
exampleDotGraph
   = DotGraph 
        "test123"
        [ NodesEdges ["a", "b", "c"] []  
        , NodesEdges ["a", "x"] []
        , NodesEdges ["a", "y"] []
        , Node "b" [Shape Box]
        , Node "c" [ Label "hello\nworld"
                   , Colour Blue
                   , FontSize 24
                   , FontName "Palatino-Italic"
                   , FontColour Red
                   , Style Filled
                   ]
        , NodesEdges ["a", "z"] [Label "hi", Weight 100]
        , NodesEdges ["x", "z"] [Label "multi-line\nlabel"]
        , EdgeSet [Style Dashed, Colour Red]
        , NodesEdges ["b", "x"] []
        , SubGraph Nothing [ Set (Rank Same)
                           , Node "b" []
                           , Node "x" []
                           ]
        ] 

--------------------------------------------------------------------------------

type Title = String

data DotGraph
   = DotGraph 
        Title 
        [Statement]
   deriving (Eq, Ord, Show)

prettyDotGraph :: DotGraph -> Doc
prettyDotGraph (DotGraph title stmts)
   = text "digraph" <+> text title <+> lbrace $$ 
     (nest 3 (vcat $ map prettyStmt stmts)) $$ 
     rbrace

data Statement
   = Set Setting                        -- rank=same 
   | NodeSet [Setting]                  -- node=val 
   | EdgeSet [Setting]                  -- edge [style=dashed,color=red]
   | Node Node [Setting]                -- c [label="hello\nworld",color=blue,fontsize=24] 
   | NodesEdges [Node] [Setting]        -- a -> z [label="hi", weight=100] 
   | SubGraph (Maybe Title) [Statement] -- {rank=same; b; x} 
   deriving (Eq, Ord, Show)

type Node = String

prettyStmt :: Statement -> Doc
prettyStmt (Set setting)
   = prettySetting setting <> semi
prettyStmt (NodeSet settings)
   = text "node" <+> prettySettings settings <> semi
prettyStmt (EdgeSet settings)
   = text "edge" <+> prettySettings settings <> semi
prettyStmt (Node node settings)
   = text node <+> prettySettings settings <> semi
prettyStmt (NodesEdges nodes settings)
   = hsep (punctuate arrow (map text nodes)) <+> prettySettings settings <> semi 
prettyStmt (SubGraph Nothing stmts)
   = braces $ fsep $ map prettyStmt stmts  
prettyStmt (SubGraph (Just title) stmts)
   = text "subgraph" <+> text title <+> braces (fsep $ map prettyStmt stmts)

arrow :: Doc
arrow = text " ->"

prettySettings :: [Setting] -> Doc
prettySettings [] = empty
prettySettings settings
   = brackets (fsep $ punctuate comma $ map prettySetting settings)
   
data Setting 
   = Label  String
   | Height Int
   | Width Int
   | Weight Int
   | Style  Style
   | Colour Colour
   | Shape  Shape
   | FontColour Colour
   | FillColour Colour
   | FontSize Int
   | FontName String
   | Rank Rank
   deriving (Eq, Ord, Show)

prettySetting :: Setting -> Doc
prettySetting (Label s)           = set "label"     $ text (show s)
prettySetting (Height h)          = set "height"    $ int h
prettySetting (Width h)           = set "width"     $ int h
prettySetting (Weight w)          = set "weight"    $ int w
prettySetting (Style style)       = set "style"     $ prettyStyle style
prettySetting (Colour colour)     = set "color"     $ prettyColour colour 
prettySetting (Shape shape)       = set "shape"     $ prettyShape shape 
prettySetting (FontColour colour) = set "fontcolor" $ prettyColour colour 
prettySetting (FillColour colour) = set "fillcolor" $ prettyColour colour 
prettySetting (FontSize s)        = set "fontsize"  $ int s
prettySetting (FontName name)     = set "fontname"  $ doubleQuotes (text name)
prettySetting (Rank rank)         = set "rank"      $ prettyRank rank 

set :: String -> Doc -> Doc
set s d = text s <> equals <> d

data Colour
   = White 
   | Black 
   | Red 
   | Green 
   | Blue 
   | Yellow 
   | Magenta 
   | Cyan 
   | Burlywood
   deriving (Eq, Ord, Show)

prettyColour :: Colour -> Doc
prettyColour White     = text "white"
prettyColour Black     = text "black"
prettyColour Red       = text "red"
prettyColour Green     = text "green"
prettyColour Blue      = text "blue"
prettyColour Yellow    = text "yellow"
prettyColour Magenta   = text "magenta"
prettyColour Cyan      = text "cyan"
prettyColour Burlywood = text "burlywood"

data Style
   = Filled 
   | Solid 
   | Dashed 
   | Dotted 
   | Bold 
   | Invis 
   deriving (Eq, Ord, Show)

prettyStyle :: Style -> Doc
prettyStyle Filled = text "filled"
prettyStyle Solid  = text "solid"
prettyStyle Dashed = text "dashed"
prettyStyle Dotted = text "dotted"
prettyStyle Bold   = text "bold"
prettyStyle Invis  = text "invis"

data Shape
   = Plaintext
   | Ellipse  
   | Circle  
   | Egg  
   | Triangle  
   | Box  
   | Diamond 
   | Trapezium 
   | Parallelogram 
   | House 
   | Hexagon 
   | Octagon
   deriving (Eq, Ord, Show)

prettyShape :: Shape -> Doc
prettyShape Plaintext     = text "plaintext"
prettyShape Ellipse       = text "ellipse"
prettyShape Circle        = text "circle"
prettyShape Egg           = text "egg"
prettyShape Triangle      = text "triangle"
prettyShape Box           = text "box"
prettyShape Diamond       = text "diamond"
prettyShape Trapezium     = text "trapezium"
prettyShape Parallelogram = text "parallelogram"
prettyShape House         = text "house"
prettyShape Hexagon       = text "hexagon"
prettyShape Octagon       = text "octagon"

data Rank
   = Same
   | Min 
   | Max
   deriving (Eq, Ord, Show)

prettyRank Same = text "same"
prettyRank Min  = text "min"
prettyRank Max  = text "max"
