{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004.

        Module:                 EDTtoDotGraph 

        Description:            Generating Dot graphs from the EDT.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{- 
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module EDTtoDotGraph 
   ( edtToDotGraph )
   where

import EDT

import Data
   ( EDT 
   , isUnevaluatedVal
   )

import DotGraph

import System.IO.Unsafe
   ( unsafePerformIO ) 

import Data.PackedString
   ( unpackPS 
   , PackedString
   )

import Meta 
   ( Val )

import List
   ( nub )

--------------------------------------------------------------------------------

data State = State { state_count :: Int 
                   , state_maxDepth :: Int
                   , state_stmts :: [Statement]
                   }

type ErrorMsg = String

newtype S a = S (State -> Either ErrorMsg (a, State))

instance Monad S where
    return a = S (\state -> Right (a, state))

    S comp >>= fun
        = S (\state ->
                     case comp state of
                        Left s -> Left s
                        Right (result, newState)
                           -> case fun result of
                                 S comp' -> comp' newState)

    fail s = S (\_ -> Left s)

run :: Int -> S a -> [Statement]
run maxDepth (S comp)
   = case comp $ State 
                 { state_maxDepth = maxDepth
                 , state_count = 0
                 , state_stmts = []
                 } of
        Left err -> error err
        Right (result, newState) -> state_stmts newState 

getCount :: S Int 
getCount = S $ \state -> Right (state_count state, state)

getAndIncCount :: S Int
getAndIncCount
   = do count <- getCount
        incCount
        return count

incCount :: S ()
incCount
   = S $ \state -> let oldCount = state_count state 
                       newState = state { state_count = oldCount + 1 }
                   in Right ((), newState) 

addStmt :: Statement -> S ()
addStmt stmt 
   = S $ \state -> let oldStmts = state_stmts state
                       newState = state { state_stmts = stmt : oldStmts }
                   in Right ((), newState)

hitMaxDepth :: Int -> S Bool
hitMaxDepth d
   = do maxDepth <- getMaxDepth
        return (d >= maxDepth) 

getMaxDepth :: S Int
getMaxDepth = S $ \state -> Right (state_maxDepth state, state)

--------------------------------------------------------------------------------

-- remove duplicate children
getChildren :: EDT -> [EDT]
getChildren = nub . unsafePerformIO . children 

getName :: EDT -> PackedString 
getName = unsafePerformIO . name 

getResult :: EDT -> Val 
getResult = unsafePerformIO . result 

edtToDotGraph :: Int -> [EDT] -> DotGraph
edtToDotGraph maxDepth edts
   = DotGraph "EDT" $ run maxDepth $ mapM_ (edtStmtsTop 0) edts

edtStmtsTop :: Int -> EDT -> S ()
edtStmtsTop depth edt
   = do unEval <- unEvaluatedNode edt
        if unEval 
           then return ()
           else do bottomOut <- hitMaxDepth depth
                   if bottomOut 
                      then return ()
                      else do count <- getAndIncCount
                              let label = unpackPS $ getName edt 
                                  node  = "node" ++ show count
                                  stmt1 = Node node [Label label]
                              addStmt stmt1
                              mapM_ (edtStmts node $ depth + 1) (getChildren edt) 

edtStmts :: String -> Int -> EDT -> S ()
edtStmts parent depth edt
   = do unEval <- unEvaluatedNode edt
        if unEval
           then return ()
           else do bottomOut <- hitMaxDepth depth
                   if bottomOut
                      then do count <- getAndIncCount
                              let node  = "node" ++ show count
                                  stmt1 = Node node [Style Invis]
                                  stmt2 = NodesEdges [parent, node] [Style Dashed]
                              addStmt stmt1
                              addStmt stmt2
                      else edtStmtsDeeper parent depth edt

edtStmtsDeeper :: String -> Int -> EDT -> S ()
edtStmtsDeeper parent depth edt
   = do count <- getAndIncCount 
        let label = unpackPS $ getName edt 
            node  = "node" ++ show count
            stmt1 = Node node [Label label]
            stmt2 = NodesEdges [parent, node] []
        addStmt stmt1
        addStmt stmt2
        mapM_ (edtStmts node $ depth + 1) (getChildren edt) 

unEvaluatedNode :: EDT -> S Bool
unEvaluatedNode edt 
   = do return $ unsafePerformIO $ isUnevaluatedResult edt

isUnevaluatedResult :: EDT -> IO Bool
isUnevaluatedResult node
   = do res <- result node
        isUnevaluatedVal res 
