{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Graph 

        Description:            Haskell encoding of heap values. 

                                The Graph type captures cycles, thunks and
                                all data types, but not functions. 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Graph 
   ( Graph (..) 
   , GraphKind (..)
   , graphKind
   , tagToKind
   , graphIsList
   , graphIsTuple
   ) where

import Data.Word
   ( Word )

--------------------------------------------------------------------------------

-- a Word should be as wide as a basic heap object which should be
-- as wide as a pointer on the machine
-- this may be 32 or 64 bits depending on the architecture
type Unique = Word 

type Tag = Int

type NumKids = Int

-- mirrors the type GraphNode in Internals.h
data Graph
   = AppNode Unique String Tag NumKids [Graph]
   | CharNode Unique Char
   | IntNode Unique Int
   | IntegerNode Unique Integer 
   | FloatNode Unique Float
   | DoubleNode Unique Double
   | NullNode
   deriving Show

instance Eq Graph where
   -- ignore the unique number (memory address)
   (AppNode _u1 str1 tag1 num1 gs1) == (AppNode _u2 str2 tag2 num2 gs2)
      = str1 == str2 && num1 == num2 && tag1 == tag2 && gs1 == gs2
   (CharNode _u1 c1) == (CharNode _u2 c2) = c1 == c2
   (IntNode _u1 i1) == (IntNode _u2 i2) = i1 == i2
   (IntegerNode _u1 i1) == (IntegerNode _u2 i2) = i1 == i2
   (FloatNode _u1 f1) == (FloatNode _u2 f2) = f1 == f2
   (DoubleNode _u1 d1) == (DoubleNode _u2 d2) = d1 == d2
   NullNode == NullNode = True
   _ == _ = False

-- predicates to say what kind of thing the graph is

data GraphKind 
   = GNode 
   | GCycle
   | GThunk
   | GChar
   | GInt
   | GInteger
   | GFloat
   | GDouble
   | GException
   | GApUpd
   | GFun
   | GNull
   deriving (Eq, Show)

graphKind :: Graph -> GraphKind
graphKind (AppNode _ _ tag _ _) = tagToKind tag
graphKind (CharNode _ _) = GChar
graphKind (IntNode _ _) = GInt
graphKind (IntegerNode _ _) = GInteger
graphKind (FloatNode _ _) = GFloat
graphKind (DoubleNode _ _) = GDouble
graphKind NullNode = GNull

-- this MUST be in sync with Internals.h
tagToKind :: Tag -> GraphKind
tagToKind 1 = GNode
tagToKind 2 = GCycle
tagToKind 3 = GThunk
tagToKind 4 = GChar
tagToKind 5 = GInt
tagToKind 6 = GInteger -- 6 is a small integer 7 is a large one!
tagToKind 7 = GInteger
tagToKind 8 = GFloat 
tagToKind 9 = GDouble
tagToKind 10 = GException 
tagToKind 11 = GApUpd 
tagToKind 12 = GFun 
tagToKind other = error $ "tagToKind: bad tag number: " ++ show other

-- true if the graph represents a list
graphIsList :: Graph -> Bool
graphIsList (AppNode _unique "[]" _tag _numKids _kids) = True
graphIsList (AppNode _unique ":" _tag _numKids _kids)  = True
graphIsList _other = False

-- true if the graph represents a n-tuple 
-- "(,)", "(,,)", ...
graphIsTuple :: Graph -> Bool
graphIsTuple (AppNode _unique desc _tag _numKids _kids) 
   = isTupleDesc desc 
   where
   isTupleDesc :: String -> Bool
   isTupleDesc [] = False
   isTupleDesc [_] = False
   isTupleDesc [_,_] = False
   isTupleDesc ('(':tail) 
      = isTupleDesc' tail
      where
      isTupleDesc' :: String -> Bool
      isTupleDesc' [] = False
      isTupleDesc' [_] = False
      isTupleDesc' ",)" = True
      isTupleDesc' (',':x:rest) = isTupleDesc' (x:rest)
      isTupleDesc' _other = False
   isTupleDesc _other = False
graphIsTuple _other = False 
