{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 GraphToDotGraph 

        Description:            For generating Dot graphs from Graph 
                                representation of values.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{- 
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module GraphToDotGraph 
   ( graphToDotGraph )
   where

import Graph 

import DotGraph

import Data.Set

import Monad
   ( unless )

import Defaults
   ( defaultGraphLabelWidth )

--------------------------------------------------------------------------------

data State = State { state_count    :: Int 
                   , state_maxDepth :: Int
                   , state_stmts    :: [Statement]
                   , state_nodes    :: Set String 
                   }

type ErrorMsg = String

newtype S a = S (State -> Either ErrorMsg (a, State))

instance Monad S where
    return a = S (\state -> Right (a, state))

    S comp >>= fun
        = S (\state ->
                     case comp state of
                        Left s -> Left s
                        Right (result, newState)
                           -> case fun result of
                                 S comp' -> comp' newState)

    fail s = S (\_ -> Left s)

run :: Int -> State -> S a -> [Statement]
run maxDepth initState (S comp)
   = case comp (initState {state_maxDepth = maxDepth}) of
        Left err -> error err
        Right (result, newState) -> state_stmts newState 

initState = State { state_count    = 0 
                  , state_stmts    = []
                  , state_maxDepth = 0
                  , state_nodes    = emptySet 
                  }

select :: (State -> b) -> S b
select f 
   = S $ \state -> Right (f state, state)

getCount :: S Int 
getCount = select state_count 

getAndIncCount :: S Int
getAndIncCount
   = do count <- getCount
        incCount
        return count

incCount :: S ()
incCount
   = S $ \state -> let oldCount = state_count state 
                       newState = state { state_count = oldCount + 1 }
                   in Right ((), newState) 

addStmt :: Statement -> S ()
addStmt stmt 
   = S $ \state -> let oldStmts = state_stmts state
                       newState = state { state_stmts = stmt : oldStmts }
                   in Right ((), newState)

hitMaxDepth :: Int -> S Bool
hitMaxDepth d
   = do maxDepth <- getMaxDepth
        return (d >= maxDepth) 

getMaxDepth :: S Int
getMaxDepth = select state_maxDepth

-- true if we've seen this node before
nodeSeen :: String -> S Bool 
nodeSeen e
   = do nodes <- getNodes
        return (elementOf e nodes)

getNodes :: S (Set String)
getNodes = select state_nodes 

addNode :: String -> S ()
addNode node
   = S $ \s -> let oldNodes = state_nodes s 
                   newNodes = addToSet oldNodes node
               in Right ((), s { state_nodes = newNodes })

unlessSeenNode :: String -> S () -> S () 
unlessSeenNode node action 
   = do seenBefore <- nodeSeen node
        unless seenBefore action 

--------------------------------------------------------------------------------

identity :: Graph -> Maybe String
identity (CharNode _u c)    
     -- dotty doesn't like literal newlines in labels
   | c == '\n' = Just "<newline>"
   | otherwise = Just $ show c
identity (IntNode _u i)     = Just $ show i
identity (IntegerNode _u i) = Just $ trunc $ show i
identity (FloatNode _u f)   = Just $ show f
identity (DoubleNode _u d)  = Just $ show d
identity NullNode        = Nothing
identity (AppNode _unique descr _tag _numKids _kids) 
   = Just $ trunc descr

trunc :: String -> String
trunc s
   | length s > defaultGraphLabelWidth
        = take defaultGraphLabelWidth s ++ "..." 
   | otherwise = s

unique :: Graph -> S String 
unique (AppNode u _descr _tag _numKids _kids) 
   = return ("node" ++ show u)
unique (CharNode u _c)    = return ("node" ++ show u)
unique (IntNode u _i)     = return ("node" ++ show u)
unique (IntegerNode u _i) = return ("node" ++ show u)
unique (FloatNode u _i)   = return ("node" ++ show u)
unique (DoubleNode u _i)  = return ("node" ++ show u)
unique NullNode 
   = do c <- getAndIncCount
        return ("nullNode" ++ show c)

children :: Graph -> [Graph]
-- XXX the reverse is to make the graph picture have the same
-- children order as the debugger
children (AppNode _unique descr _tag _numKids kids) = reverse kids 
children other = []

graphToDotGraph :: Int -> Graph -> DotGraph
graphToDotGraph maxDepth graph 
   = DotGraph "Value" $ run maxDepth initState $ graphStmtsTop 0 graph 

graphStmtsTop :: Int -> Graph -> S ()
graphStmtsTop depth graph 
   = do bottomOut <- hitMaxDepth depth
        if bottomOut 
           then return ()
           else do case identity graph of
                      Nothing -> return ()
                      Just ident
                         -> do node <- unique graph  
                               let label = ident 
                                   stmt1 = Node node [Label label]
                               addStmt stmt1
                               addNode node
                               mapM_ (graphStmts node $ depth + 1) (children graph) 

graphStmts :: String -> Int -> Graph -> S ()
graphStmts parent depth graph 
   = do bottomOut <- hitMaxDepth depth
        if bottomOut
           then do node <- unique graph
                   let stmt1 = Node node [Style Invis]
                       stmt2 = NodesEdges [parent, node] [Style Dashed]
                   addStmt stmt2
                   addStmt stmt1
           else graphStmtsFurther parent depth graph

graphStmtsFurther :: String -> Int -> Graph -> S ()
graphStmtsFurther parent depth graph
   = case graphKind graph of
        GCycle -> do node <- unique graph
                     let stmt = NodesEdges [parent, node] []
                     addStmt stmt
        GThunk -> do node <- unique graph
                     let stmt1 = Node node [Label "?", Colour Red, Style Filled]
                         stmt2 = NodesEdges [parent, node] []
                     addStmt stmt2
                     addStmt stmt1
        GNode  -> do node <- unique graph 
                     let edgeStmt = NodesEdges [parent, node] []
                     addStmt edgeStmt 
                     unlessSeenNode node $
                        do let ident = apNodeName graph 
                               nodeStmt = Node node [Label ident]
                           addStmt nodeStmt 
                           addNode node
                           mapM_ (graphStmts node $ depth + 1) (children graph) 
        other  -> do let possibleName = identity graph 
                     case possibleName of
                        Nothing -> return ()
                        Just ident 
                           -> do node <- unique graph
                                 let edgeStmt = NodesEdges [parent, node] []
                                 addStmt edgeStmt 
                                 unlessSeenNode node $
                                    do let nodeStmt = Node node [Label ident] 
                                       addStmt nodeStmt 
                                       addNode node

apNodeName :: Graph -> String
apNodeName (AppNode _unique descr _tag _numKids _kids) = descr
