{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004 

        Module:                 GraphToExp

        Description:            Convert a graph into a description.
                                (Descriptions are much nicer for the printing
                                 interface to deal with). 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module GraphToExp
   ( graphToExp ) where

import Meta 
   ( Description (..)
   , Exception (..)
   , FunMap
   , Exp (..)
   , Pat(..)
   , Literal(..)
   , Equation(..)
   , Val (..)
   )

import Graph 
   ( Graph (..)
   , graphKind
   , GraphKind (..)
   , graphIsTuple
   )

import Data
   ( FunTable )

import Data.FiniteMap

import List 
   ( nub )

import Monad

import Data.Word
   ( Word )

import ReifyHs
   ( reify )

import System.IO.Unsafe
   ( unsafePerformIO )

--------------------------------------------------------------------------------

-- checking for cycles

type NameMap = FiniteMap Word String

-- find out where any cycles occur in the graph and name them
cycleNodes :: Graph -> NameMap
cycleNodes graph
   = listToFM $ reverse $ zip (nub $ getCycles graph) nameSupply 
   where
   nameSupply :: [String]
   nameSupply = ["_x" ++ show i | i <- [1..]]
   getCycles :: Graph -> [Word]
   getCycles graph@(AppNode unique _desc _tag _numKids kids)
      = case graphKind graph of
           GCycle -> unique : concatMap getCycles kids
           other -> concatMap getCycles kids
   getCycles other = []

--------------------------------------------------------------------------------

-- a state monad

data State = State { state_nameMap :: NameMap 
                   , state_funTable :: FunTable
                   }

type ErrorMsg = String

newtype S a = S (State -> Either ErrorMsg (a, State))

instance Monad S where
    return a = S (\state -> Right (a, state))

    S comp >>= fun
        = S (\state ->
                     case comp state of
                        Left s -> Left s
                        Right (result, newState)
                           -> case fun result of
                                 S comp' -> comp' newState)

    fail s = S (\_ -> Left s)

run :: State -> S Exp -> (Exp, State)
run initState (S comp)
   = case comp initState of
        Left err -> error err
        Right (result, newState) -> (result, newState)

initState funTable = State { state_nameMap = emptyFM 
                           , state_funTable = funTable
                           }

getNameMap :: S NameMap 
getNameMap = S $ \state -> Right (state_nameMap state, state)

getFunTable :: S FunTable 
getFunTable = S $ \state -> Right (state_funTable state, state)

setNameMap :: NameMap -> S ()
setNameMap nameMap
   = S $ \state -> Right ((), state { state_nameMap = nameMap })

lookupNameMap :: Word -> S (Maybe String)
lookupNameMap i
   = do nameMap <- getNameMap
        return $ lookupFM nameMap i

lookupFunTable :: Int -> S (Maybe [(Val, Val)])
lookupFunTable i
   = do table <- getFunTable
        return $ lookupFM table i
        
graphToExp :: Bool -> FunTable -> Graph -> Exp
graphToExp detectCycles funTable g 
   = fst $ run (initState funTable) $ graphToExpTop detectCycles g

--------------------------------------------------------------------------------

-- convert a Graph into an Exp

graphToExpTop :: Bool -> Graph -> S Exp
graphToExpTop detectCycles graph
   = do when detectCycles $ setNameMap $ cycleNodes graph 
        toExp graph 

toExp :: Graph -> S Exp 
toExp g | isLiteral g    = return $ Elit $ graphToLit g 
toExp g | isEncodedFun g = encodedFunction g 
toExp g@(AppNode unique desc tag numKids kids)
   | graphIsTuple g = graphToTuple g
   | isCompleteString g 
        = return $ Elit $ Ls $ graphToString g
   | otherwise 
        = do case graphKind g of
                GNode -> if numKids <= 0
                            then return $ Ei desc 
                            -- is this the start of a cycle?
                            else do name <- lookupNameMap unique
                                    case name of
                                       -- not the start of a cycle
                                       Nothing 
                                          -> do kExps <- kidsExps kids
                                                return $ Ea $ Ei desc : kExps 
                                       -- yes, the start of a cycle
                                       Just ident -> do kExps <- kidsExps kids 
                                                        let bodyExp = Ea $ Ei desc : kExps
                                                        return $ startCycle ident bodyExp 
                GThunk     -> return $ EDescr $ DThunk unique
                GCycle     -> do name <- lookupNameMap unique
                                 case name of 
                                    Nothing -> return $ EDescr $ DCycle unique
                                    Just ident -> return $ Ei ident 
                GApUpd     -> return $ EDescr $ DException AsyncExcept 
                GException -> do kExps <- kidsExps kids
                                 return $ EDescr $ DException $ SyncExcept $ Ea kExps 
                GFun       -> return Enull 

   where
   kidsExps :: [Graph] -> S [Exp]
   kidsExps = mapM toExp 
toExp NullNode = return Enull 

isEncodedFun :: Graph -> Bool
isEncodedFun (AppNode unique desc tag numKids kids)
   = desc `elem` ["F0", "F1", "F2"]
isEncodedFun other = False

encodedFunction :: Graph -> S Exp 
encodedFunction (AppNode unique desc tag numKids kids) 
   | desc == "F0" 
        = do funMap <- toFunMap $ funIndex kids 
             return $ EDescr $ DFun $ funMap
   | desc == "F2" = return Eio 
   | desc == "F1" = return $ EDescr $ DAbstract "<function>" 
   where
   -- the function encoder has two arguments
   -- the first is a function and the second is the index number
   -- we only want the second of the two
   funIndex :: [Graph] -> Int
   funIndex [_, IntNode _unique i] = i
   funIndex other 
      = error $ "encodedFunction: strange looking encoded function: " ++ show other 


startCycle :: String -> Exp -> Exp 
startCycle name exp 
   = Elet [newEquation] (Ei name) 
   where
   newEquation = Dp (Pi name) exp 

-- true if a graph encodes a literal
isLiteral :: Graph -> Bool
isLiteral g = graphKind g `elem` [GChar, GInt, GInteger, GFloat, GDouble]

-- convert a graph to a literal
graphToLit :: Graph -> Literal
graphToLit (CharNode _u c)    = Lc c
graphToLit (IntNode _u i)     = Li $ fromIntegral i
graphToLit (IntegerNode _u i) = Li i
graphToLit (FloatNode _u f)   = Lf (realToFrac f)
graphToLit (DoubleNode _u d)  = Lf d

-- graphs that represent (fully evaluated) strings
graphToString :: Graph -> String 
graphToString (AppNode _unique "[]" _tag _numKids _kids) = []
graphToString (AppNode _unique ":" _tag _numKids [x,xs])
   = graphToChar x : graphToString xs 
   where
   graphToChar :: Graph -> Char 
   graphToChar (CharNode _u c) = c
-- graphToString other = error $ "graphToString: badly formed string: " ++ show other
graphToString other = "< " ++ show other  ++ " >"

-- true if the graph represents a complete string
isCompleteString :: Graph -> Bool
isCompleteString (AppNode _unique ":" _tag _numKids [x,xs])
   = isChar x && isStringTail xs
   where
   isChar :: Graph -> Bool
   isChar g = graphKind g == GChar
   isStringTail :: Graph -> Bool
   isStringTail (AppNode _unique ":" _tag _numKids [x,xs])
      = isChar x && isStringTail xs
   isStringTail (AppNode _unique "[]" _tag _numKids _kids) 
      = True
   isStringTail other = False
isCompleteString other = False

-- graphs that represent tuples 
graphToTuple :: Graph -> S Exp 
graphToTuple (AppNode _unique _desc _tag _numKids kids@(x:xs))
   = liftM Etu $ tuple kids
   where 
   tuple [] = return [] 
   tuple (x:xs) = liftM2 (:) (toExp x) (tuple xs)

-- a graph is a string if it is a list and its first item is a character
-- this is a conservative test, but should be enough for usual use
isGraphString :: Graph -> Bool
isGraphString (AppNode _unique ":" _tag _numKids [first,_])
   = graphKind first == GChar
isGraphString other = False

toFunMap :: Int -> S FunMap 
toFunMap ident
   = do lookup <- lookupFunTable ident 
        case lookup of
           Nothing -> return []
           Just records
              -> do functionMap <- mapM funArgToExp records
                    return $ functionMap 
   where
   funArgToExp :: (Val,Val) -> S (Exp,Exp) 
   funArgToExp (V v1, V v2)
      = do let graph1 = unsafePerformIO $ reify v1 
               graph2 = unsafePerformIO $ reify v2 
           exp1 <- toExp graph1
           exp2 <- toExp graph2
           return (exp1, exp2)
