/*------------------------------------------------------------------------------

        Copyright:              Bernie Pope

        Module:                 HeapGraph.c 

        Description:            Code for representing values from the ghc heap
                                as a graph.

        Primary Authors:        Bernie Pope

        Notes:                  See the file ../LICENCE for licence information

------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "Internals.h"

/* builds a node with sufficient space for the children,
 * but does not assign the children
 */

GraphNode *makeGraphNode (StgWord u, Tag t, char *desc, int num)
{
   GraphNode *newNode;

   newNode = (GraphNode *) malloc (sizeof (GraphNode));

   if (newNode == NULL)
   {
      fprintf (stderr, "\nerror: makeGraphNode: insufficient memory, malloc failed\n");
      exit (-1);
   }
   else
   {
      newNode->unique = u;
      newNode->tag = t;
      newNode->numChildren = num;

      if (t == Node)
      {
         newNode->val.descriptor = desc; /* we want to share the descriptors, don't strcpy() */
      }

      if (num > 0)
      {
         newNode->children = (GraphNode **) malloc (num * sizeof (GraphNode *));
         if (newNode->children == NULL)
         {
            fprintf (stderr, "\nerror: makeGraphNode: insufficient memory, malloc failed\n");
            exit (-1);
         }
      }
      else
      {
         newNode->children = NULL;
      }
         
      return newNode;
   }
}

void freeGraph (GraphNode *node)
{
   GraphNode *current = node;
   GraphNode *rightChild;
   int numKids;
   int i;

   while (current)
   {
      rightChild = NULL;
      numKids = current->numChildren;
      if (numKids > 0)
      {
         for (i = 0; i < numKids - 1; i++)
         {
            freeGraph (current->children[i]);
         }
         rightChild = current->children[numKids-1];
         free (current->children);
      }

      /* must remember to free the stable pointers used for large integers */
      else if (current->tag == LargeInteger)
      {
         freeStablePtr (current->val.largeIntegerSPtr);
      }

      free (current);

      current = rightChild;
   }
}

List *cons (StgWord x, List *tail)
{
   List *head;

   head = (List *) malloc (sizeof (List));

   if (head == NULL)
   {
      fprintf (stderr, "cons: malloc failed, exiting\n");
      exit (-1);
   }

   head->val = x;
   head->next = tail;

   return head;
} 

List *freeListNode (List *l)
{
   List *tail;

   tail = l->next;
   free (l);

   return tail;
}
