{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Help 

        Description:            Display help information.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Help 
   ( help
   , info
   , kidsInfo 
   , parentsInfo 
   , setInfo
   , helpInfo
   , refreshInfo
   , adviceInfo
   , dumpInfo
   , infoInfo
   , correctInfo
   , erroneousInfo
   , inadmissInfo
   , uknownInfo
   , postponeInfo
   , quitInfo
   , drawInfo
   , forgetInfo
   , sizeInfo
   , depthInfo
   , restartInfo
   , observeInfo
   , jumpInfo
   , backInfo
   ) 
   where

import IO 

import IOUtils

import Data 
   ( InputSpec (..)
   , InputInfo (..)
   )

import List 
   ( intersperse  
   , delete
   )

import Monad
   ( unless )

import ColourString 
   ( decorColour
   , decorIOLn
   , alertIOLn
   )

--------------------------------------------------------------------------------

help :: [InputSpec] -> [String] -> IO ()
help inputSpecs []
   = do newline 
        tabIO $ putStrLn helpMsg
        newline
        let helplines = getShortHelp inputSpecs
        portrayHelp helplines 
        newline
   where
   portrayHelp :: [([String], String)] -> IO ()
   portrayHelp [] = return ()
   portrayHelp ((names, msg):rest)
      = do nameList <- decorColour $ ljustify 15 $ concat $ intersperse "," names
           let line = nameList ++ " -- " ++ msg
           tabIO $ putStrLn line
           portrayHelp rest


help inputSpecs [topic] 
   = do case getLongHelpForTopic topic inputSpecs of
           Nothing 
              -> tabIO $ alertIOLn $ "I can't help you with that (" ++ topic ++ ")"
           Just (lines, synonyms)
              -> do newline 
                    tabIO $ putStrLn $ topic ++ ":"
                    unless (null synonyms) $
                       do newline
                          tabIO $ putStrLn $ 
                                  "synonyms = " ++ (concat $ intersperse ", " synonyms)
                    newline
                    portrayHelp lines 
                    newline
   where
   portrayHelp :: [String] -> IO ()
   portrayHelp [] = return ()
   portrayHelp (line:lines)
      = do tabIO $ putStrLn line
           portrayHelp lines

help inputSpecs other
   = tabIO $ alertIOLn $ "I can't help you with that (" ++ unwords other ++ ")"

getShortHelp :: [InputSpec] -> [([String], String)]
getShortHelp [] = []
getShortHelp (entry:rest)
   = (entryNames, shortHelp) : getShortHelp rest
   where
   entryNames = in_names entry 
   shortHelp  = shortInfo $ in_infos entry

getLongHelpForTopic :: String -> [InputSpec] -> Maybe ([String], [String])
getLongHelpForTopic topic [] = Nothing
getLongHelpForTopic topic (entry:rest)
   | topic `elem` thisEntryNames 
        = Just (longInfo $ in_infos entry, delete topic thisEntryNames)
   | otherwise
        = getLongHelpForTopic topic rest
   where
   thisEntryNames = in_names entry

helpMsg :: String
helpMsg = "help <command> gives more specific information on a command" 

-- show program information
info :: IO ()
info 
   = do newline
        tabIO $ putStrLn "Information about this program:"
        newline
        tabIO $ decorIOLn $ "buddha version " ++ VERSION
        tabIO $ decorIOLn $ "copyright (C) 2004, Bernie Pope"
        tabIO $ decorIOLn $ "licensed under the GNU GPL"
        tabIO $ decorIOLn $ "http://www.cs.mu.oz.au/~bjpop/buddha"
        newline

-- help on specific commands

kidsInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Show the children of the current derivation."
   long  = [ short 
           , "Each derivation corresponds to a node in a tree."
           , "A node may have zero or more children. This command"
           , "will list all the children of the current node."
           , "Each node in the tree is identified by a unique"
           , "number. This number is printed with each child."
           , "You can tell buddha to go directly to one of those"
           , "nodes by issuing the \"jump\" command with the"
           , "number of the desired node as its argument."
           , "If there are no children then obviously you cannot"
           , "go to any of them!"
           , "  "
           , "For example, suppose the current node has 5 children."
           , "A list of those children can be obtained by typing:"
           , "   kids"
           , "which can be abbreviated to:"
           , "   k"
           , "If the third child has an identity of, say, \"12\","
           , "you can go directly to that child by jumping to it:"
           , "   jump 12"
           , "which can be abbreviated to:"
           , "   j 12"
           ]

parentsInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Show the parents of the current derivation."
   long  = [ short
           , "Each derivation corresponds to a node in a tree."
           , "A node may have zero or more parents. Function calls"
           , "always have one parent, except those which form the"
           , "roots of the EDT." 
           , "Constants may have multiple parents. This command"
           , "will list all the parents of the current node."
           , "Each node in the tree is identified by a unique"
           , "number. This number is printed with each parent."
           , "You can tell buddha to go directly to one of those"
           , "nodes by issuing the \"jump\" command with the"
           , "number of the desired node as its argument."
           , "If there are no parents then obviously you cannot"
           , "go to any of them!"
           , "  "
           , "For example, suppose the current node has one parent."
           , "This can be obtained by typing:"
           , "   parents"
           , "which can be abbreviated to:"
           , "   p"
           , "If the parent has an identity of, say, \"12\","
           , "you can go directly to that node by jumping to it:"
           , "   jump 12"
           , "which can be abbreviated to:"
           , "   j 12"
           ]
 
setInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "View or change one of buddha's settings."
   long  = [ short 
           , "To view the currect settings type \"set\" on its own."
           , "To set a particular option type \"set\" followed by"
           , "the option name and its new value."
           , "   For example: set width 90"
           , "   This will set the screen width to 90 columns."
           , "  "
           , "The possible settings are:"
           , "   width         = The screen width, its value is a positive"
           , "                   integer."
           , "   remember      = If true buddha will remember your answers."
           , "                   Otherwise it will forget them."
           , "   trace         = If true buddha will tell you more"
           , "                   information about what it is doing."
           , "   cycles        = If true buddha will show cycles properly"
           , "                   in data structures."
           , "   truncate      = The largest size in characters of the"
           , "                   display of an argument or result in a"
           , "                   derivation. Anything larger than this size"
           , "                   will be truncated. Truncation points are"
           , "                   indicated by three dots \"...\"."
           , "   drawFile      = The name of the file for buddha to write"
           , "                   to when the \"draw\" command is used."
           , "                   Don't put any spaces in the filename."
           , "   drawDepth     = The maximum depth for drawing objects"
           , "                   when the \"draw\" command is used."
           , "                   The depth value should be some integer"
           , "                   greater than zero."
           , "   prompt        = The prompt used at the start of each"
           , "                   command line. It can be any string."
           , "   colours       = Toggle colour output."
           , "   prompt-colour = Colour of the prompt."
           , "   name-colour   = ...function names."
           , "   decor-colour  = ...various decorations."
           , "   value-colour  = ...arguments and result." 
           , "   alert-colour  = ...warning messages."
           , "   "
           , "Here are some more examples:"
           , "   set"
           , "   set width 120"
           , "   set remember true"
           , "   set remember false"
           , "   set cycles true"
           , "   set cycles false"
           , "   set truncate 3000"
           , "   set drawFile /tmp/out.dot"
           , "   set drawDepth 12"
           , "   set prompt input>"
           , "   set name-colour green"
           ]

helpInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Display help on buddha's commands."
   long  = [ short 
           , "If you type \"help\" on its own then you get a short"
           , "list of all the possible commands available with a"
           , "brief description of what they offer."
           , "If you want more detailed information on a particular"
           , "command then type \"help\" followed by the name of that"
           , "command."
           , "   "
           , "For example if you want to know more about the \"set\""
           , "command then type:"
           , "   help set"
           , "which can be abbreviated to:"
           , "   h set"
           , "or alternatively:"
           , "   ? set"
           ]

refreshInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Re-draw the most recent derivation."
   long  = [ short 
           , "Sometimes it is easy to get lost in the command line"
           , "interface. If you want to see the current derivation"
           , "then use this command."
           ]

adviceInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "A snippet of wisdom to waylay your boredom."
   long  = [ short 
           , "Debugging can be a long and tiresome task."
           , "If you find yourself getting bored or discouraged"
           , "then you can ask for some advice from buddha."
           , "It may not help you to find the current bug, but it"
           , "probably can't hurt."
           ]

dumpInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Dump buddha's internal structures. For hackers only."
   long  = [ short 
           , "Warning: this command is really only for"
           , "people hacking buddha. You might get enormous"
           , "amounts of stuff printed on the display - so"
           , "use this command carefully."
           , "   "
           , "Buddha keeps two tables that it uses to build"
           , "the EDT from. The first table is a record of the"
           , "program's call graph. This is displayed when you"
           , "use the command:"
           , "   dump calls"
           , "The second table records information about the"
           , "applications of functions that appear in higher"
           , "order contexts (ie the arguments and results of"
           , "other functions, that are also functions, such as"
           , "the first argument to Prelude.map). This table is"
           , "displayed when you use the command:"
           , "   dump funs"
           , "   "
           , "In addition to those tables you can also view the"
           , "configuration stack used by buddha to allow you"
           , "to go go back to a node after jumping from it:"
           , "   dump jumpStack"
           ]

infoInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short =   "Information about this version of buddha."
   long  = [ short 
           , "This command displays the version number and"
           , "copyright/licencing information about buddha."
           , "It also mentions the website where you can find"
           , "more information about buddha."
           ]

correctInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "The current derivation is correct."
   long  = [ short
           , "If the current derivation satisfies your intentions"
           , "then you should judge it to be correct."
           , "   "
           , "Typing \"correct\" or just \"c\" is effectively a"
           , "judgement that says the current derivation is valid."
           , "   "
           , "In other words, when the function is applied to the"
           , "given arguments it is supposed to produce the result"
           , "that is shown in the derivation."
           ]

erroneousInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "The current derivation is _not_ correct."
   long  = [ short
           , "If the current derivation _does not_ satisfy your"
           , "intentions then you should judge it to be erroneous."
           , "   "
           , "Typing \"erroneous\" or just \"e\" is effectively a"
           , "judgement that says the current derivation is not valid."
           , "   "
           , "In other words, when the function is applied to the"
           , "given arguments it is _not_ supposed to produce the"
           , "result that is shown in the derivation."
           ]

inadmissInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "The function was applied to the wrong arguments."
   long  = [ short
           , "Sometimes, due to a bug in the program, a function will"
           , "be applied to some arguments for which you have no idea"
           , "what the result should be."
           , "   "
           , "You think to yourself: Hey, this function should never"
           , "have been applied to this set of arguments, what's going"
           , "on?"
           , "   "
           , "In such a situation it is unnatural to say that the"
           , "derivation is either correct or erroneous. Therefore buddha"
           , "offers a third kind of judgement, called \"inadmissible\","
           , "(or just \"i\") which should be used in this kind of"
           , "situation."
           , "   "
           , "A good example of inadmissibility is the merge function"
           , "which is supposed to be only given sorted arguments."
           , "There may be a bug in the program which causes merge to"
           , "be applied to one or two unsorted lists. In this case"
           , "you probably have no idea what merge should do ---"
           , "the intended behaviour is undefined in this instance."
           ]

uknownInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "The correctness of the derivation is unknown."
   long  = [ short
           , "If you have no idea whether the current derivation is"
           , "correct, erroneous, or inadmissible then use this"
           , "command. It can be shortened to \"u\""
           ]

postponeInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Avoid the current derivation if possible."
   long  = [ short
           , "If you are shown a derivation that you would rather not"
           , "have to judge at this point in time, you can issue this"
           , "command and buddha will try to look elsewhere for a bug."
           , "   "
           , "However, if all other paths are eventually exhausted and"
           , "no bugs are found, then buddha will return to this"
           , "derivation once more."
           , "   "
           , "Unfortunately, sometimes buddha will have no choice but"
           , "to return you to the derivation immediately. In this"
           , "situation \"postpone\" is not useful at all --- you will"
           , "have to make another kind of judgement, or do some"
           , "browsing."
           ]

quitInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Quit."
   long  = [ short, "This command terminates the debugging session." ]

drawInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Draw the edt, or an argument or result."
   long  = [ short
           , "The \"draw\" command allows you to output pretty pictures"
           , "of either the edt or the arguments or result of a"
           , "derivation. The pictures are actually graphs written"
           , "in the \"dot\" language."
           , "   "
           , "The pictures (graphs) can be viewed using the commonly"
           , "available program \"dotty\". Alternatively they can be"
           , "converted to the postscript language using the program"
           , "(also) called \"dot\"."
           , "   "
           , "The syntax for the \"draw\" command is as follows:"
           , "   "
           , "   draw thing [options]"
           , "   "
           , "where \"thing\" is one of:"
           , "   edt"
           , "   arg <number>"
           , "   result"
           , "   "
           , "Note that the \"arg\" thing must be followed by a number"
           , "identifying the appropriate argument to draw. You must use"
           , "the same numbers as given in the display of the derivation."
           , "   "
           , "The valid options are:"
           , "   -f <filename>"
           , "   -d <depth bound>"
           , "   "
           , "The filename refers to the file that you want to write the"
           , "\"dot\" graph into: for example /tmp/foo.dot"
           , "The depth bound refers to how deep you would like the"
           , "picture to be. This is important for large edts or values"
           , "because \"dot\" is not happy with very big graphs."
           , "Buddha terminates the branches of the graphs when the depth"
           , "bound is reached."
           , "   "
           , "If you don't supply any options then buddha will use the"
           , "default settings. The value of those defaults can be viewed"
           , "and changed using the \"set\" command."
           , "   "
           , "The drawing of the edt is done from the current node"
           , "downwards. The resulting graph depicts a \"call graph\""
           , "which can be useful when browsing through the tree."
           , "   "
           , "Here are some example usages:"
           , "   draw arg 3"
           , "   draw result"
           , "   draw edt"
           , "   draw edt -f /tmp/edt.dot"
           , "   draw edt -f /home/me/wally -d 15"
           , "   draw arg 1 -d 12"
           , "   draw result -d 5 -f res.dot"
           ]

forgetInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Forget any judgements that have been remembered."
   long  = [ short
           , "By default buddha remembers the judgements that"
           , "you make about the derivations that it shows."
           , "If you issue this command then it will clear its"
           , "memory - thus all judgements remembered so far"
           , "will be forgotten."
           , "   "
           , "Whether or not future judgements are remembered"
           , "depends on the setting called \"remember\". You can" 
           , "change this setting using the \"set\" command."
           , "   "
           , "See the help article on \"set\" for more"
           , "information."
           ]

sizeInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Show the size of the EDT from the current derivation."
   long  = [ short
           , "This command counts the number of nodes that are"
           , "in the EDT (callgraph) that is rooted at the"
           , "current node."
           ]

depthInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Show max depth of the EDT from the current derivation."
   long  = [ short
           , "This command determines the maximum depth"
           , "of all branches of the EDT (callgraph) rooted"
           , "at the current node."
           ]

restartInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Re-start debugging from the top of the EDT."
   long  = [ short
           , "This command allows you to jump from your current"
           , "position in the EDT to the top. This has"
           , "the effect of starting the debugging session all over"
           , "again. It can be quite useful if you get lost."
           ]

observeInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Show calls made to a given function or constant."
   long  = [ short
           , "If you want to see all the calls to a function"
           , "\"foo\", use:"
           , "   observe foo"
           , "Beware: if there are many calls to the observed"
           , "function you will get a lot of output printed."
           ]

jumpInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Jump to a derivation in the tree."
   long  = [ short
           , "Each node in the EDT has a unique number. That number"
           , "is printed in square brackets for each derivation."
           , "You can jump directly to the node by specifying its"
           , "number."
           , "    "
           , "For example:"
           , "   jump 12"
           , "takes you directly to whatever derivation is at node"
           , "number 12."
           , "   "
           , "To go back to the node that you jumped from, use the"
           , "\"back\" command."
           ]

backInfo
   = InInfo { shortInfo = short, longInfo = long }
   where
   short = "Go back to the last derivation that you jumped from."
   long  = [ short
           , "The \"jump\" command moves you from one node in the EDT"
           , "to another. When you make such a jump buddha remembers"
           , "the old node and any partial diagnosis that might have"
           , "been made up to that point." 
           , "   "
           , "The \"back\" command will take you back to the node"
           , "that you jumped from and you can resume debugging from"
           , "exactly where you left."
           , "   "
           , "It doesn't matter how many jumps you have made, you will"
           , "always be able to go back through all of them."
           ]
