{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Meta 

        Description:            Representation of Syntax and values.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Meta 
   ( Description (..)
   , Exception (..)
   , FunMap
   , Exp (..)
   , Stmt (..)
   , Pat(..)
   , Literal(..)
   , Equation(..)
   , Alt(..)
   , Val (..)
   , v
   , Type (..)
   , QType
   , Constraint
   , Context
   ) where

import Data.Word
   ( Word )

--------------------------------------------------------------------------------

type Ident = String

-- an object value, not the quoted form
data Val = forall a . V a

-- types
data Type
   = TFun   Type Type
   | TTuple [Type]
   | TList  Type
   | TApp   Type Type
   | TVar   Ident 
   | TCon   Ident 
  deriving (Eq, Ord, Show)

-- qualified types
type QType = (Context, Type)

type Context    = [Constraint]
type Constraint = ( Ident     -- the class name
                  , [Type]    -- arguments, Haskell 98 only allows one var type here
                  )

instance Eq Val where
   _v1 == _v2 = False

instance Show Val where
   show _v = "<Val>"

instance Ord Val where
   _v1 <= _v2 = True

-- Descriptions are for printing values to the Oracle
data Description
   = DFun FunMap            -- function as a map
   | DThunk Word            -- a thunk
   | DCycle Word            -- a (abstract) cycle
   | DExp Exp               -- a term
   | DException Exception   -- an exception
   | DAbstract String       -- something abstract
   deriving (Eq, Ord, Show)

-- an exceptional value, only for printing
data Exception
   = AsyncExcept
   | SyncExcept Exp
   deriving (Eq, Ord, Show)

-- the map of a function
type FunMap = [( Exp -- argument
               , Exp -- result
              )]

-- expression
data Exp
   = Ei Ident             -- identifier
   | Elit Literal         -- literal
   | Ea  [Exp]            -- prefix application
   | Eia Exp Exp Exp      -- infix application
   | Elam [Pat] Exp       -- lambda
   | ElamS String Int Int -- represent a lambda just by its src location
   | Elet [Equation] Exp  -- let
   | Ec Exp [Alt]         -- case
   | Etu [Exp]            -- tuple
   | Elist [Exp]          -- list
   | Elc Exp [Stmt]       -- list comprehension
   | Ep Exp               -- parens
   | Ev Val               -- value
   | Eio                  -- an IO value
   | Enull                -- where we don't really want a representation 
   | EDescr Description   -- a function map, thunk, cycle, exception (used for printing)
   deriving (Eq, Ord, Show)

v :: a -> Exp
v x = Ev (V x)

-- statements
data Stmt
   = Sg Pat Exp       -- generator
   | Sq Exp           -- qualifier
   | Sl [Equation]    -- let
   deriving (Eq, Ord, Show)

-- literal value
data Literal
   = Lc Char               -- character
   | Li Integer            -- int or integer
   | Ls String             -- quoted string 
   | Lf Double             -- float or double, XXX representation probably should be a rational
   deriving (Eq, Ord, Show)

-- pattern
data Pat 
   = Pi Ident          -- ident
   | Plit Literal      -- literal
   | Pn Pat            -- negated 
   | Pia Pat Ident Pat -- infix application
   | Pa Ident [Pat]    -- application
   | Ptu [Pat]         -- tuple
   | Plist [Pat]       -- list
   | Pp Pat            -- paren
   | Pas Ident Pat     -- as
   | Pw                -- wildcard
   | Pir Pat           -- irrefutable pattern
   deriving (Eq, Ord, Show)

-- bindings
data Equation
   = Df Ident [Pat] Exp    -- function binding
   | Dp Pat Exp            -- pattern binding
   deriving (Eq, Ord, Show)

-- case alternative
data Alt
   = A Pat Exp
   deriving (Eq, Ord, Show)
