{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Oracle 

        Description:            The Oracle. 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module Oracle 
   ( askOracle ) 
   where

import List 
   ( null )

import Data.FiniteMap 
   ( addToFM_C
   , lookupFM
   )

import Data 
   ( DebugState (..)
   , Config (..)
   , Judgement (..)
   , JudgeDatabase
   , Response (..)
   , Derivation (..)
   , EDT
   , readGlobalState
   , readGlobalStateConfig
   , updateGlobalState
   )

import CLI 
   ( cli )

import IOUtils 
   ( tabIO )

import Data.PackedString
   ( unpackPS )

import EDT
   ( derivation )

import Monad
   ( when )

import ColourString
   ( decorIOLn )

--------------------------------------------------------------------------------

ifTrace :: String -> IO ()
ifTrace msg 
   = do wantTrace <- readGlobalStateConfig config_trace 
        when wantTrace $
           tabIO $ decorIOLn $ "<oracle: " ++ msg ++ ">"

askOracle :: Bool -> EDT -> IO Response
askOracle showNode node 
   = do judgeDatabase <- readGlobalState state_judgeDatabase 
        derivation <- derivation node
        ifTrace $ "looking up answer in memory for an application of " ++ 
                        (unpackPS $ deriv_name derivation)
        case lookupDerivation judgeDatabase derivation of
           -- prune constants that we have seen before to avoid getting into loops
           Just judgement -> do ifTrace $ "found this judgement: " ++ show judgement 
                                if isConstant derivation
                                   then do ifTrace $ "pruning a constant that I've seen before" 
                                           return $ Left Correct
                                   else return $ Left judgement
           -- go to the command line interface
           Nothing 
              -> do ifTrace $ "answer not in memory, asking the user"
                    response <- cli showNode node
                    case response of
                       Left judgement 
                          -> do remember <- readGlobalStateConfig config_rememberAnswers 
                                when (remember && judgementToRemember judgement) $
                                   do ifTrace "adding the answer to memory" 
                                      updateJudgeDatabase (derivation, judgement)
                                return response 
                       Right command -> return response

-- True if a judegment is worth remembering
judgementToRemember :: Judgement -> Bool
judgementToRemember Unknown = False
judgementToRemember other   = True 

-- look for a derivation in the database
lookupDerivation :: JudgeDatabase -> Derivation -> Maybe Judgement
lookupDerivation db deriv
   = case lookupFM db (deriv_name deriv) of
        Nothing -> Nothing
        Just entries -> lookupEntry entries deriv
   where
   lookupEntry :: [(Derivation, Judgement)] -> Derivation -> Maybe Judgement
   lookupEntry [] d = Nothing
   lookupEntry ((d2, j) : rest) d1
      | d1 == d2 = Just j
      | otherwise = lookupEntry rest d1

-- add a derivation to the database if it is not already there
updateJudgeDatabase :: (Derivation, Judgement) -> IO () 
updateJudgeDatabase item@(deriv, judgement)
   = updateGlobalState updater
   where
   updater :: DebugState -> DebugState
   updater state
      = let oldJudgements = state_judgeDatabase state
            newJudgements = addToFM_C combiner oldJudgements (deriv_name deriv) [item]
        in state { state_judgeDatabase = newJudgements } 
   -- invariant: we only add to the map if it is not already there
   -- this invariant is not checked in the code for efficiency
   combiner :: [(Derivation, Judgement)] -> [(Derivation, Judgement)] 
                                         -> [(Derivation, Judgement)]
   combiner oldItem [newItem] = newItem : oldItem

-- true if a derivation corresponds to a constant
-- (something without arguments)
isConstant :: Derivation -> Bool
isConstant d = null $ deriv_args d
