{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004 

        Module:                 PrettyExp

        Description:            Pretty print Exp representation of values.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module PrettyExp
   ( prettyExp
   , prettyPat
   , prettyQType
   , renderExp
   ) where

import Meta 
   ( Description (..)
   , Exception (..)
   , FunMap
   , Exp (..)
   , Stmt (..)
   , Pat(..)
   , Literal(..)
   , Equation(..)
   , Alt(..)
   , QType
   , Constraint
   , Context
   , Type (..)
   )

import Text.PrettyPrint

--------------------------------------------------------------------------------

renderExp :: Exp -> String
renderExp e = render $ prettyExp False e

prettyDescription :: Bool -> Description -> Doc
prettyDescription _ps (DFun fmap) = prettyFunMap fmap 
prettyDescription _ps (DThunk _unique) = text "?"
prettyDescription _ps (DCycle _unique) = text "<cycle>"
prettyDescription ps (DExp e) = prettyExp ps e
prettyDescription ps (DException e) = prettyException ps e
prettyDescription ps (DAbstract str) = text str 

prettyException :: Bool -> Exception -> Doc
prettyException _ps AsyncExcept = text "!"
prettyException ps (SyncExcept e) 
   = prettySynException e
   where
   prettySynException :: Exp -> Doc
   prettySynException exp
      = text "<Exception:" <+> (prettyExp False $ getExceptionConstructor exp) <> char '>'
   -- XXX this is fragile - it might break if GHC changes its implementation
   -- of the exception constructors in Control.Exception
   getExceptionConstructor :: Exp -> Exp 
   getExceptionConstructor (Ea [Ea es@(_:_)]) = head es 
   getExceptionConstructor e@(Ei _) = e 

prettyExp :: Bool -> Exp -> Doc
prettyExp _ps (Ei ident) = text ident
prettyExp _ps (Elit lit) = prettyLit lit
prettyExp ps exp@(Ea es) 
   | isList exp = prettyList exp 
   | isPrefixOpApp es = prefixOpApp ps es
   | otherwise  = parensIf ps $ hsep $ map (prettyExp True) es
prettyExp ps (Eia e1 e2 e3)
   = parensIf ps $ hsep $ map (prettyExp True) [e1,e2,e3]
-- check if the lambda is an operator section
prettyExp ps (Elam pats e)
   | isLambdaSection pats e = prettyLambdaSection ps pats e
   | otherwise 
        = parensIf ps $ 
           text "\\" <> 
           hsep (map prettyPat pats) <+> 
           text "->" <+>
           prettyExp False e
prettyExp ps (ElamS file row col)
   = text "<lambda: file =" <+> 
     text file <> comma <+>
     text "row =" <+> int row <> comma <+>
     text "col =" <+> int col <> text ">"
-- braces not needed if there is just one equation
prettyExp ps (Elet [eq] e)
   = parensIf ps $ 
        text "let" <+> 
        prettyEquation eq <+> 
        text "in" <+> 
        prettyExp False e 
prettyExp ps (Elet eqs e)
   = parensIf ps $ 
        text "let" <+> 
        braces (hsep (punctuate semi (map prettyEquation eqs))) <+> 
        text "in" <+> 
        prettyExp False e 
prettyExp ps (Ec e alts)
   = parensIf ps $
        text "case" <+> 
        prettyExp False e <+> 
        text "of" <+> 
        braces (hsep $ punctuate semi (map prettyAlt alts))
prettyExp ps (Etu es)
   = parens $ hsep $ punctuate comma $ map (prettyExp False) es
prettyExp _ps (Elist es)
   = brackets $ hsep $ punctuate comma $ map (prettyExp False) es
prettyExp ps (Elc e stmts)
   = brackets $
        prettyExp False e <+>
        text "|" <+>
        hsep (punctuate comma (map prettyStmt stmts))
prettyExp _ps (Ep e)
   = parens $ prettyExp False e
prettyExp ps (EDescr d) = prettyDescription ps d
prettyExp ps Enull = text "<null>" 
prettyExp ps Eio = text "<IO>" 

isList :: Exp -> Bool
isList (Ea [cons, _h, _t])
   | cons == Ei ":" = True 
   | otherwise = False
isList other = False

prettyList :: Exp -> Doc
-- []
prettyList exp@(Ei "[]") = text "[]"
-- [x]
prettyList exp@(Ea [_cons, h, Ei "[]"])
   = brackets $ prettyExp False h
-- (x:y:zs) or (x:?)
prettyList exp@(Ea [_cons, h, t])
   = hsep (lbrack <> prettyExp False h <> comma : prettyTail t)
   where
   prettyTail :: Exp -> [Doc]
   prettyTail exp@(Ea [_cons, h, Ei "[]"])
      = [prettyExp False h <> rbrack]
   prettyTail exp@(Ea [_cons, h, t])
      = prettyExp False h <> comma : prettyTail t
   prettyTail (EDescr (DThunk _unique)) = [text "..?"]
   prettyTail other = [prettyExp False other]

prettyLit :: Literal -> Doc
-- prettyLit (Lc c) = char '\'' <> char c <> char '\''
prettyLit (Lc c) = text $ show c 
prettyLit (Li i) = integer i
-- the show is needed to make sure non-printing chars are displayed
-- in escape mode (ie with a backslash) rather than treated literally
prettyLit (Ls s) = text $ show s
prettyLit (Lf d) = double d 

prettyStmt :: Stmt -> Doc
prettyStmt (Sg pat e) = prettyPat pat <+> text "<-" <+> prettyExp False e
prettyStmt (Sq e) = prettyExp False e
prettyStmt (Sl eqs) = text "let" <+> hsep (map prettyEquation eqs)

prettyPat :: Pat -> Doc
prettyPat (Pi ident) = text ident
prettyPat (Plit lit) = prettyLit lit
prettyPat (Pn p) = text "-" <> prettyPat p
prettyPat (Pia p1 ident p2) 
   = prettyPat p1 <+> text ident <+> prettyPat p2 
prettyPat (Pa ident pats)
   = text ident <+> hsep (map prettyPat pats)
prettyPat (Ptu ps)
   = parens $ hsep $ punctuate comma (map prettyPat ps)
prettyPat (Plist ps)
   = brackets $ hsep $ punctuate comma (map prettyPat ps)
prettyPat (Pp pat)
   = parens $ prettyPat pat
prettyPat (Pas ident pat)
   = text ident <> text "@" <> prettyPat pat
prettyPat Pw = text "_"
prettyPat (Pir pat) = text "~" <> prettyPat pat

prettyEquation :: Equation -> Doc
prettyEquation (Df ident pats e)
   = text ident <+> 
     hsep (map prettyPat pats) <+> 
     equals <+> 
     prettyExp False e
prettyEquation (Dp pat e)
   = prettyPat pat <+> equals <+> prettyExp False e

prettyAlt :: Alt -> Doc
prettyAlt (A pat e) 
   = prettyPat pat <+> text "->" <+> prettyExp False e

parensIf :: Bool -> Doc -> Doc
parensIf test d = if test then parens d else d

-- XXX could remove duplicates from the map here
prettyFunMap :: FunMap -> Doc
prettyFunMap funMap 
   = braces' $ hsep $ punctuate comma $ map prettyFunElem funMap 
   where
   prettyFunElem :: (Exp, Exp) -> Doc
   prettyFunElem (e1, e2)
      = prettyExp False e1 <+> text "->" <+> prettyExp False e2 
   braces' :: Doc -> Doc
   braces' d = braces $ space <> d <> space

-- true if the application is a prefix app of an operator
isPrefixOpApp :: [Exp] -> Bool
isPrefixOpApp [e1,e2,e3]
   | isOperator e1 = True
   | otherwise = False
isPrefixOpApp other = False

isOperator :: Exp -> Bool
isOperator (Ei ident) = isOpIdent ident
isOperator other = False
isOpIdent :: String -> Bool
isOpIdent [] = False
isOpIdent ident = head ident == '(' && last ident == ')'

prefixOpApp :: Bool -> [Exp] -> Doc
prefixOpApp ps [e1,e2,e3]
   = parensIf ps $
        prettyExp True e2 <+> 
        prettyExp False (dropParens e1) <+> 
        prettyExp True e3  

-- assumes that isOperator is True on this 
dropParens :: Exp -> Exp
dropParens (Ei ident) = Ei $ init $ tail $ ident
dropParens other = other

-- check if a lambda can be turned into an operator section:
isLambdaSection :: [Pat] -> Exp -> Bool
isLambdaSection [Pi pvar] (Ea [Ei _, Ei evar, e3])
   | pvar == evar = True
   | otherwise = False
isLambdaSection _pats _exp = False

prettyLambdaSection :: Bool -> [Pat] -> Exp -> Doc
prettyLambdaSection parens [_pat] (Ea [e1@(Ei _), e2, e3])
   = prettyExp parens $ Ea [modE1, e3]
   where
   modE1 = if isOperator e1 then dropParens e1 else addTicks e1

-- put back ticks around an operator name
addTicks :: Exp -> Exp
addTicks (Ei ident) = Ei $ '`':(ident ++ "`")
addTicks other = other 

prettyQType :: QType -> Doc
prettyQType (context, ty)
   = prettyContext context <+> prettyType False ty

prettyContext :: Context -> Doc
prettyContext [] = empty
prettyContext [c] = prettyConstraint c <+> doubleBarArrow 
prettyContext cs@(_:_)
   = (parens $ hsep $ punctuate comma $ map prettyConstraint cs) <+> doubleBarArrow 

prettyConstraint :: Constraint -> Doc
prettyConstraint (ident, tys)
   = hsep (text ident : map (prettyType False) tys)

doubleBarArrow :: Doc
doubleBarArrow = text "=>"

prettyType :: Bool -> Type -> Doc
prettyType ps (TFun t1 t2)
   = parensIf ps $ prettyType leftParens t1 <+> text "->" <+> prettyType False t2
   where
   leftParens :: Bool
   leftParens = isFunType t1
   isFunType :: Type -> Bool
   isFunType (TFun _t1 _t2) = True
   isFunType other          = False
prettyType _ps (TTuple ts)
   = parens $ hsep $ punctuate comma $ map (prettyType False) ts
prettyType _ps (TList t)
   = brackets $ prettyType False t 
prettyType ps (TApp t1 t2)
   = parensIf ps $ prettyType True t1 <+> prettyType True t2
prettyType _ps (TVar var)
   = text var
prettyType _ps (TCon con) 
   = text con
