{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2004

        Module:                 Settings 

        Description:            Display/edit settings.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Settings 
   ( set
   , getSetting
   ) 
   where

import IO 

import IOUtils

import Data 
   ( DebugState (..)
   , Config (..)
   , updateGlobalStateConfig
   , updateGlobalState
   , getGlobalState
   , emptyJudgeDatabase
   , Setting (..)
   , databaseSize
   )

import Defaults
   ( minScreenWidth )

import AnsiColour
   ( Colour
   , allColours )

import ColourString
   ( alertIOLn 
   , decorIO
   )

--------------------------------------------------------------------------------

set :: Setting -> IO () 
set ShowSetting 
   = do newline 
        tabIO $ putStrLn "The current settings:" 
        newline 
        state <- getGlobalState 
        let config = state_config state
        display "width"         (show $ config_screenWidth config) 
        display "remember"      (rememberStr state config) 
        display "trace"         (show $ config_trace config)
        display "cycles"        (show $ config_showCycles config)
        display "truncate"      (show $ config_truncate config)
        display "drawFile"      (config_drawFile config)
        display "drawDepth"     (show $ config_drawDepth config)
        display "prompt"        (config_prompt config)
        display "colours"       (show $ config_colours config)
        display "promptColour"  (show $ config_promptColour config)
        display "nameColour"    (show $ config_nameColour config)
        display "decorColour"   (show $ config_decorColour config)
        display "valueColour"   (show $ config_valueColour config)
        display "alertColour"   (show $ config_alertColour config)
        newline
   where
   rememberStr :: DebugState -> Config -> String
   rememberStr state config
      = show (config_rememberAnswers config) ++ " (" ++ (judgeMentStr size) ++ " remembered)"
      where
      size = databaseSize $ state_judgeDatabase state
   -- get the plurals right
   judgeMentStr size
      | size == 1 = "1 judgement"
      | otherwise = show size ++ " judgements"
   display :: String -> String -> IO ()
   display item value
      = do tabIO $ decorIO $ ljustify 15 item
           putStr " = "
           putStrLn value
set (Width i)
   = updateGlobalStateConfig (\s -> s { config_screenWidth = i })
set (Remember b)
   = updateGlobalStateConfig (\s -> s { config_rememberAnswers = b })
set ClearMemory
   = updateGlobalState (\s -> s { state_judgeDatabase = emptyJudgeDatabase })
set (Trace b)
   = updateGlobalStateConfig (\s -> s { config_trace = b })
set (Cycle b)
   = updateGlobalStateConfig (\s -> s { config_showCycles = b })
set (Truncate i)
   = updateGlobalStateConfig (\s -> s { config_truncate = i })
set (DrawFile file)
   = updateGlobalStateConfig (\s -> s { config_drawFile = file })
set (DrawDepth d)
   = updateGlobalStateConfig (\s -> s { config_drawDepth = d })
set (Prompt p)
   = updateGlobalStateConfig (\s -> s { config_prompt = p })
set (Colour b)
   = updateGlobalStateConfig (\s -> s { config_colours = b })
set (PromptColour c)
   = updateGlobalStateConfig (\s -> s { config_promptColour = c })
set (DecorColour c)
   = updateGlobalStateConfig (\s -> s { config_decorColour = c })
set (NameColour c)
   = updateGlobalStateConfig (\s -> s { config_nameColour = c })
set (ValueColour c)
   = updateGlobalStateConfig (\s -> s { config_valueColour = c })
set (AlertColour c)
   = updateGlobalStateConfig (\s -> s { config_alertColour = c })

getSetting :: [String] -> IO (Maybe Setting)
getSetting [] = return $ Just ShowSetting 
getSetting ["width", n]
   = getIntSetting n "width" (Just minScreenWidth) Nothing Width
getSetting ["remember", str]
   = getBooleanSetting str "remember" Remember
getSetting ["trace", str]
   = getBooleanSetting str "trace" Trace
getSetting ["cycles", str]
   = getBooleanSetting str "cycles" Cycle 
getSetting ["truncate", n]
   = getIntSetting n "truncate" (Just 0) Nothing Truncate
getSetting ["drawFile", str]
   = return $ Just $ DrawFile str
getSetting ["drawDepth", str]
   = getIntSetting str "drawDepth" (Just 0) Nothing DrawDepth 
getSetting ("prompt":str:strs)
   = return $ Just $ Prompt (unwords (str:strs))
getSetting ["colours", str]
   = getBooleanSetting str "colour" Colour
getSetting ["promptColour", str]
   = getColourSetting str "promptColour" PromptColour
getSetting ["decorColour", str]
   = getColourSetting str "decorColour" DecorColour
getSetting ["nameColour", str]
   = getColourSetting str "nameColour" NameColour
getSetting ["valueColour", str]
   = getColourSetting str "valueColour" ValueColour
getSetting ["alertColour", str]
   = getColourSetting str "alertColour" AlertColour
getSetting other
   = do tabIO $ alertIOLn "that setting is not available"
        return Nothing

getBooleanSetting :: String -> String -> (Bool -> Setting) -> IO (Maybe Setting)
getBooleanSetting str setName mkOpt
   = case truth str of
        Nothing
           -> do tabIO $ alertIOLn $ "the argument for " ++ setName ++
                                     " must be either true or false"
                 return Nothing
        Just b -> return $ Just $ mkOpt b 

getIntSetting :: String -> String -> Maybe Int -> Maybe Int -> (Int -> Setting) -> IO (Maybe Setting)
getIntSetting str setName lo hi mkOpt
   = case safeReadNat str lo hi of
        Nothing -> do tabIO $ alertIOLn $ str ++ " is not a valid argument for " ++ setName
                      tabIO $ alertIOLn $ "must be an integer" ++ showRange lo hi
                      return Nothing
        Just i  -> return $ Just $ mkOpt i
   where
   showRange :: Maybe Int -> Maybe Int -> String
   showRange Nothing Nothing = []
   showRange (Just i) Nothing  = " >= " ++ show i 
   showRange (Just i) (Just j) = " >= " ++ show i ++ "and <= " ++ show j
   showRange Nothing (Just j)  = " <= " ++ show j 

getColourSetting :: String -> String -> (Colour -> Setting) -> IO (Maybe Setting)
getColourSetting str setName mkOpt
   = case validColour str of
        Nothing
           -> do tabIO $ alertIOLn $ "the argument for " ++ setName ++
                                     " must be one of these colours:"
                 tabIO $ alertIOLn $ unwords $ map show allColours
                 return Nothing
        Just c -> return $ Just $ mkOpt c 

truth :: String -> Maybe Bool
truth s
   | s `elem` ["true",  "T", "t", "True",  "on",  "yes", "1"] = Just True
   | s `elem` ["false", "F", "f", "False", "off", "no",  "0"] = Just False
   | otherwise = Nothing

validColour :: String -> Maybe Colour 
validColour str
   = lookup str colourNameMap 
   where
   colourNameMap = [(show c, c) | c <- allColours]
