{-# OPTIONS -O0 -fno-cse #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003 

        Module:                 TablesUnsafe

        Description:            Tables for recording function calls and
                                applications for higher-order arguments. 

        Primary Authors:        Bernie Pope

        Notes:                  Note very carefully that this code uses
                                unsafePerformIO extensively to implement
                                global mutable variables.

                                To stop the compiler for doing the wrong
                                optimisations we must turn them off.

                                Tread very carefully in this module.

                                Also note that the code in this module is
                                likely to be called many times in the execution
                                of a debugging program - so please make it 
                                frugal. Careless extra cycles or bytes could
                                end up costing a lot.

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-}

module TablesUnsafe
   ( getFunCount
   , updateFunTable
   , callTable 
   , funTable
   , update
   , ref
   , con
   , readCallCount
   , inc
   ) where

import Data 
   ( Record (..)
   , FunRecord (..)
   )

import System.IO.Unsafe
   ( unsafePerformIO )

import Data.IORef
   ( IORef
   , newIORef
   , readIORef
   , writeIORef
   )

import Meta
   ( Val (V) )

import Data.PackedString
   ( PackedString )

--------------------------------------------------------------------------------

foreign import ccall unsafe "incC" inc :: Int -> Int
foreign import ccall "readCallCountC" readCallCount :: IO Int

{-# NOINLINE callTable #-}
callTable :: IORef [Record]
callTable = unsafePerformIO (newIORef [])

{-# NOINLINE funTable #-}
funTable :: IORef [FunRecord]
funTable = unsafePerformIO (newIORef [])

{-# NOINLINE funCount #-}
funCount :: IORef Int
funCount = unsafePerformIO (newIORef 0)

{-# NOINLINE getFunCount #-}
getFunCount :: (Int -> a) -> a 
getFunCount f 
   = let nextCount 
          = (unsafePerformIO $
               do oldCount <- readIORef funCount 
                  let newCount = oldCount + 1
                  writeIORef funCount $! newCount
                  return oldCount)
     in seq nextCount (f nextCount)

{-# NOINLINE update #-}
update :: Int -> Int -> PackedString -> [Val] -> Int -> PackedString -> a -> a 
update parent nodeId childName args line modName result
   = seq (unsafePerformIO 
             (do oldBase <- readIORef callTable
                 writeIORef callTable $! Rec parent nodeId childName args (V result) line modName : oldBase))
         result 

{-# NOINLINE updateFunTable #-}
updateFunTable :: Int -> Val -> a -> a 
updateFunTable number arg result
   = seq (unsafePerformIO 
             (do oldBase <- readIORef funTable 
                 writeIORef funTable (FunRec number arg (V result) : oldBase)))
         result 

{-# NOINLINE ref #-}
ref :: (a, Int) -> Int -> a
ref (value, child) parent
   = seq (unsafePerformIO
             (do oldBase <- readIORef callTable
                 writeIORef callTable $! Ref parent child : oldBase))
         value
         
{-# NOINLINE con #-}
con :: PackedString -> [Val] -> Int -> PackedString -> a -> (a, Int)
con name freeVars line modName value
   = let count = inc (length freeVars) in
     seq count (seq (unsafePerformIO
                        (do oldBase <- readIORef callTable
                            writeIORef callTable $! Constant count name (V value) line modName : oldBase))
                    (value, count))
