{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Depend

        Description:            Compute dependencies between modules.

        Primary Authors:        Bernie Pope

        Notes:                  Currently a hack based on ghc's -M option.
                                Ultimately we need a more robust method,
                                perhaps by using something like hmake.

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Depend 
   ( dependencies ) 
   where

import Data.Graph

import Text.ParserCombinators.Parsec

import Data.FiniteMap

import System          
   ( system )

import List            
   ( intersperse )

import ChatIO          
   ( ifChat 
   , ChatLevel (..) 
   )

import BuddhaName      
   ( buddhaNameVersion )

--------------------------------------------------------------------------------

type ModName = String

dependParse :: String -> FiniteMap ModName [ModName] 
dependParse str = foldr processLine emptyFM $ lines str

processLine :: String -> FiniteMap ModName [ModName] -> FiniteMap ModName [ModName] 
processLine line depends 
   = case parse parseLine "" line of
        Left err -> depends 
        Right (a, b) -> addToFM_C (\old new -> new ++ old) depends a [b] 

parseLine :: Parser (String, String)
parseLine
   = do spaces
        a <- nameDot "o"
        spaces
        colon
        spaces
        b <- dotSlashFile <|> hsOrHiFile
        return (a,b)

dotSlashFile :: Parser String
dotSlashFile
   = do string "./"
        hsOrHiFile

hsOrHiFile :: Parser String
hsOrHiFile
   = do name <- moduleName
        char '.'
        hsOrHiExtension <|> lhsExtension
        return name

lhsExtension :: Parser ()
lhsExtension
   = do string "lhs"
        return ()

hsOrHiExtension :: Parser () 
hsOrHiExtension 
   = do char 'h'
        char 'i' <|> char 's'
        return ()
        
colon :: Parser Char
colon = char ':'

nameDot :: String -> Parser String
nameDot afterDot
   = do name <- moduleName
        char '.'
        string afterDot 
        return name

-- any sequence of characters, but stop at a dot
-- this means the module name can't start with a dot
moduleName :: Parser String
moduleName = many1 (noneOf ".")

--------------------------------------------------------------------------------

edgeList :: String -> [(ModName, ModName, [ModName])]
edgeList str
   = [(m, m, deps) | (m, deps) <- fmToList $ dependParse str]

moduleOrder :: String -> [ModName]
moduleOrder str
   = flattenSCCs $ stronglyConnComp $ edgeList str

--------------------------------------------------------------------------------

-- magic command to give to ghc to make it compute module dependencies
ghcOptCommand :: String
ghcOptCommand = HASKELL_COMPILER ++ " -M -optdep-f -optdep" ++ dependFile

dependFile :: String
dependFile = "./Buddha/depend"

-- presume that the filepaths is non-empty and only contains the
-- names of Haskell files
dependencies :: [FilePath] -> IO [ModName]
dependencies files
   = do let systemCmd = concat $ ghcOptCommand : " " : (intersperse " " files)
        ifChat Verbose $ putStrLn $ buddhaNameVersion ++ 
                         ": calculating dependencies: " ++ systemCmd
        status <- system systemCmd 
        dependFileContents <- readFile dependFile
        return $ moduleOrder dependFileContents 
