{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003 

        Module:                 Derive

        Description:            Deriving type class instances.
                                A kind of poor-person's implementation which does
                                not handle any tricky context reduction stuff.
                                For most programs what is in here is sufficient.
                                If you want anything more complex then you will have to
                                write your own instances in the code! 

                                Apologies to anyone who might try to read this code
                                - it was mostly written in a big hurry and is
                                very messy, and hard to follow.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Derive 
   ( deriveInstances ) 
   where

import Language.Haskell.Syntax           -- everything 

import SyntaxUtils      
   ( bogusSrcLoc
   , trustedSrcLoc
   , fromHsName
   , mkParen
   , mkParenPat
   , foldApp 
   )

import Error            
   ( fatalError )

import List            
   ( nub 
   , intersperse
   , null
   )

import Maybe           
   ( mapMaybe )

import Infix           
   ( InfixMap 
   , lookupInfixMap
   , Fixity (..) 
   )

--------------------------------------------------------------------------------

-- some well known names
trueQName       = UnQual $ HsIdent "True"
truePat         = HsPApp trueQName [] 
falsePat        = HsPApp falseQName []  
falseQName      = UnQual $ HsIdent "False"
errorQName      = Qual prelude_mod $ HsIdent "error"
equalsQName     = Qual prelude_mod equalsName 
equalsOp        = HsQVarOp equalsQName
equalsName      = HsSymbol "=="
andOp           = HsQVarOp andQName 
andName         = HsSymbol "&&"
andQName        = Qual prelude_mod andName 
orOp            = HsQVarOp orQName 
orName          = HsSymbol "||"
orQName         = Qual prelude_mod orName 
fromEnumQName   = Qual prelude_mod fromEnumName 
fromEnumName    = HsIdent "fromEnum"
toEnumQName     = Qual prelude_mod toEnumName 
toEnumName      = HsIdent "toEnum"
showsPrecQName  = Qual prelude_mod showsPrecName 
showsPrecName   = HsIdent "showsPrec"
showsPrecPrimeQName  = Qual prelude_mod showsPrecPrimeName 
showsPrecPrimeName   = HsIdent "showsPrec'"
showParenQName  = Qual prelude_mod showParenName 
showParenName   = HsIdent "showParen"
showStringQName = Qual prelude_mod showStringName
showStringName  = HsIdent "showString"
showApplyQName  = Qual prelude_mod showApplyName
showApplyName   = HsIdent "showApply"
gtQName         = Qual prelude_mod gtName 
gtName          = HsSymbol ">"
gtOp            = HsQVarOp gtQName 
composeQOp      = HsQVarOp $ Qual prelude_mod $ HsSymbol "."
minBoundQName   = Qual prelude_mod minBoundName 
minBoundName    = HsIdent "minBound"
maxBoundQName   = Qual prelude_mod maxBoundName 
maxBoundName    = HsIdent "maxBound"
lteQName        = Qual prelude_mod lteName 
lteName         = HsSymbol "<="
lteOp           = HsQVarOp lteQName 
gteQName        = Qual prelude_mod gteName 
gteName         = HsSymbol ">="
gteOp           = HsQVarOp gteQName 
minusQName      = Qual prelude_mod minusName 
minusName       = HsSymbol "-"
minusOp         = HsQVarOp minusQName 
plusQName       = Qual prelude_mod plusName 
plusName        = HsSymbol "+"
plusOp          = HsQVarOp plusQName 
timesQName      = Qual prelude_mod timesName 
timesName       = HsSymbol "*"
timesOp         = HsQVarOp timesQName 
ltQName         = Qual prelude_mod ltName 
ltName          = HsSymbol "<"
ltOp            = HsQVarOp ltQName 
readsPrecQName  = Qual prelude_mod readsPrecName 
readsPrecName   = HsIdent "readsPrec"
readParenQName  = Qual prelude_mod readParenName 
readParenName   = HsIdent "readParen"
concatQName     = Qual prelude_mod concatName 
concatName      = HsIdent "concat"
lexQName        = Qual prelude_mod lexName 
lexName         = HsIdent "lex"

data DeriveInfo 
     = DeriveInfo
       { deriveInfo_cntxt    :: HsContext
       , deriveInfo_typeName :: HsName
       , deriveInfo_argNames :: [HsName]
       , deriveInfo_conDecls :: [HsConDecl]
       , deriveInfo_srcLoc   :: SrcLoc
       , deriveInfo_infixMap :: InfixMap
       }

deriveInstances :: InfixMap -> [HsDecl] -> [HsDecl]
deriveInstances infixMap decls 
   = concatMap derivInsts decls
   where
   derivInsts :: HsDecl -> [HsDecl]
   derivInsts (HsDataDecl sloc cntxt typeName args condecls derives)
      = derive info derives
      where
      info = DeriveInfo   { deriveInfo_cntxt    = cntxt
                          , deriveInfo_typeName = typeName
                          , deriveInfo_argNames = args
                          , deriveInfo_srcLoc   = sloc
                          , deriveInfo_infixMap = infixMap
                          , deriveInfo_conDecls = condecls -- this is fixed when used
                          }
   derivInsts (HsNewTypeDecl sloc cntxt typeName args condecl derives) 
      = derive info derives
      where
      info = DeriveInfo   { deriveInfo_cntxt    = cntxt
                          , deriveInfo_typeName = typeName
                          , deriveInfo_argNames = args
                          , deriveInfo_srcLoc   = sloc
                          , deriveInfo_infixMap = infixMap
                          , deriveInfo_conDecls = [condecl] -- this is fixed when used
                          }
   derivInsts otherDecl = []

-- derive all the required class instances for a given type
derive :: DeriveInfo -> [HsQName] -> [HsDecl]
derive deriveInfo derives
   = mapMaybe deriveDispatch (nub derives) 
   where
   deriveDispatch :: HsQName -> Maybe HsDecl
   deriveDispatch className
      = (selectClass className) deriveInfo 
      where
      selectClass name
         | isEq   name    = deriveEq 
         | isEnum name    = deriveEnum
         | isShow name    = deriveShow 
         | isBounded name = deriveBounded 
         | isOrd name     = deriveOrd
         | isRead name    = deriveRead
         | isIx   name    = deriveIx
         | otherwise      = nullDerive

-- an empty derive for classes we don't know about
nullDerive :: DeriveInfo -> Maybe HsDecl
nullDerive _info = Nothing  

--------------------------------------------------------------------------------

-- general utilities 
typeFromNames :: HsName -> [HsName] -> HsType
typeFromNames tyCon args
   = foldl HsTyApp (HsTyCon (UnQual tyCon)) $ map HsTyVar args

mkContext :: HsQName -> HsContext -> [HsName] -> HsContext
mkContext className typeContext args
   = nub (newContext ++ typeContext)
   where
   newContext = [(className, [HsTyVar v]) | v <- args]

mkVarNames :: Int -> [HsName]
mkVarNames n = [HsIdent $ "x" ++ show i | i <- [1..n]]

instantiateCon :: HsName -> [HsName] -> HsPat
instantiateCon conName args
   = mkParenPat $ HsPApp (UnQual conName) $ map HsPVar args

-- boilerplate for deriving arbitrary classes
deriveOneClass :: DeriveInfo -> HsQName 
                             -> (DeriveInfo -> [HsDecl]) 
                             -> Maybe HsDecl 
deriveOneClass deriveInfo className memberDecls 
   = Just $ HsInstDecl sloc classContext className [thisType] (memberDecls deriveInfo)
   where
   sloc     = deriveInfo_srcLoc   deriveInfo
   argNames = deriveInfo_argNames deriveInfo
   cntxt    = deriveInfo_cntxt    deriveInfo
   typeName = deriveInfo_typeName deriveInfo
   thisType = typeFromNames typeName argNames 
   classContext = mkContext className cntxt argNames 

--------------------------------------------------------------------------------

-- deriving Eq

eqClassName = Qual prelude_mod $ HsIdent "Eq" 
eqClassUnQualName = UnQual $ HsIdent "Eq" 

isEq :: HsQName -> Bool
isEq name = name == eqClassName || name == eqClassUnQualName

deriveEq :: DeriveInfo -> Maybe HsDecl
deriveEq info 
   = deriveOneClass info eqClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info
            sloc     = deriveInfo_srcLoc info
        in [eqMember sloc conDecls]

-- _ == _ = False
defaultEqMatch :: SrcLoc -> HsMatch
defaultEqMatch sloc
   = match
   where
   match = HsMatch sloc 
                   equalsName
                   [HsPWildCard, HsPWildCard]
                   (HsUnGuardedRhs $ HsCon falseQName) [] 

-- if there is only one constructor then there is no need for the default case
eqMember :: SrcLoc -> [HsConDecl] -> HsDecl
eqMember sloc condecls 
   | length condecls < 2 = HsFunBind $ matches
   | otherwise           = HsFunBind $ matches ++ [defaultEqMatch sloc]
   where 
   matches = map (equalsMatch sloc) condecls 

equalsMatch :: SrcLoc -> HsConDecl -> HsMatch 
equalsMatch sloc (HsConDecl conLoc conName tyArgs)
   = match 
   where 
   numArgs = length tyArgs
   newVarNames = mkVarNames (2 * numArgs)
   (topVars, bottomVars) = splitAt numArgs newVarNames 
   conInstance1 = instantiateCon conName topVars
   conInstance2 = instantiateCon conName bottomVars
   match = HsMatch sloc 
                   equalsName 
                   [conInstance1, conInstance2] 
                   (HsUnGuardedRhs $ mkEqualsRhsExp (topVars,bottomVars)) [] 

mkEqualsRhsExp :: ([HsName],[HsName]) -> HsExp
mkEqualsRhsExp ([],[]) = HsCon trueQName 
mkEqualsRhsExp (topVars, bottomVars) 
   = conjoinEqs $ pairWiseEquals topVars bottomVars 
   where
   pairWiseEquals :: [HsName] -> [HsName] -> [HsExp]
   pairWiseEquals [] [] = []
   pairWiseEquals (x1:xs) (y1:ys) 
      = thisEq : pairWiseEquals xs ys
      where
      thisEq = HsInfixApp (HsVar (UnQual x1)) equalsOp (HsVar (UnQual y1))
   conjoinEqs :: [HsExp] -> HsExp
   conjoinEqs [e1] = e1
   conjoinEqs (e1:e2:es)
      = HsInfixApp e1 andOp (conjoinEqs (e2:es)) 
   conjoinEqs other = error $ "conjoinEqs " ++ show other

--------------------------------------------------------------------------------

-- deriving Enum

enumClassName       = Qual prelude_mod $ HsIdent "Enum" 
enumClassUnQualName = UnQual $ HsIdent "Enum" 

isEnum :: HsQName -> Bool
isEnum name = name == enumClassName || name == enumClassUnQualName

deriveEnum :: DeriveInfo -> Maybe HsDecl
deriveEnum info 
   = deriveOneClass info enumClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info 
            sloc     = deriveInfo_srcLoc info
        in  [fromEnumMember sloc conDecls, toEnumMember sloc conDecls]

toEnumMember :: SrcLoc -> [HsConDecl] -> HsDecl
toEnumMember sloc cons
   = HsFunBind $ toEnumMatches sloc 0 cons 
 
toEnumMatches :: SrcLoc -> Int -> [HsConDecl] -> [HsMatch]
toEnumMatches _ _ [] = []
toEnumMatches sloc index (HsConDecl conLoc conName [] : cons)
   = match : matches
   where
   match = HsMatch sloc 
                   toEnumName
                   [indexPat]
                   (HsUnGuardedRhs $ mktoEnumRhsExp conName) []
   matches = toEnumMatches sloc (index + 1) cons
   indexPat = HsPLit $ HsInt $ fromIntegral index

toEnumMatches _sloc index (HsConDecl conLoc conName args@(_:_) : _cons)
   = fatalError __FILE__ __LINE__ errorStr
   where
   errorStr = "attempt to derive Enum on a non-nullary constructor: " ++ fromHsName conName

mktoEnumRhsExp :: HsName -> HsExp
mktoEnumRhsExp conName = HsCon (UnQual conName)

fromEnumMember :: SrcLoc -> [HsConDecl] -> HsDecl
fromEnumMember sloc cons = HsFunBind $ fromEnumMatches sloc 0 cons 
 
fromEnumMatches :: SrcLoc -> Int -> [HsConDecl] -> [HsMatch]
fromEnumMatches _ _ [] = []
fromEnumMatches sloc index (HsConDecl conLoc conName [] : cons)
   = match : matches
   where
   match = HsMatch sloc 
                   fromEnumName
                   [conPat]
                   (HsUnGuardedRhs $ mkfromEnumRhsExp index) []
   matches = fromEnumMatches sloc (index + 1) cons
   conPat = HsPApp (UnQual conName) [] 

fromEnumMatches sloc index (HsConDecl conLoc conName args@(_:_) : _cons)
   = fatalError __FILE__ __LINE__ errorStr
   where
   errorStr 
      = "attempt to derive Enum on a non-nullary constructor: " ++ 
        fromHsName conName

mkfromEnumRhsExp :: Int -> HsExp
mkfromEnumRhsExp index = HsLit $ HsInt $ fromIntegral index

--------------------------------------------------------------------------------

-- deriving Show 

showClassName       = Qual prelude_mod showName 
showClassUnQualName = UnQual showName 
showName            = HsIdent "Show"

isShow :: HsQName -> Bool
isShow name = name == showClassName || name == showClassUnQualName

deriveShow :: DeriveInfo -> Maybe HsDecl
deriveShow info 
   = deriveOneClass info showClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info 
            infixMap = deriveInfo_infixMap info
            sloc     = deriveInfo_srcLoc info
        in  [showMember sloc infixMap conDecls]

-- this implements fromShow, toShow 

showMember :: SrcLoc -> InfixMap -> [HsConDecl] -> HsDecl
showMember sloc infixMap conDecls
   = HsFunBind $ map (showsPrecMatch sloc infixMap) conDecls

showsPrecMatch :: SrcLoc -> InfixMap -> HsConDecl -> HsMatch
showsPrecMatch sloc infixMap (HsConDecl conLoc conName args)
   = HsMatch sloc  
             showsPrecName
             [dPat,conInstance,sPat]
             (mkShowsPrecRhs conName argVars infixMap)
             []
   where
   argVars     = mkVarNames $ length args
   conInstance = instantiateCon conName argVars 
   dPat        = HsPVar outerPrecName 
   sPat        = HsPVar nextStringName 

outerPrecName  = HsIdent "d"
outerPrecQName = UnQual outerPrecName 

nextStringName  = HsIdent "s"
nextStringQName = UnQual nextStringName 

-- the precedence of application
appPrec :: Integer
appPrec = 10

mkShowsPrecRhs :: HsName -> [HsName] -> InfixMap -> HsRhs 
-- nullary constructors appear differently
mkShowsPrecRhs conName [] _infixMap
   = HsUnGuardedRhs rhsExp
   where
   rhsExp = foldApp [HsVar showStringQName, 
                     HsLit $ HsString $ fromHsName conName,
                     HsVar nextStringQName] 

-- note we make use of Prelude.showApply which is non-standard
-- but significantly reduces the amount of gratuitous higher-order
-- code in the derived instance, which is good for buddha

mkShowsPrecRhs conName argNames@(_:_) infixMap 
   = HsUnGuardedRhs rhsExp
   where
   rhsExp = foldApp [HsVar showApplyQName, 
                     mkParen parenD, 
                     listOfShowThings,
                     HsVar nextStringQName] 
   parenD = foldApp [HsVar gtQName, 
                     HsVar outerPrecQName, 
                     HsLit $ HsInt thisFixity]
   showsStringCon = foldApp [HsVar showStringQName, 
                             HsLit $ HsString $ fromHsName conName]
   showsPrecArgs = map showsPrecArg argNames
   showsPrecArg :: HsName -> HsExp
   showsPrecArg name 
      = foldApp [HsVar showsPrecPrimeQName, 
                 HsLit $ HsInt $ thisFixity + 1, 
                 HsVar $ UnQual name]
   -- see if the constructor should go before or in between its arguments
   listOfShowThings 
      | isInfix   = HsList (head showsPrecArgs : showsStringCon : tail showsPrecArgs)
      | otherwise = HsList (showsStringCon:showsPrecArgs)
   fixity = lookupInfixMap (UnQual conName) infixMap
   (isInfix, thisFixity)
       = case fixity of
            Nothing  -> (False, appPrec)
            Just val -> (True,  fromIntegral $ fixity_prec val)

--------------------------------------------------------------------------------

-- deriving Bounded 

boundedClassName       = Qual prelude_mod boundedName 
boundedClassUnQualName = UnQual boundedName 
boundedName            = HsIdent "Bounded"

isBounded :: HsQName -> Bool
isBounded name = name == boundedClassName || name == boundedClassUnQualName

deriveBounded :: DeriveInfo -> Maybe HsDecl
deriveBounded info 
   = deriveOneClass info boundedClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info 
            sloc     = deriveInfo_srcLoc info
        in boundedMembers sloc conDecls

boundedMembers :: SrcLoc -> [HsConDecl] -> [HsDecl] 
boundedMembers sloc [HsConDecl conLoc conName args@(_:_)]
   = [minBoundDecl, maxBoundDecl]
   where
   minBoundDecl
      = HsFunBind [HsMatch sloc minBoundName [] minBoundBody []]
   minBoundBody 
      = HsUnGuardedRhs $ 
           foldApp ((HsCon $ UnQual conName) : replicate numArgs (HsVar minBoundQName))
   maxBoundDecl
      = HsFunBind [HsMatch sloc maxBoundName [] maxBoundBody []]
   maxBoundBody 
      = HsUnGuardedRhs $ 
           foldApp ((HsCon $ UnQual conName) : replicate numArgs (HsVar maxBoundQName))
   numArgs = length args

boundedMembers sloc conDecls@(_:_)
   = [minBoundDecl, maxBoundDecl]
   where
   minBoundDecl
      = HsFunBind [HsMatch sloc minBoundName [] minBoundBody []]
   minBoundBody = HsUnGuardedRhs $ HsCon $ UnQual firstConName 
   maxBoundDecl
      = HsFunBind [HsMatch sloc maxBoundName [] maxBoundBody []]
   maxBoundBody = HsUnGuardedRhs $ HsCon $ UnQual lastConName 
   firstConName = getConName (head conDecls)
   lastConName  = getConName (last conDecls)
   getConName (HsConDecl _conLoc name _args) = name

--------------------------------------------------------------------------------

-- deriving Ord 

ordClassName       = Qual prelude_mod ordName 
ordClassUnQualName = UnQual ordName 
ordName            = HsIdent "Ord"

isOrd :: HsQName -> Bool
isOrd name = name == ordClassName || name == ordClassUnQualName

deriveOrd :: DeriveInfo -> Maybe HsDecl
deriveOrd info 
   = deriveOneClass info ordClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info 
            sloc     = deriveInfo_srcLoc info
        in [ordMembers sloc conDecls]

-- if there is only one constructor there is no need for the default case
ordMembers :: SrcLoc -> [HsConDecl] -> HsDecl 
ordMembers sloc cons 
   | length cons < 2 = HsFunBind matches
   | otherwise       = HsFunBind $ matches ++ [defaultOrdMatch sloc]
   where
   matches = ordMatches sloc cons

defaultOrdMatch :: SrcLoc -> HsMatch
defaultOrdMatch sloc
   = match
   where
   match = HsMatch sloc 
                   lteName 
                   [HsPWildCard, HsPWildCard]
                   (HsUnGuardedRhs $ HsCon falseQName) [] 

ordMatches :: SrcLoc -> [HsConDecl] -> [HsMatch]
ordMatches sloc [] = []
ordMatches sloc (thisCon : restCons)
   = (lteSelf sloc thisCon : lteRest sloc thisCon restCons) ++ 
      ordMatches sloc restCons

lteSelf :: SrcLoc -> HsConDecl -> HsMatch
lteSelf sloc (HsConDecl _conLoc conName [])
   = HsMatch sloc 
             lteName 
             [conInstance, conInstance] 
             (HsUnGuardedRhs $ HsCon trueQName)
             [] 
   where
   conInstance = instantiateCon conName []

lteSelf sloc (HsConDecl _conLoc conName args@(_:_))
   = HsMatch sloc 
             lteName 
             [conInstance1, conInstance2] 
             (HsUnGuardedRhs $ lteRhsExp topArgs bottomArgs)
             [] 
   where
   numArgs  = length args
   newVars = mkVarNames (numArgs * 2)
   (topArgs, bottomArgs) = splitAt numArgs newVars
   conInstance1 = instantiateCon conName topArgs 
   conInstance2 = instantiateCon conName bottomArgs 

lteRhsExp :: [HsName] -> [HsName] -> HsExp
lteRhsExp [x] [y] 
   = foldApp [HsVar lteQName, 
              HsVar $ UnQual x, 
              HsVar $ UnQual y]
lteRhsExp (x1:x2:xs) (y1:y2:ys)
   = orExp
   where
   ltExp    = mkParen $ foldApp [HsVar ltQName, 
                                 HsVar $ UnQual x1, 
                                 HsVar $ UnQual y1]
   orExp    = mkParen $ foldApp [HsVar orQName, 
                                 ltExp, andExp]
   eqExp    = mkParen $ foldApp [HsVar equalsQName, 
                                 HsVar $ UnQual x1, 
                                 HsVar $ UnQual y1]
   andExp   = mkParen $ foldApp [HsVar andQName, 
                                 eqExp, rightExp]
   rightExp = mkParen $ lteRhsExp (x2:xs) (y2:ys)

lteRest :: SrcLoc -> HsConDecl -> [HsConDecl] -> [HsMatch]
lteRest _sloc thisCon [] = []
lteRest sloc this@(HsConDecl _conLoc1 thisConName args1) 
                  (HsConDecl _conLoc2 nextConName args2 : rest)
   = thisMatch : restMatches
   where
   restMatches = lteRest sloc this rest
   thisMatch = HsMatch sloc 
                       lteName
                       [thisConInstance, nextConInstance]
                       (HsUnGuardedRhs $ HsCon trueQName)
                       []
   newVars = mkVarNames $ numArgs1 + numArgs2
   (thisNewVars, nextNewVars) = splitAt numArgs1 newVars
   numArgs1 = length args1
   numArgs2 = length args2
   thisConInstance = instantiateCon thisConName thisNewVars
   nextConInstance = instantiateCon nextConName nextNewVars

--------------------------------------------------------------------------------

-- deriving Read 

readClassName       = Qual prelude_mod readName 
readClassUnQualName = UnQual readName 
readName            = HsIdent "Read"

isRead :: HsQName -> Bool
isRead name = name == readClassName || name == readClassUnQualName

deriveRead :: DeriveInfo -> Maybe HsDecl
deriveRead info 
   = deriveOneClass info readClassName members
   where
   members info 
      = let conDecls = deriveInfo_conDecls info 
            infixMap = deriveInfo_infixMap info
            sloc     = deriveInfo_srcLoc info
        in [readMember sloc infixMap conDecls]

readMember :: SrcLoc -> InfixMap -> [HsConDecl] -> HsDecl 
readMember sloc infixMap cons 
   = HsFunBind [readMatch sloc infixMap cons]

initInputName = HsIdent "s0"

readMatch :: SrcLoc -> InfixMap -> [HsConDecl] -> HsMatch
readMatch sloc infixMap conDecls
   = HsMatch sloc 
             readsPrecName
             [HsPVar outerPrecName, HsPVar initInputName]
             (HsUnGuardedRhs $ appendAllReads $ map (readCon infixMap) conDecls)
             []

appendAllReads :: [HsExp] -> HsExp
appendAllReads exps
   = foldApp [HsVar concatQName, HsList exps]

readCon :: InfixMap -> HsConDecl -> HsExp
readCon infixMap (HsConDecl conLoc conName args)
   = foldApp [readParenExp, mkParen lambdaExp, HsVar $ UnQual initInputName]
   where
   readParenExp = foldApp [HsVar readParenQName, mkParen compareExp] 
   compareExp
      = case null args of
           True  -> HsCon falseQName 
           False -> foldApp [HsVar gtQName, HsVar $ UnQual $ outerPrecName, 
                                            HsLit $ HsInt thisFixity]
   lambdaExp = HsLambda trustedSrcLoc 
                        [(HsPVar initInputName)]
                        (listCompExp isInfix thisFixity conName (length args))
   fixity = lookupInfixMap (UnQual conName) infixMap
   (isInfix, thisFixity)
       = case fixity of
            Nothing  -> (False, appPrec)
            Just val -> (True,  fromIntegral $ fixity_prec val)

listCompExp :: Bool -> Integer -> HsName -> Int -> HsExp
listCompExp isInfix thisFixity conName numArgs
   = HsListComp parseResult stmts
   where
   parseResult = HsTuple [conApp, HsVar $ UnQual $ remainingInputName]
   conApp = foldApp ((HsCon $ UnQual conName) : map (HsVar . UnQual) newArgNames)
   newArgNames = mkVarNames numArgs 
   conLexTuple = (HsPLit $ HsString $ fromHsName conName, HsVar $ lexQName)
   argTuples = [(HsPVar a, readsPrecApp) | a <- newArgNames]
   readsPrecApp = foldApp [HsVar readsPrecQName, HsLit $ HsInt (thisFixity + 1)]
   (stmts, finalCount) = parseStmts 0 pairOrdering 
   -- to allow the constructor to go second if it is infix
   pairOrdering 
      | numArgs == 0 = [conLexTuple]
      | isInfix = (head argTuples) : conLexTuple : (tail argTuples)
      | otherwise = conLexTuple : argTuples
   remainingInputName = HsIdent $ "s" ++ show finalCount

parseStmts :: Int -> [(HsPat, HsExp)] -> ([HsStmt], Int)
parseStmts n exps
   = parseStmtsAcc n exps []
   where
   parseStmtsAcc n [] acc = (reverse acc, n)
   parseStmtsAcc n ((pat,e):rest) acc
      = parseStmtsAcc (n+1) rest (thisStmt : acc)
      where
      thisStmt = HsGenerator trustedSrcLoc
                             (HsPTuple [pat, HsPVar $ inputIdent (n+1)])
                             (foldApp [e, HsVar $ UnQual $ inputIdent n])
      inputIdent :: Int -> HsName
      inputIdent i = HsIdent $ "s" ++ show i

--------------------------------------------------------------------------------

-- deriving Ix 

ixClassName       = Qual (Module "Ix") ixName
ixClassUnQualName = UnQual ixName
ixName            = HsIdent "Ix"

isIx :: HsQName -> Bool
isIx name = name == ixClassName || name == ixClassUnQualName

deriveIx :: DeriveInfo -> Maybe HsDecl
deriveIx info
   -- = deriveOneClass info ixClassName members
   = deriveOneClass info ixClassUnQualName members
   where
   members info
      = let conDecls = deriveInfo_conDecls info
            sloc     = deriveInfo_srcLoc info
        in ixMembers sloc conDecls

ixMembers :: SrcLoc -> [HsConDecl] -> [HsDecl]
ixMembers sloc cons 
   = [ inRangeMembers sloc cons
     , indexMembers sloc cons
     , rangeMembers sloc cons
     ] 

inRangeMembers :: SrcLoc -> [HsConDecl] -> HsDecl
inRangeMembers sloc cons 
   | all nullary cons = inRangeNullary sloc cons
   | length cons == 1 = inRangeNonNull sloc (head cons)
   | otherwise = fatalError __FILE__ __LINE__ $ "attempt to derive Ix on an invalid type"

indexMembers :: SrcLoc -> [HsConDecl] -> HsDecl
indexMembers sloc cons 
   | all nullary cons = indexNullary sloc cons
   | length cons == 1 = indexNonNull sloc (head cons)
   | otherwise = fatalError __FILE__ __LINE__ $ "attempt to derive Ix on an invalid type"

rangeMembers :: SrcLoc -> [HsConDecl] -> HsDecl
rangeMembers sloc cons 
   | all nullary cons = rangeNullary sloc cons
   | length cons == 1 = rangeNonNull sloc (head cons)
   | otherwise = fatalError __FILE__ __LINE__ $ "attempt to derive Ix on an invalid type"

nullary :: HsConDecl -> Bool
nullary (HsConDecl sloc name xs) = null xs
nullary (HsRecDecl sloc name xs) = null xs

inRangeName = HsIdent "inRange"
inRangeQName = UnQual inRangeName

inRangeNullary :: SrcLoc -> [HsConDecl] -> HsDecl
inRangeNullary sloc cons
   = HsFunBind [inRangeNullaryMatch sloc cons]
   where
   inRangeNullaryMatch :: SrcLoc -> [HsConDecl] -> HsMatch
   inRangeNullaryMatch sloc cons
      = HsMatch sloc inRangeName [loHiPat, xPat]
                (HsUnGuardedRhs rhs) [posDef cons]
   rhs = andInfix
   andInfix = HsInfixApp gteInfix andOp lteInfix
   gteInfix = HsInfixApp posX gteOp posLo
   posX  = foldApp [HsVar posQName, HsVar xQName]
   posLo = foldApp [HsVar posQName, HsVar loQName]
   lteInfix = HsInfixApp posX lteOp posHi
   posHi = foldApp [HsVar posQName, HsVar hiQName]

xName = HsIdent "x"
xQName = UnQual xName
xsName = HsIdent "xs"
xsQName = UnQual xsName
loName = HsIdent "lo"
loQName = UnQual loName
hiName = HsIdent "hi"
hiQName = UnQual hiName

loHiPat = HsPTuple [HsPVar loName, HsPVar hiName]
xPat = HsPVar xName

posQName = UnQual posName
posName = HsIdent "pos"

rangeName = HsIdent "range"
rangeQName = UnQual rangeName 

indexName = HsIdent "index"
indexQName = UnQual indexName 

-- pos computes the position of a constructor 
-- data T = A | B | C | D
-- pos A = 0, pos B = 1, pos C = 2, pos D = 3
posDef :: [HsConDecl] -> HsDecl
posDef cons = HsFunBind $ posMatches 0 (map conNameLoc cons)

posMatches :: Int -> [(HsName, SrcLoc)] -> [HsMatch]
posMatches _ [] = []
posMatches index ((conName, conLoc) : cons)
   = match : matches
   where
   match = HsMatch conLoc 
                   posName
                   [conPat]
                   (HsUnGuardedRhs $ mkposRhsExp index) []
   matches = posMatches (index + 1) cons
   conPat = HsPApp (UnQual conName) []

mkposRhsExp :: Int -> HsExp
mkposRhsExp index 
   = HsExpTypeSig bogusSrcLoc (HsLit $ HsInt $ fromIntegral index)
                              (HsQualType [] (HsTyCon $ UnQual $ HsIdent "Int"))

conNameLoc :: HsConDecl -> (HsName, SrcLoc)
conNameLoc (HsConDecl sloc name args) = (name, sloc)
conNameLoc (HsRecDecl sloc name args) = (name, sloc)

indexNullary :: SrcLoc -> [HsConDecl] -> HsDecl
indexNullary sloc cons 
   = HsFunBind [indexNullaryMatch sloc cons]
   where
   indexNullaryMatch :: SrcLoc -> [HsConDecl] -> HsMatch
   indexNullaryMatch sloc cons
      = HsMatch sloc indexName [loHiPat, xPat]
                (HsGuardedRhss [trueRhs, falseRhs]) [posDef cons]
   trueRhs = HsGuardedRhs sloc guard1 body1
   guard1 = foldApp [HsVar inRangeQName, 
                     HsTuple [HsVar loQName, HsVar hiQName], 
                     HsVar xQName]
   body1 = HsInfixApp posX minusOp posLo
   posX  = foldApp [HsVar posQName, HsVar xQName]
   posLo = foldApp [HsVar posQName, HsVar loQName]
   falseRhs = HsGuardedRhs sloc guard2 body2
   guard2 = HsVar $ UnQual $ HsIdent "otherwise"
   body2 = foldApp [HsVar errorQName, HsLit (HsString "index error")]

rangeNullary :: SrcLoc -> [HsConDecl] -> HsDecl
rangeNullary sloc cons
   = HsFunBind [rangeNullaryMatch sloc cons]
   where
   rangeNullaryMatch :: SrcLoc -> [HsConDecl] -> HsMatch
   rangeNullaryMatch sloc cons
      = HsMatch sloc rangeName [loHiPat]
                (HsGuardedRhss [rhs1, rhs2]) [posDef cons, rangePrimeDef]
   rhs1 = HsGuardedRhs sloc guard1 (HsList []) 
   rhs2 = HsGuardedRhs sloc guard2 body2
   guard1 = HsInfixApp posLo gtOp posHi 
   guard2 = HsVar $ UnQual $ HsIdent "otherwise"
   posLo = foldApp [HsVar posQName, HsVar loQName]
   posHi = foldApp [HsVar posQName, HsVar hiQName]
   body2 = foldApp [HsVar rangePrimeQName, consToList cons]

consToList :: [HsConDecl] -> HsExp
consToList cons
   = HsList $ consToList' cons
   where
   consToList' [] = []
   consToList' (c:cs) = HsCon (conName c) : consToList' cs
   conName con = UnQual $ fst $ conNameLoc con 

{- range' [] = []
   range' (x:xs) 
      | pos x > pos hi = []
      | pos x >= pos lo = x : range' xs
      | otherwise = range' xs
-}

rangePrimeName = HsIdent "range'"
rangePrimeQName = UnQual rangePrimeName 

rangePrimeDef :: HsDecl
rangePrimeDef 
   = HsFunBind [baseCase, recursiveCase]
   where
   consQName = Special HsCons 
   consOp = HsQConOp consQName
   baseCase :: HsMatch
   baseCase = HsMatch trustedSrcLoc rangePrimeName [HsPList []] 
                      (HsUnGuardedRhs (HsList [])) []
   recursiveCase
      = HsMatch trustedSrcLoc rangePrimeName 
                [(HsPInfixApp (HsPVar xName) consQName (HsPVar xsName))]
                (HsGuardedRhss [rhs1, rhs2, rhs3]) []
   rhs1 = HsGuardedRhs trustedSrcLoc guard1 (HsList [])
   guard1 = HsInfixApp posX gtOp posHi
   rhs2 = HsGuardedRhs trustedSrcLoc guard2 body2 
   guard2 = HsInfixApp posX gteOp posLo
   body2 = HsInfixApp (HsVar xQName) consOp rangePrimeXs 
   posX  = foldApp [HsVar posQName, HsVar xQName]
   posHi = foldApp [HsVar posQName, HsVar hiQName]
   posLo = foldApp [HsVar posQName, HsVar loQName]
   rhs3 = HsGuardedRhs trustedSrcLoc guard3 rangePrimeXs 
   rangePrimeXs = foldApp [HsVar rangePrimeQName, HsVar xsQName]
   guard3 = HsVar $ UnQual $ HsIdent "otherwise"

inRangeNonNull :: SrcLoc -> HsConDecl -> HsDecl
inRangeNonNull sloc con 
   = HsFunBind [match] 
   where
   arity = conArity con
   conName = fst $ conNameLoc con 
   match = HsMatch sloc inRangeName [pat1, pat2]
                   (HsUnGuardedRhs rhs) []
   rhs = foldl1 (\e1 e2 -> HsInfixApp e1 andOp e2) inRangeExps 
   inRangeExps 
       = [foldApp [HsVar inRangeQName, 
                         HsTuple [HsVar $ UnQual $ HsIdent $ "l" ++ show i, 
                                  HsVar $ UnQual $ HsIdent $ "u" ++ show i],
                         HsVar $ UnQual $ HsIdent $ "i" ++ show i]
                  | i <- [1..arity]]
   pat1 = HsPTuple [pat11, pat12]
   pat2 = mkConPApp conName arity "i" 
   pat11 = mkConPApp conName arity "l"
   pat12 = mkConPApp conName arity "u"

mkConPApp :: HsName -> Int -> String -> HsPat
mkConPApp name arity var
   = HsPApp (UnQual name) (map (HsPVar . HsIdent) [var ++ show i | i <- [1..arity]]) 

conArity :: HsConDecl -> Int
conArity (HsConDecl sloc name args) = length args
conArity (HsRecDecl sloc name args) = sum (map (length.fst) args)

rangeNonNull :: SrcLoc -> HsConDecl -> HsDecl
rangeNonNull sloc con 
   = HsFunBind [match] 
   where
   arity = conArity con
   conName = fst $ conNameLoc con 
   match = HsMatch sloc rangeName [pat1]
                   (HsUnGuardedRhs rhs) []
   rhs = HsListComp exp stmts
   stmts = [ HsGenerator trustedSrcLoc (HsPVar $ HsIdent $ "i" ++ show n)
                         (foldApp [HsVar rangeQName, 
                                   HsTuple [HsVar $ UnQual $ HsIdent $ "l" ++ show n,
                                            HsVar $ UnQual $ HsIdent $ "u" ++ show n]])
             | n <- [1..arity]]
   exp = foldApp $ (HsCon $ UnQual conName) :
                   (map (HsVar . UnQual . HsIdent) ["i" ++ show n | n <- [1..arity]])
   pat1 = HsPTuple [pat11, pat12]
   pat11 = mkConPApp conName arity "l"
   pat12 = mkConPApp conName arity "u"

indexNonNull :: SrcLoc -> HsConDecl -> HsDecl
indexNonNull sloc con 
   = HsFunBind [match] 
   where
   arity = conArity con
   conName = fst $ conNameLoc con 
   match = HsMatch sloc indexName [pat1, pat2]
                   (HsUnGuardedRhs rhs) []
   rhs = indexBody arity
   pat1 = HsPTuple [pat11, pat12]
   pat2 = mkConPApp conName arity "i" 
   pat11 = mkConPApp conName arity "l"
   pat12 = mkConPApp conName arity "u"
   indexBody 1 = foldApp [HsVar indexQName, HsTuple [HsVar $ UnQual $ HsIdent "l1",
                                                    HsVar $ UnQual $ HsIdent "u1"],
                          HsVar $ UnQual $ HsIdent "i1"]
   indexBody n = HsInfixApp (e1 n) plusOp (e2 n)
   e1 n = foldApp [HsVar indexQName, HsTuple [HsVar $ UnQual $ HsIdent $ "l" ++ show n,
                                                    HsVar $ UnQual $ HsIdent $ "u" ++ show n],
                          HsVar $ UnQual $ HsIdent $ "i" ++ show n]
   e2 n = HsInfixApp (e3 n) timesOp (mkParen (e4 n))
   e3 n = foldApp [HsVar rangeSizeQName, HsTuple [HsVar $ UnQual $ HsIdent $ "l" ++ show n,
                                                    HsVar $ UnQual $ HsIdent $ "u" ++ show n]]
   rangeSizeQName = UnQual $ HsIdent "rangeSize" 
   e4 n = indexBody (n - 1)
