{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2002 

        Module:                 Desugar 

        Description:            Desugaring of the abstract syntax.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Desugar 
   ( runDesugar
   , desugarDecl
   , isMarkedSrcLoc 
   ) where

import Language.Haskell.Syntax           -- everything 

import SyntaxUtils      
   ( bogusSrcLoc 
   , trustedSrcLoc
   , madeUpSrcLoc
   , opExp
   , mkParen
   , mkParenPat
   , foldApp
   , dropQualifier
   , isQualified
   , updateName
   , patContainsLit
   , isSimplePat
   , varsFromPat
   )

import Utils            
   ( fst3
   , snd3
   , trd3 
   , (.&&.) 
   )

import Error            
   ( notSupported 
   , fatalError
   )

import Synonym

import Monad           
   ( mapAndUnzipM 
   , liftM
   , liftM2
   )

import Rename          
   ( defaultVarPrefix )

import Record          
   ( RecordMap
   , findUpdateCandidates
   , lookupRecord 
   )

--------------------------------------------------------------------------------

-- some well known names
-- True and False are not qualified but they should be. The reason that they
-- are not qualified is that Bool is not defined in Prelude_Buddha, but in 
-- Prelude. So when we transform the prelude this causes some naming problems.
-- A possible solution is to make Prelude_Buddha define Bool and make some
-- conversion function for primitives: Prelude_Buddha.Bool -> Prelude.Bool
-- and vice versa, but the implications of this change are not yet known
-- so we'll leave it as it is.

-- another solution might be to qualify them to something like P.True
-- in the actual prelude (to be transformed) we could have
-- import GHC.Prelude as P
-- and in regular (non prelude) modules we could have
-- import Prelude as P 

trueQName  = UnQual $ HsIdent "True"
truePat  = HsPApp trueQName [] 
falsePat = HsPApp falseQName []  
falseQName = UnQual $ HsIdent "False"
errorQName = Qual prelude_mod $ HsIdent "error"
returnQName = Qual prelude_mod $ HsIdent "return"
undefinedQName = Qual prelude_mod $ HsIdent "undefined"
failQName = Qual prelude_mod $ HsIdent "fail"
enumFromQName        = Qual prelude_mod $ HsIdent "enumFrom"
enumFromThenQName    = Qual prelude_mod $ HsIdent "enumFromThen"
enumFromToQName      = Qual prelude_mod $ HsIdent "enumFromTo"
enumFromThenToQName  = Qual prelude_mod $ HsIdent "enumFromThenTo"
-- we have to use the debug version, not negate
negateQName = Qual prelude_mod $ HsIdent "negate_bUdDhA"
-- equalsQName = UnQual $ HsSymbol "=="
equalsQName = Qual prelude_mod $ HsSymbol "=="
equalsOp = HsQVarOp equalsQName

desugarModule :: HsModule -> Desugar HsModule
desugarModule (HsModule sloc modName exports imports decls)
   = do newDecls <- mapM desugarDecl decls 
        return $ HsModule sloc modName exports imports newDecls

desugarDecl :: HsDecl -> Desugar HsDecl

desugarDecl (HsClassDecl sloc cntxt className args decls)
   = do newDecls <- mapM desugarDecl decls
        return $ HsClassDecl sloc cntxt className args newDecls

desugarDecl (HsInstDecl sloc cntxt className args decls)
   = do newDecls <- mapM desugarDecl decls
        return $ HsInstDecl sloc cntxt className args newDecls

desugarDecl decl@(HsDataDecl sloc cntxt typeName args condecls derives)
   = do newConDecls <- desugarConDecls condecls
        let thisDecl = HsDataDecl sloc cntxt typeName args newConDecls [] 
        return thisDecl 

desugarDecl decl@(HsNewTypeDecl sloc cntxt typeName args condecl derives)
   = do [newConDecl] <- desugarConDecls [condecl]
        let thisDecl = HsNewTypeDecl sloc cntxt typeName args newConDecl [] 
        return thisDecl 
         
desugarDecl (HsFunBind matches)
   = do newMatches <- desugarMatches matches
        return $ HsFunBind newMatches

{- the use of SrcLocs is very crucial here to determine how each construct is
   later transformed -} 

desugarDecl (HsPatBind sloc pat rhs wheres)
   = do newRhs <- desugarPatBindRhs rhs
        newWheres <- mapM desugarDecl wheres 
        newPat <- desugarRecPat pat
        return $ HsPatBind sloc newPat newRhs newWheres

desugarDecl (HsTypeSig sloc names qualType)
   = do synMap <- getSynonymMap 
        let newQType = remSynsFromQualType synMap qualType
        return $ HsTypeSig sloc names newQType

desugarDecl other = return other

desugarConDecls :: [HsConDecl] -> Desugar [HsConDecl]
desugarConDecls conDecls 
   = mapM desugarConDecl conDecls

desugarConDecl :: HsConDecl -> Desugar HsConDecl
desugarConDecl (HsConDecl sloc name args) 
   = do synMap <- getSynonymMap
        let newArgs = map (remSynsFromBangType synMap) args
        return $ HsConDecl sloc name newArgs 

desugarConDecl other 
   = fatalError __FILE__ __LINE__ $ "desugarConDecl: unknown constructor decl: " ++ show other
       
{-
   guarded rhs are turned into case 
 
   this is trivial for pattern bindings because there can only
   be one equation

    | g1 = e1
    | g2 = e2
    | gm = em

   becomes
 
   case g1 of
      True -> e1
      False -> case g2 of
                  True -> e2
                  False -> ... case gm of 
                                  True -> em 
-}
 
desugarPatBindRhs :: HsRhs -> Desugar HsRhs
desugarPatBindRhs (HsUnGuardedRhs e) 
   = liftM HsUnGuardedRhs $ desugarExp e

desugarPatBindRhs (HsGuardedRhss grhss)
  = do rhsExp <- desugarGuardedRhss grhss
       newRhsExp <- desugarExp rhsExp
       return $ HsUnGuardedRhs newRhsExp 

desugarGuardedRhss :: [HsGuardedRhs] -> Desugar HsExp
desugarGuardedRhss [] 
   = fail "desugarGuardedRhss: zero guarded rhss"
-- currently just make a singleton guard a case expression
-- don't deal with the possibly erroneous false case

desugarGuardedRhss [HsGuardedRhs sloc guard rhs]
  = do let newAlt = HsAlt (markSrcLoc sloc) (HsPApp trueQName []) (HsUnGuardedAlt rhs) []
       return $ HsCase guard [newAlt]

desugarGuardedRhss ((HsGuardedRhs sloc guard rhs):g2:gs)
   = do let trueAlt = HsAlt (markSrcLoc sloc) truePat  (HsUnGuardedAlt rhs) []
        falseCase <- desugarGuardedRhss (g2:gs)
        let falseAlt = HsAlt sloc falsePat (HsUnGuardedAlt falseCase) []
        return $ HsCase guard [trueAlt, falseAlt]

desugarMatches :: [HsMatch] -> Desugar [HsMatch]
desugarMatches [] = return []
-- functions of one equation are dealt with specially to avoid possible
-- redundant unfolding of pattern matching (id x = x) 
desugarMatches [match]
   = do newMatch <- desugarOneMatch match 
        return [newMatch]
desugarMatches ms@(match@(HsMatch sloc name pats rhs wheres) : rest@(_:_)) 
   = do newVars <- sequence $ replicate (length pats) freshVar
        let newVarNames = map fst3 newVars
            newVarPats = map snd3 newVars
        newRhs <- manyEquationsToOne newVarNames ms 
        return [HsMatch sloc name newVarPats newRhs []]

-- make sure all patterns in the head of an equation are
-- irrefutable. Refutable patterns are replaced by a variable
-- and a case is used on the rhs to match them
desugarOneMatch :: HsMatch -> Desugar HsMatch
desugarOneMatch match@(HsMatch sloc name pats rhs wheres)
   = case all isIrrefPat pats of
        -- all patterns are irrefutable
        True -> do newRhs <- desugarPatBindRhs rhs
                   newDecls <- mapM desugarDecl wheres
                   return $ HsMatch sloc name pats newRhs newDecls 
        -- at least one pattern is not irrefutable
        False -> do newRhs <- desugarPatBindRhs rhs 
                    newDecls <- mapM desugarDecl wheres 
                    (newPats, subs) <- replaceRefutablePats pats
                    let (discrim, casePat) =
                           case compare (length subs) 1 of
                              EQ -> let [(e,p)] = subs in (e,p)
                              GT -> (HsTuple $ map fst subs, HsPTuple $ map snd subs)
                    let newAlt = HsAlt bogusSrcLoc casePat 
                                       (hsRhsToHsGuardedAlts newRhs) 
                                       newDecls 
                    desugaredCase <- desugarCase $ HsCase discrim [newAlt]
                    let rhsWithCase = HsUnGuardedRhs desugaredCase 
                    -- return $ HsMatch sloc name newPats rhsWithCase (concat newDecls)
                    return $ HsMatch sloc name newPats rhsWithCase [] 

-- replace all refutable patterns in a list with variables
-- and record the substitutions made
replaceRefutablePats :: [HsPat] -> Desugar ([HsPat], [(HsExp, HsPat)])
replaceRefutablePats [] = return ([], [])
replaceRefutablePats (pat : rest)
   = do (restPats, restSubs) <- replaceRefutablePats rest
        case isIrrefPat pat of
           True  -> return (pat : restPats, restSubs)
           False -> do (_, newVarPat, newVarExp) <- freshVar 
                       return (newVarPat : restPats, (newVarExp, pat) : restSubs)

manyEquationsToOne :: [HsName] -> [HsMatch] -> Desugar HsRhs
manyEquationsToOne names patsRhss
   = do let discrim = case compare (length names) 1 of 
                         EQ -> (HsVar . UnQual) $ head names
                         GT -> HsTuple $ map (HsVar . UnQual) names 
        newAlts <- mkAlts patsRhss
        let newCase = HsCase discrim newAlts
        caseNoSugar <- desugarExp newCase
        return $ HsUnGuardedRhs caseNoSugar 
    where
    mkAlts :: [HsMatch] -> Desugar [HsAlt]
    mkAlts [] = return []
    mkAlts (HsMatch sloc _name pats rhs decls : rest)
       = do newRest <- mkAlts rest
            let newPat = case compare (length pats) 1 of
                            EQ -> head pats 
                            GT -> HsPTuple pats 
            let newAlt = HsAlt (markSrcLoc sloc) newPat 
                               (hsRhsToHsGuardedAlts rhs) decls
            return (newAlt : newRest) 

-- convert a rhs to an alternative
hsRhsToHsGuardedAlts :: HsRhs -> HsGuardedAlts
hsRhsToHsGuardedAlts (HsUnGuardedRhs e)
   = HsUnGuardedAlt e
hsRhsToHsGuardedAlts (HsGuardedRhss rhss)
   = HsGuardedAlts $ map hsGuardedRhsToHsGuardedAlt rhss

hsGuardedRhsToHsGuardedAlt :: HsGuardedRhs -> HsGuardedAlt
hsGuardedRhsToHsGuardedAlt (HsGuardedRhs sloc e1 e2)
   = HsGuardedAlt (markSrcLoc sloc) e1 e2

{- expressions -}
desugarExp :: HsExp -> Desugar HsExp

desugarExp exp@(HsVar v) = return exp 

desugarExp exp@(HsCon c) = return exp

desugarExp exp@(HsLit l) = return exp

{- infix applications become prefix applications -}

desugarExp (HsInfixApp e1 op e2)
   = do newE1 <- desugarExp e1 
        newE2 <- desugarExp e2 
        return $ foldApp [opExp op, mkParen newE1, mkParen newE2]

desugarExp (HsApp e1 e2)
   = liftM2 HsApp (desugarExp e1) (desugarExp e2)

{- negated expressions are desugared
  
   According to the Haskell Language Report: 3.4  Operator Applications

   -e  =  negate (e)
-}
desugarExp (HsNegApp e)
   = liftM (HsApp $ HsVar negateQName) $ desugarExp e

{- \p1 ... pn -> e  = \x1 ... xn -> case (x1, ..., xn) of (p1, ..., pn) -> e 

   optimise for the simple case of (all patterns are variables)

   \v1 .. vn -> e = \v1 .. vn -> e

-}

desugarExp (HsLambda sloc pats e)
   = liftM (HsLambda sloc pats) $ desugarExp e

desugarExp (HsLet decls e)
   = do newDecls <- mapM desugarDecl decls
        newE <- desugarExp e
        return $ HsLet newDecls newE

{- if then else are desugared into case 
   According to the Haskell Language Report: 3.6 Conditionals
     if e1 then e2 else e3  =  case e1 of { True -> e2 ; False -> e3 } 
-}
desugarExp (HsIf discrim trueCase falseCase)
   = do newDiscrim <- desugarExp discrim 
        newTrueCase <- desugarExp trueCase
        newFalseCase <-desugarExp falseCase
        let trueAlt  = HsAlt bogusSrcLoc truePat  (HsUnGuardedAlt newTrueCase)  [] 
            falseAlt = HsAlt bogusSrcLoc falsePat (HsUnGuardedAlt newFalseCase) []
        desugarCase $ HsCase newDiscrim [trueAlt, falseAlt] 

desugarExp exp@(HsCase _e _alts)
   = desugarCase exp

desugarExp (HsDo stmts)
   = desugarDoStmts stmts 

desugarExp (HsTuple exps)
   = liftM HsTuple $ mapM desugarExp exps

desugarExp (HsList exps)
   = liftM HsList $ mapM desugarExp exps

desugarExp (HsParen e)
   = liftM HsParen $ desugarExp e

{- sections

   According to the Haskell Language Report: 3.5  Sections

   The following identities hold:
      (e op) = \ x -> (op) e x   
      (op e) = \ x -> (op) x e
     
  where op is a binary operator, e is an expression, 
  and x is a variable that does not occur free in e.
-}
desugarExp (HsLeftSection e op)
   = do (_, newVarPat, newVarExp) <- freshVar
        newE <- desugarExp e
        return $ mkParen $ foldApp [opExp op, newE]

desugarExp (HsRightSection op e)
   = do (_, newVarPat, newVarExp) <- freshVar
        newE <- desugarExp e
        let newApp = foldApp [opExp op, newVarExp, newE]
        return $ mkParen $ HsLambda madeUpSrcLoc [newVarPat] newApp

{- Enumerations (arithmetic sequences)

   According to the report: 3.10  Arithmetic Sequences

   Translation:
   Arithmetic sequences satisfy these identities:
        [ e1.. ] = enumFrom e1
        [ e1,e2.. ] = enumFromThen e1 e2
        [ e1..e3 ] = enumFromTo e1 e3
        [ e1,e2..e3 ] = enumFromThenTo e1 e2 e3
-}

desugarExp (HsEnumFrom e)
   = do newExp <- desugarExp e
        return $ HsApp (HsVar enumFromQName) (mkParen newExp)

desugarExp (HsEnumFromTo e1 e2)
   = do newExp1 <- desugarExp e1
        newExp2 <- desugarExp e2
        return $ foldApp [HsVar enumFromToQName, newExp1, newExp2]

desugarExp (HsEnumFromThen e1 e2)
   = do newExp1 <- desugarExp e1
        newExp2 <- desugarExp e2
        return $ foldApp [HsVar enumFromThenQName, newExp1, newExp2] 

desugarExp (HsEnumFromThenTo e1 e2 e3)
   = do newExp1 <- desugarExp e1
        newExp2 <- desugarExp e2
        newExp3 <- desugarExp e3
        return $ foldApp [HsVar enumFromThenToQName, newExp1, newExp2, newExp3]

-- for the moment, turn it into a do expression and then desugar that
{-  [ e | s1 ... sn] ---> do { s1 ... sn ; return e } ---> ... -}
desugarExp (HsListComp e stmts)
   = do let lastStmt = HsQualifier $ HsApp (HsVar returnQName) (mkParen e)
            listStmts = map toListStmt stmts
            doExp = HsDo $ listStmts ++ [lastStmt]
        desugarExp doExp
   where
   -- XXX 
   -- this is an inlined version of Monad.guard
   -- it is a hack to avoid having to import Monad (sigh)
   toListStmt :: HsStmt -> HsStmt
   toListStmt (HsQualifier e) = HsQualifier $ HsIf e listUnit emptyList
   toListStmt otherStmt       = otherStmt
   listUnit = HsList [HsCon unit_con_name]
   emptyList = HsList []

desugarExp (HsExpTypeSig sloc e qualType)
   = do newE <- desugarExp e
        synMap <- getSynonymMap
        let newQualType = remSynsFromQualType synMap qualType
        return $ HsExpTypeSig sloc newE newQualType 

desugarExp (HsRecConstr conName fieldUpDates)
   = do selectors <- getSelectors conName
        let selsDefaults = zip selectors $ repeat (HsVar undefinedQName)
        newFieldUpdates <- mapM desugarFieldUpdate fieldUpDates
        return $ mkParen $ foldApp (HsCon conName : pick selsDefaults newFieldUpdates)

desugarExp (HsRecUpdate exp fieldUpdates)
   = do updateCandidates <- getUpdateCandidates fieldUpdates
        newFieldUpdates <- mapM desugarFieldUpdate fieldUpdates
        newAlts <- updateAlts updateCandidates newFieldUpdates 
        return $ mkParen $ HsCase exp newAlts
   where
   updateAlts :: [(HsQName, [HsQName])] -> [HsFieldUpdate] -> Desugar [HsAlt]
   updateAlts [] _updates = return []
   updateAlts ((conName, args):rest) updates
      = do newVars <- sequence $ replicate (length args) freshVar
           let newVarExps = map trd3 newVars
               newVarPats = map snd3 newVars
               thisPat    = HsPApp conName newVarPats 
               thisRhs    = foldApp (HsCon conName : pick (zip args newVarExps) updates)
               thisAlt    = HsAlt bogusSrcLoc thisPat (HsUnGuardedAlt thisRhs) []
           restAlts <- updateAlts rest updates
           return $ thisAlt : restAlts 

desugarExp otherExp 
   = fail $ "desugarExp: this expression is not supported: " ++ show otherExp

desugarFieldUpdate :: HsFieldUpdate -> Desugar HsFieldUpdate
desugarFieldUpdate (HsFieldUpdate qName exp)
   = liftM (HsFieldUpdate qName) $ desugarExp exp

pick :: [(HsQName, HsExp)] -> [HsFieldUpdate] -> [HsExp]
pick [] _ = []
pick ((selector, defaultExp) : rest) updates 
   = selectorValue selector defaultExp updates : pick rest updates
   where
   selectorValue :: HsQName -> HsExp -> [HsFieldUpdate] -> HsExp
   selectorValue _name defaultExp [] = defaultExp
   selectorValue name defaultExp (HsFieldUpdate fName val : rest)
      | dropQualifier name == dropQualifier fName = mkParen val
      | otherwise = selectorValue name defaultExp rest

{- statments -}
desugarStmt :: HsStmt -> Desugar HsStmt
desugarStmt (HsGenerator sloc pat exp)
   = liftM (HsGenerator sloc pat) $ desugarExp exp
desugarStmt (HsQualifier exp)
   = liftM HsQualifier $ desugarExp exp
desugarStmt (HsLetStmt decls)
   = do newDecls <- mapM desugarDecl decls
        return $ HsLetStmt newDecls

-- true if this alternative does not have guards
unGuardedAlt :: HsAlt -> Bool
unGuardedAlt (HsAlt _sloc _pat (HsUnGuardedAlt _e) _wheres) = True
unGuardedAlt _other = False

desugarUnGuardedAlt :: HsAlt -> Desugar HsAlt
desugarUnGuardedAlt alt@(HsAlt sloc pat (HsUnGuardedAlt e) decls)
   = do newE <- desugarExp e
        newDecls <- mapM desugarDecl decls
        -- turn wheres into let bindings if necessary
        let finalExp = if null newDecls 
                          then newE
                          else HsLet newDecls newE
        return $ HsAlt sloc pat (HsUnGuardedAlt finalExp ) [] 
desugarUnGuardedAlt other
   = fail $ "desugarUnGuardedAlt: applied to guarded alternative:" ++ show other 

guardedAltsToCase :: HsExp -> [HsGuardedAlt] -> HsExp
guardedAltsToCase lastExp [HsGuardedAlt sloc e1 e2] 
   = HsCase e1 [trueAlt, falseAlt] 
   where
   trueAlt  = HsAlt sloc truePat (HsUnGuardedAlt e2) []
   falseAlt = HsAlt bogusSrcLoc falsePat (HsUnGuardedAlt lastExp) []
guardedAltsToCase lastExp (HsGuardedAlt sloc e1 e2:alt2:rest)
   = HsCase e1 [trueAlt, falseAlt]
   where
   trueAlt  = HsAlt sloc truePat (HsUnGuardedAlt e2) []
   falseAlt = HsAlt bogusSrcLoc falsePat 
                    (HsUnGuardedAlt $ guardedAltsToCase lastExp (alt2:rest)) []

monadThenQName, monadBindQName :: HsQName 
monadThenQName = UnQual $ HsSymbol ">>" 
monadBindQName = UnQual $ HsSymbol ">>="

desugarDoStmts :: [HsStmt] -> Desugar HsExp
desugarDoStmts [] = fail "desugarDoStmts: empty statements in do notation"
   -- rule 1, do {e} = e
desugarDoStmts [HsQualifier e] = desugarExp e
   -- can't have a generator at the end 
   -- can't have a let at the end 
desugarDoStmts [_lastStmt]
   = fail "last statement in do notation not an expression" 
   -- rule 2, do {e; stmts} = e >> do {stmts}
desugarDoStmts (HsQualifier e : ss@(_:_))
   = do newE <- desugarExp e
        newStmts <- desugarDoStmts ss
        return $ foldApp [HsVar monadThenQName, mkParen newE, mkParen newStmts]
{- rule 3, do {p <- e; stmts} 
           = e >>= (\v -> case v of p -> do {stmts}
                                    _ -> fail "...pattern match error...")

   We can do a little better than this.

   If the pattern 'p' can't fail, (it is irrefutable) 
   then there is no point in generating the failure case

           = e >>= (\p -> do {stmts})

-}
desugarDoStmts (HsGenerator srcLoc pat e : ss@(_:_))
   = do newE      <- desugarExp e
        newStmts  <- desugarDoStmts ss
        (_, newVarPat, newVarExp) <- freshVar
        case isIrrefPat pat of
           True  -> do let lambdaExp =  mkParen $ HsLambda bogusSrcLoc [pat] newStmts
                       return $ foldApp [HsVar monadBindQName, mkParen newE, lambdaExp]
           False -> do let fileNameStr = srcFilename srcLoc 
                           coordStr    = show $ (srcLine srcLoc, srcColumn srcLoc)
                           srcLocStr = fileNameStr ++ " " ++ coordStr
                           failString = HsLit $ HsString $ "pattern match error: " ++ srcLocStr 
                           failExp = foldApp [HsVar failQName, failString] 
                           altSucceed = HsAlt bogusSrcLoc pat (HsUnGuardedAlt newStmts) []
                           altFail    = HsAlt bogusSrcLoc HsPWildCard (HsUnGuardedAlt failExp) []
                           caseExp = HsCase newVarExp [altSucceed, altFail]
                           lambdaExp = mkParen $ HsLambda bogusSrcLoc [newVarPat] caseExp
                       return $ foldApp [HsVar monadBindQName, mkParen newE, lambdaExp]

   -- rule 4
desugarDoStmts (HsLetStmt decls : ss@(_:_))
   = do newDecls <- mapM desugarDecl decls
        newStmts <- desugarDoStmts ss
        return $ HsLet newDecls newStmts 

--------------------------------------------------------------------------------

-- a monad to help support the desugaring

data State =
   State 
   { state_varCount   :: Int         -- to make each new variable unique
   , state_synonymMap :: SynonymMap  -- map of all type synonyms in scope
   , state_recordMap  :: RecordMap   -- map of record constructors and their selectors
   }

newtype Desugar a = Desugar (State -> Either String (a, State))

-- instantiate the Monad class so we can use do notation
instance Monad Desugar where
    return a
        = Desugar (\state -> Right (a, state))

    Desugar comp >>= fun
        = Desugar (\state ->
                        case comp state of
                           Left s -> Left s
                           Right (result, newState) 
                              -> case fun result of
                                    Desugar comp' -> comp' newState)

    fail s = Desugar (\_ -> Left s)

-- run a transformation
runDesugar :: Int -> SynonymMap -> RecordMap -> Desugar a -> Either String (a, Int)
runDesugar count synMap recMap (Desugar comp)
   = case comp initState of
        Left s -> Left s
        Right (result, newState)
           -> Right (result, state_varCount newState)
   where
   initState = State { state_varCount   = count
                     , state_synonymMap = synMap
                     , state_recordMap  = recMap }

-- select a component of the state
select :: (State -> a) -> Desugar a
select selector = Desugar (\state -> Right (selector state, state))

getVarCount :: Desugar Int
getVarCount = select state_varCount

getSynonymMap :: Desugar SynonymMap
getSynonymMap = select state_synonymMap

getRecordMap :: Desugar RecordMap
getRecordMap = select state_recordMap

getSelectors :: HsQName -> Desugar [HsQName]
getSelectors conName
   = do recMap <- getRecordMap
        case lookupRecord recMap conName of
           Nothing -> return []
           Just selectors -> return selectors

getUpdateCandidates :: [HsFieldUpdate] -> Desugar [(HsQName, [HsQName])]
getUpdateCandidates updates
   = do recMap <- getRecordMap
        let canonicalUpdates = canonify updates
        let candidateList = findUpdateCandidates recMap canonicalUpdates
        if null candidateList
           then fail $ "no records have all these fields: " ++ (show $ map updateName updates)
           else return candidateList
   where
   -- make sure the field selectors are either all qualified, or all
   -- unqualified. If there is a mix then make them all unqualified
   canonify :: [HsFieldUpdate] -> [HsFieldUpdate]  
   canonify updates
      | any isUnQualifiedUpdate updates = map unQualifyUpdate updates
      | otherwise = updates
   isUnQualifiedUpdate :: HsFieldUpdate -> Bool
   isUnQualifiedUpdate (HsFieldUpdate qName _val) = isQualified qName
   unQualifyUpdate :: HsFieldUpdate -> HsFieldUpdate
   unQualifyUpdate (HsFieldUpdate qName val) = HsFieldUpdate (dropQualifier qName) val

incVarCount :: Desugar ()
incVarCount = Desugar (\state ->
                          let oldVarCount = state_varCount state
                          in Right ((), state {state_varCount = oldVarCount + 1}))

-- compute a fresh variable
freshVar :: Desugar (HsName, HsPat, HsExp)
freshVar
   = do currentCount <- getVarCount
        incVarCount
        let newVarStr  = defaultVarPrefix ++ show currentCount
            newVarName = HsIdent newVarStr
        return (newVarName, HsPVar newVarName, HsVar $ UnQual newVarName)


--------------------------------------------------------------------------------

-- determine whether a pattern is irrefutable

{- from section 3.17.2  Informal Semantics of Pattern Matching

   The irrefutable patterns are as follows: a variable, a wildcard, 
   N apat where N is a constructor defined by newtype and apat is irrefutable 
   (see Section 4.2.3), var@apat where apat is irrefutable, or of the form 
   ~apat (whether or not apat is irrefutable). All other patterns are refutable.

   we don't get all of these because we don't know whether a constructor
   is from a newtype or not, though it doesn't matter too much for us.
-}


isIrrefPat :: HsPat -> Bool
isIrrefPat (HsPVar _name)   = True
isIrrefPat (HsPIrrPat _pat) = True
isIrrefPat HsPWildCard      = True
isIrrefPat (HsPParen pat)   = isIrrefPat pat
isIrrefPat (HsPAsPat _name pat) = isIrrefPat pat
isIrrefPat _otherPat        = False

--------------------------------------------------------------------------------

{- desugaring case statments -}

-- convert an record patterns to ordinary contructor applications
-- K { v1 = p1 , v2 = p2 } ---> K p1 _ p2 
desugarCase (HsCase discrim alts)
   = do newAlts <- mapM desugarRecPatInAlt alts
        desugarCaseWorker $ HsCase discrim newAlts 
   where
   desugarRecPatInAlt :: HsAlt -> Desugar HsAlt
   desugarRecPatInAlt (HsAlt sloc pat alts wheres)
      = do newPat <- desugarRecPat pat
           return $ HsAlt sloc newPat alts wheres 

{- we don't desugar a case if:
    1) none of the alternatives are guarded, and
    2) none of the patterns in the alternatives have literals
    3) none of the patterns is a record pattern 

   instead we just recursively desugar the branches of the alternatives
-}
desugarCaseWorker (HsCase discrim alts)
   | all (unGuardedAlt .&&. noLitPatAlt) alts
        = do newAlts <- mapM desugarUnGuardedAlt alts
             newDiscrim <- desugarExp discrim
             return $ HsCase newDiscrim newAlts
   where
   noLitPatAlt :: HsAlt -> Bool
   noLitPatAlt (HsAlt _sloc pat _alts _exp) = not $ patContainsLit pat   

{-
   discriminant must be a variable, and also the rhs of the
   second alternative

   case v1 of
      p -> rhs1
      _ -> v2 

   case v of { _ -> e; _ -> e' } = e
  
   case v1 of { v2 -> e1; _ -> e2 } = case v1 of { v2 -> e1 }

   case v of { k -> e; _ -> e' } = if (v==k) then e else e'

   case v of { -p -> e; _ -> e' } = if (v == -k) then e else e' 

   ...
-}

desugarCaseWorker (HsCase var@(HsVar v1) 
                   [HsAlt sloc1 
                          pat  
                          (HsUnGuardedAlt rhs1)
                          wheres1,
                    HsAlt sloc2
                          HsPWildCard 
                          (HsUnGuardedAlt rhs2@(HsVar _))
                          wheres2
                   ] 
            )
   = do let rhs1MaybeWithWheres 
               = case null wheres1 of
                    True -> rhs1 
                    False -> HsLet wheres1 rhs1
            rhs2MaybeWithWheres
               = case null wheres2 of
                    True -> rhs2
                    False -> HsLet wheres2 rhs2
        case pat of
           HsPWildCard -> desugarExp rhs1MaybeWithWheres 

           HsPVar v2   -> do newRhs1 <- desugarExp rhs1MaybeWithWheres 
                             return $ HsCase var [HsAlt sloc1 pat (HsUnGuardedAlt newRhs1) []]  
           
           HsPLit lit -> do let eqExp = HsInfixApp var equalsOp (HsLit lit)
                            desugarExp $ HsIf eqExp rhs1MaybeWithWheres rhs2MaybeWithWheres
           
           HsPNeg (HsPLit lit) 
                      -> do let eqExp = HsInfixApp var equalsOp (HsNegApp (HsLit lit))
                            desugarExp $ HsIf eqExp rhs1MaybeWithWheres rhs2MaybeWithWheres

           HsPNeg otherPat -> fail $ "illegal negative pattern: " ++ show pat 

           otherPat -> do (newPat, subs) <- subLiteralsInPat otherPat
                          case null subs of
                             -- none of the patterns needed expanding (no literals nested inside)
                             True -> do newRhs1 <- desugarExp rhs1MaybeWithWheres
                                        newRhs2 <- desugarExp rhs2  
                                        return $ HsCase var [HsAlt sloc1 pat (HsUnGuardedAlt newRhs1) [],
                                                             HsAlt sloc2 HsPWildCard (HsUnGuardedAlt newRhs2) []]
                             -- at least one of the patterns needs expanding
                             False -> do expandedCase <- desugarExp $ expandCase subs rhs1MaybeWithWheres 
                                                                                      rhs2MaybeWithWheres
                                         newRhs2 <- desugarExp rhs2
                                         return $ HsCase var [HsAlt sloc1 newPat (HsUnGuardedAlt expandedCase) [],
                                                              HsAlt sloc2 HsPWildCard (HsUnGuardedAlt newRhs2) []]


{- make the rhs of the second alternative a variable
   by hoisting the rhs upwards - this avoids much duplicated code
  
   case v of 
      p1 match1
      _  -> rhs2

   = case rhs2 of
        y -> case v of
                p1 match1 
                _ -> y 

   Or maybe better:

   = let y = rhs2 
     in case v of
           p1 match1
           _ -> rhs2
-}

{-
desugarCaseWorker (HsCase var@(HsVar v1) 
                   [HsAlt sloc1 
                          pat  
                          rhs1
                          wheres,
                    HsAlt sloc2
                          HsPWildCard 
                          (HsUnGuardedAlt rhs2) -- the second rhs is not a variable, but it is unguarded
                          []                    -- the second alt has no where decls XXX is this necessary?
                   ] 
            )
   = do (_, newVarPat, newVarExp) <- freshVar
        let newRhs = case rhs1 of
                        HsUnGuardedAlt _exp -> rhs1
                        HsGuardedAlts alts -> HsUnGuardedAlt $ desugarGuardedAlts newVarExp alts
        let newCase = HsCase var [HsAlt sloc1 pat newRhs wheres, 
                                  HsAlt sloc2 HsPWildCard (HsUnGuardedAlt newVarExp) []]
            newAlt = HsAlt bogusSrcLoc newVarPat (HsUnGuardedAlt newCase) [] 
        desugarCaseWorker $ HsCase rhs2 [newAlt] 
-}

desugarCaseWorker (HsCase var@(HsVar v1) 
                   [HsAlt sloc1 
                          pat  
                          rhs1
                          wheres1,
                    HsAlt sloc2
                          HsPWildCard 
                          (HsUnGuardedAlt rhs2) -- the second rhs is not a variable, but it is unguarded
                          wheres2                
                   ] 
            )
   = do (_, newVarPat, newVarExp) <- freshVar
        let newRhs = case rhs1 of
                        HsUnGuardedAlt _exp -> rhs1
                        -- HsGuardedAlts alts -> HsUnGuardedAlt $ desugarGuardedAlts newVarExp alts
                        HsGuardedAlts alts -> HsUnGuardedAlt $ guardedAltsToCase newVarExp alts
        let newCase = HsCase var [HsAlt sloc1 pat newRhs wheres1, 
                                  HsAlt sloc2 HsPWildCard (HsUnGuardedAlt newVarExp) []]
        newRhs2 <- desugarExp rhs2
        newWheres2 <- mapM desugarDecl wheres2
        newCaseNoSugar <- desugarExp newCase
        let newBind = HsPatBind trustedSrcLoc newVarPat (HsUnGuardedRhs newRhs2) newWheres2
        return $ HsLet [newBind] newCaseNoSugar 

{- 

   make all cases binary (exactly two choices)

   case v of
      p1 alts1
      p2 alts2
      ...
      pn altsn

   = case v of
        p1 alts1
        _  -> case v of
                 p2 alts2
                 _ -> ... case v of
                             pn altsn
                             _  -> error "No match"
-}

desugarCaseWorker (HsCase var@(HsVar v1) [alt])
   = desugarCase $ HsCase var [alt, noMatchAlt]

{-
desugarCaseWorker (HsCase var@(HsVar v1) (alt1:alt2:alts))
   = do restCases <- desugarCaseWorker (HsCase var (alt2:alts))
        let secondAlt = HsAlt bogusSrcLoc HsPWildCard (HsUnGuardedAlt restCases) []
        desugarCaseWorker $ HsCase var [alt1, secondAlt]  
-}
desugarCaseWorker (HsCase var@(HsVar v1) (alt1:alt2:alts))
   = desugarCase $ binaryAlts var (alt1:alt2:alts) 
   where
   binaryAlts :: HsExp -> [HsAlt] -> HsExp
   binaryAlts var [alt]
      = HsCase var [alt, noMatchAlt]
   binaryAlts var (alt1:alt2:alts)
      = HsCase var [alt1, secondAlt]
      where
      restCases = binaryAlts var (alt2:alts)
      secondAlt = HsAlt bogusSrcLoc HsPWildCard (HsUnGuardedAlt restCases) []

{- 
   -- XXX is the let binding wise here?

   make sure the discriminant is a variable
   haskell report says:
      case e of alts == (\v -> case v of alts) e

   maybe better to avoid the lambda:
      case e of alts == let v = e in case v of alts
-}

desugarCaseWorker (HsCase nonVarExp alts)
   = do newExp <- desugarExp nonVarExp
        (_, newVarPat, newVarExp) <- freshVar
        newCase <- desugarCase $ HsCase newVarExp alts
{-
        return $ foldApp [mkParen $ HsLambda bogusSrcLoc [newVarPat] newCase, 
                          mkParen newExp]
-}
        let newDecl = HsPatBind trustedSrcLoc newVarPat (HsUnGuardedRhs newExp) []
        return $ HsLet [newDecl] newCase 

desugarCaseWorker otherCase
   = fatalError __FILE__ __LINE__ $ 
       "desugarCaseWorker: unknown case expression: " ++ show otherCase

-- the generic alternative if no cases match
noMatchAlt :: HsAlt
noMatchAlt = HsAlt bogusSrcLoc HsPWildCard 
                   (HsUnGuardedAlt $ HsApp (HsVar errorQName)
                                           (HsLit $ HsString "No Match")) []

-- subsitute all literals in a pattern with new variables
-- and return a the new pattern and a list of the substitutions that
-- were made
subLiteralsInPat :: HsPat -> Desugar (HsPat, [(HsName, HsPat)])
subLiteralsInPat p@(HsPVar v) = return (p, [])
subLiteralsInPat p@(HsPLit lit) 
   = do (newVarName, newVarPat, _) <- freshVar
        return (newVarPat, [(newVarName, p)])
-- I think you can only negate a numeric literal
subLiteralsInPat p@(HsPNeg (HsPLit numLit))
   = do (newVarName, newVarPat, _) <- freshVar
        return (newVarPat, [(newVarName, p)])
subLiteralsInPat (HsPInfixApp p1 op p2)
   = do (newPat1, subs1) <- subLiteralsInPat p1
        (newPat2, subs2) <- subLiteralsInPat p2
        return (HsPInfixApp newPat1 op newPat2, subs1 ++ subs2)
subLiteralsInPat (HsPApp con pats)
   = do (newPats, newSubs) <- mapAndUnzipM subLiteralsInPat pats
        return (HsPApp con newPats, concat newSubs)
subLiteralsInPat (HsPTuple pats)
   = do (newPats, newSubs) <- mapAndUnzipM subLiteralsInPat pats
        return (HsPTuple newPats, concat newSubs)
subLiteralsInPat (HsPList pats)
   = do (newPats, newSubs) <- mapAndUnzipM subLiteralsInPat pats
        return (HsPList newPats, concat newSubs)
subLiteralsInPat (HsPParen pat)
   = do (newPat, subs) <- subLiteralsInPat pat
        return (HsPParen newPat, subs)
subLiteralsInPat pat@(HsPRec _qname _fields)
   = notSupported __FILE__ __LINE__ $ "record patterns: " ++ show pat
subLiteralsInPat (HsPAsPat name pat)
   = do (newPat, subs) <- subLiteralsInPat pat
        return (HsPAsPat name newPat, subs) 
subLiteralsInPat HsPWildCard = return (HsPWildCard, []) 
subLiteralsInPat (HsPIrrPat pat)
   = do (newPat, subs) <- subLiteralsInPat pat
        return (HsPIrrPat newPat, subs) 
                                    
expandCase :: [(HsName, HsPat)] -> HsExp -> HsExp -> HsExp
expandCase [(name,pat)] rhs1 rhs2
   = HsCase (HsVar $ UnQual name) [alt1, alt2]
   where
   alt1 = HsAlt bogusSrcLoc
                pat (HsUnGuardedAlt rhs1) []
   alt2 = HsAlt bogusSrcLoc
                HsPWildCard (HsUnGuardedAlt rhs2) []
expandCase ((name,pat):rest) rhs1 rhs2
   = HsCase (HsVar $ UnQual name) [alt1, alt2]
   where
   alt1 = HsAlt bogusSrcLoc
                pat (HsUnGuardedAlt (expandCase rest rhs1 rhs2)) []
   alt2 = HsAlt bogusSrcLoc
                HsPWildCard (HsUnGuardedAlt rhs2) []

-- turn all record patterns into normal patterns
desugarRecPat :: HsPat -> Desugar HsPat
desugarRecPat pat@(HsPVar _name) = return pat
desugarRecPat pat@(HsPLit _lit) = return pat
desugarRecPat (HsPNeg pat) = liftM HsPNeg $ desugarRecPat pat
desugarRecPat (HsPInfixApp pat1 qName pat2) 
   = do newPat1 <- desugarRecPat pat1
        newPat2 <- desugarRecPat pat2
        return $ HsPInfixApp newPat1 qName newPat2 
desugarRecPat (HsPApp qName pats)
   = liftM (HsPApp qName) $ mapM desugarRecPat pats
desugarRecPat (HsPTuple pats)
   = liftM HsPTuple $ mapM desugarRecPat pats
desugarRecPat (HsPList pats) 
   = liftM HsPList $ mapM desugarRecPat pats
desugarRecPat (HsPParen pat) = liftM HsPParen $ desugarRecPat pat
desugarRecPat (HsPRec conName patFields)
   = do selectors <- getSelectors conName
        return $ HsPApp conName $ patArgs selectors patFields
   where
   patArgs :: [HsQName] -> [HsPatField] -> [HsPat]
   patArgs [] _fields = []
   patArgs (name : rest) fields 
      = case findSelector name fields of
           Nothing -> HsPWildCard : patArgs rest fields
           Just p  -> mkParenPat p : patArgs rest fields
   findSelector :: HsQName -> [HsPatField] -> Maybe HsPat
   findSelector selName [] = Nothing
   findSelector selName1 (HsPFieldPat selName2 pat : rest)
      | selName1 == selName2 = Just pat
      | otherwise = findSelector selName1 rest 

desugarRecPat (HsPAsPat name pat)
   = liftM (HsPAsPat name) $ desugarRecPat pat 
desugarRecPat HsPWildCard = return HsPWildCard
desugarRecPat (HsPIrrPat pat) = liftM HsPIrrPat $ desugarRecPat pat

--------------------------------------------------------------------------------

-- marking srcLocs

markSrcLoc :: SrcLoc -> SrcLoc
markSrcLoc sloc
   = sloc { srcFilename = []
          , srcColumn = -1
          }

isMarkedSrcLoc :: SrcLoc -> Bool
isMarkedSrcLoc sloc
   = srcFilename sloc == [] &&
     srcColumn sloc   == -1 &&
     srcLine sloc > 0
