{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 FileIO 

        Description:            Code for reading/writing files and directories

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}


module FileIO 
   ( createFileOverWrite
   , tryOpenFile 
   , directorySep  
   , fileExistsAndIsReadable
   , copyFile
   , baseName
   )
   where

import IO

import Directory

import Error                    
   ( abortWithError 
   , ErrorCode (..)
   )

import ChatIO                   
   ( ifChat 
   , ChatLevel (..) 
   )

import BuddhaName               
   ( buddhaNameVersion )

import System
   ( system 
   , ExitCode (..)
   ) 

--------------------------------------------------------------------------------

-- the symbol used as directory seperator
-- XXX this should really be obtained when buddha is configured
-- this probably won't work on windows for example
directorySep :: String
directorySep = "/"

directorySepChar :: Char 
directorySepChar = '/'

-- tries to create a file in a given directory
-- if the file is already there then it overwrites it
-- if the directory does not exist it creates it first 

createFileOverWrite :: FilePath -> FilePath -> IO Handle
createFileOverWrite directory filename
   = do dirExists <- doesDirectoryExist directory
        case dirExists of
           -- dir does not exist so make it
           False -> do res <- try $ createDirectory directory
                       case res of
                          -- couldn't make the dir, so bomb
                          Left error -> do let errorStr = ioeGetErrorString error
                                           abortWithError FileIOError $ dirCreateError directory errorStr
                          -- okay really open the file
                          Right ()   -> tryOpenFile WriteMode  [directory ++ directorySep ++ filename]
           True  -> do perms <- getPermissions directory
                       case writable perms of
                          True  -> tryOpenFile WriteMode [directory ++ directorySep ++ filename]
                          -- dir exists but not writeable, bomb
                          False -> abortWithError FileIOError $ 
                                        dirCreateError directory "not writeable" 

-- read contents of a file, accepts a list of paths for the file
-- the first successful open handle is returned, if all paths fail
-- and error is returned

tryOpenFile :: IOMode -> [FilePath] -> IO Handle 
tryOpenFile mode paths 
   = tryOpenFile' mode paths []
   where
   tryOpenFile' :: IOMode -> [FilePath] -> [(FilePath,String)] -> IO Handle 
   tryOpenFile' _mode [] errors = abortWithError FileIOError $ fileOpenError errors 
   tryOpenFile' mode (filepath:paths) errors
      = do ifChat Verbose $ hPutStr stdout $ buddhaNameVersion ++ 
                                      ": trying to open file: " ++ filepath 
           res <- try $ openFile filepath mode
           case res of
              Left error   -> do ifChat Verbose $ hPutStrLn stdout " ... failed" 
                                 let errorStr = ioeGetErrorString error
                                 tryOpenFile' mode paths (errors ++ [(filepath,errorStr)])
              Right handle -> do ifChat Verbose $ hPutStrLn stdout " ... succeeded"
                                 return handle

fileOpenError :: [(FilePath,String)] -> String
fileOpenError es
   = msg ++ (unlines $ map (\(f,s) -> show f ++ " " ++ s) es)
   where
   msg = "attempt to open file failed because:\n"

-- an error message when a directory can't be created 
dirCreateError :: FilePath -> String -> String
dirCreateError dir errStr
   = "attempt to create directory : " ++ dir ++
     " failed because: " ++ errStr

fileExistsAndIsReadable :: FilePath -> IO Bool
fileExistsAndIsReadable f 
   = do exists <- doesFileExist f
        if exists 
           then do perms <- getPermissions f 
                   if readable perms 
                      then return True
                      else return False
           else return False
            
--------------------------------------------------------------------------------

-- copy a file, returns the exit status of the copy command 

copyCmd :: String
copyCmd = "cp"

copyFile :: FilePath -> FilePath -> IO ExitCode
copyFile from to
   = do let cmd = copyCmd ++ " " ++ from ++ " " ++ to
        ifChat Verbose $ putStrLn $ buddhaNameVersion ++ ": executing: " ++ cmd
        system cmd

--------------------------------------------------------------------------------

-- find the basename of a file in a filepath

baseName :: FilePath -> String
baseName filePath
   = reverse $ takeWhile (/= directorySepChar) (reverse filePath)
