{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Ident 

        Description:            Identifier information:
                                - arity
                                - binding type 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Ident 
   ( Arity
   , IdentQual (..)
   , IdentBind (..)
   , IdentMap
   , IdentInfo (..)
   , emptyIdentMap
   , lookupIdent
   , plusIdentMap
   , identMapToList
   , constructorIdents
   , joinIdentMap
   , showIdentMap
   , listToIdentMap
   , declIdents
   , isLetBound
   , isPatBound
   )
   where

import Language.Haskell.Syntax                  

import SyntaxUtils              
   ( numArgsInQualType
   , varsFromPat
   , isQualifiedType 
   )

import IfaceUtils               
   ( insertMap
   , addListItemsToMap 
   , IdentQual (..)
   )

import Data.FiniteMap 

import Error                    
   ( fatalError )

import TypeSigMap               
   ( TypeSigMap
   , lookupTypeSig 
   )

--------------------------------------------------------------------------------

-- the binding occurrence of an identifier
data IdentBind  
   = Pat        -- bound in a pattern (aka lambda bound)
   | LetFun     -- function (one or more arguments to the left of '=')
   | LetConst   -- let bound constant, not overloaded
   | LetConstOL -- overloaded, let bound constant 
   | Con        -- data constructor
   deriving (Eq, Ord, Show, Read)

isLetBound :: IdentBind -> Bool
isLetBound x = x `elem` [LetFun, LetConst, LetConstOL]

isPatBound :: IdentBind -> Bool
isPatBound Pat = True
isPatBound _other = False

type Arity      = Int

  -- all the information for a given identifier
data IdentInfo 
   = IdentInfo 
     { identInfo_bindType :: IdentBind
     , identInfo_arity    :: Arity 
     }                    
     deriving Show

  -- map from name to ident info
type IdentMap = FiniteMap HsQName IdentInfo

  -- pretty crude printing of ident map
showIdentMap :: IdentMap -> String
showIdentMap identMap 
   = unlines $ map show mapList 
   where
   mapList = identMapToList identMap 

emptyIdentMap :: IdentMap
emptyIdentMap = emptyFM

identMapToList :: IdentMap -> [(HsQName, IdentInfo)]
identMapToList = fmToList

listToIdentMap ::  [(HsQName, IdentInfo)] -> IdentMap
listToIdentMap = listToFM

lookupIdent :: IdentMap -> HsQName -> Maybe IdentInfo 
lookupIdent identMap name
   = case name of
        -- hardwired values for (), tuples and cons
        Special specCon 
           -> case specCon of
                 HsUnitCon    -> Just (IdentInfo { identInfo_bindType = Con
                                                 , identInfo_arity = 0 })
                 HsTupleCon i -> Just (IdentInfo { identInfo_bindType = Con
                                                 , identInfo_arity = i })
                 HsCons       -> Just (IdentInfo { identInfo_bindType = Con
                                                 , identInfo_arity = 2 })
                 other -> fatalError __FILE__ __LINE__ $ 
                             "lookupIdent: unknown special symbol: " ++ show name
        other -> lookupFM identMap name 

plusIdentMap :: IdentMap -> IdentMap -> IdentMap
plusIdentMap = plusFM

--------------------------------------------------------------------------------

-- collect the constructor arities from a list of decls
-- there should never be any duplicates, so plusFM is sufficient
constructorIdents :: (HsName -> HsQName) -> [HsDecl] -> IdentMap 
constructorIdents _qualifier [] = emptyIdentMap
constructorIdents qualifier (decl@(HsDataDecl _sloc _cntxt _name _args condecls _derives):decls)
   = conDeclIdents qualifier condecls `plusFM` constructorIdents qualifier decls
constructorIdents qualifier ((HsNewTypeDecl _sloc _cntxt _name _args condecl _derives):decls)
   = conDeclIdents qualifier [condecl] `plusFM` constructorIdents qualifier decls
constructorIdents qualifier (_anythingElse:decls)
   = constructorIdents qualifier decls

-- collect the constructor arities from a list of constructor decls
conDeclIdents :: (HsName -> HsQName) -> [HsConDecl] -> IdentMap

conDeclIdents _qualifier [] = emptyIdentMap

conDeclIdents qualifier ((HsConDecl _srcLoc name args) : rest)
   = insertMap QualAndUnQual restIdents (qualifier name) info
   where
   restIdents = conDeclIdents qualifier rest
   info = IdentInfo {identInfo_arity = length args, identInfo_bindType = Con} 

conDeclIdents qualifier ((HsRecDecl _srcLoc name args) : rest)
   = insertMap QualAndUnQual restIdents (qualifier name) info 
   where
   restIdents = conDeclIdents qualifier rest
   recArgs :: ([HsName],HsBangType) -> Int 
   recArgs (names, _types) = length names
   info = IdentInfo {identInfo_arity    = sum $ map recArgs args,
                     identInfo_bindType = Con}

-- collect the variable arities from a list of variable decls
declIdents :: Bool -> 
              (HsName -> HsQName) 
              -> IdentQual  
              -> TypeSigMap
              -> [HsDecl] 
              -> IdentMap

declIdents _isTopLevel _qualifier _idQual _typeSigs [] = emptyIdentMap

-- a bit tedious due to non-simple patterns that can bind 
-- a number of variables at once 
-- as in: (x, y, z) = rhs
declIdents isTopLevel qualifier idQual typeSigs ((HsPatBind _srcLoc pat _rhs _wheres) : decls) 
   = addListItemsToMap idQual patInfos restIdents
   where
   restIdents = declIdents isTopLevel qualifier idQual typeSigs decls 
   patInfos :: [(HsQName, IdentInfo)]
   patInfos = [(qualifier v, info v) | v <- varsFromPat pat]
   bindStyle name = constBindStyle isTopLevel typeSigs name
   arity = case pat of
              HsPVar varName 
                 -> case lookupTypeSig typeSigs varName of
                       Nothing -> 0
                       Just qType -> numArgsInQualType qType
              otherPar -> 0
   info v = if arity > 0 
              then IdentInfo { identInfo_arity = arity, identInfo_bindType = LetFun }
              else IdentInfo { identInfo_arity = arity, identInfo_bindType = bindStyle v }

declIdents isTopLevel qualifier idQual typeSigs ((HsFunBind matches@(match:_)) : decls)
   = matchIdent match $ declIdents isTopLevel qualifier idQual typeSigs decls
   where
   matchIdent :: HsMatch -> IdentMap -> IdentMap
   matchIdent (HsMatch _sloc name pats _rhs _wheres) identMap
      = insertMap idQual identMap (qualifier name) info
      where
      info = IdentInfo {identInfo_arity = arity, identInfo_bindType = LetFun}
      arity = case lookupTypeSig typeSigs name of
                 Nothing -> length pats
                 Just qType -> numArgsInQualType qType 

declIdents isTopLevel qualifier idQual typeSigs 
           ((HsClassDecl _sloc _cntxt _name _args declsClass) : declsRest)
   = joinIdentMap classIdents restIdents 
   where
   classIdents = classMemberIdents qualifier declsClass 
   restIdents = declIdents isTopLevel qualifier idQual typeSigs declsRest

declIdents isTopLevel qualifier idQual typeSigs (_anythingElse:decls)
   = declIdents isTopLevel qualifier idQual typeSigs decls

{- class idents

   we compute the arity of class members from the number of outermost
   arrows in the type scheme

   the reason is that different implementations of the member functions
   may have differing numbers of patterns, so we presume that the
   arity comes from the type.
  
   when we transform instance declarations we may need to add or subtract
   patterns when they don't match the type arity

   we expect the list of decls to be a list of type signatures 
   and default declarations

   f :: a -> a      (arity 1)
   k :: t           (arity 0)
   g :: (Int -> Int) -> (a,a) -> Bool   (arity 2)
-}

classMemberIdents :: (HsName -> HsQName) -> [HsDecl] -> IdentMap

classMemberIdents _qualifier [] = emptyIdentMap

classMemberIdents qualifier ((HsTypeSig _sloc names qType) : decls)
   = addListItemsToMap QualAndUnQual infoList restIdents 
   where
   arity = numArgsInQualType qType
   restIdents = classMemberIdents qualifier decls
   infoList = [(qualifier name, IdentInfo bindStyle arity) | name <- names]
   bindStyle = if arity == 0 then LetConstOL else LetFun

classMemberIdents qualifier (_otherDecl : decls)
   = classMemberIdents qualifier decls

--------------------------------------------------------------------------------

-- joining two arity maps together
-- doesn't check for duplicates, probably should
-- XXX most definitely should - please fix me 
joinIdentMap :: IdentMap -> IdentMap -> IdentMap 
joinIdentMap map1 map2 = plusFM map1 map2

--------------------------------------------------------------------------------

-- in the absence of a type signature if a let-bound constant
-- is defined in a nested scope (not at the top level of a module)
-- then we assume pessimistically that it is overloaded

constBindStyle :: Bool -> TypeSigMap -> HsName -> IdentBind
constBindStyle isTopLevel sigMap name 
   = case lookupTypeSig sigMap name of
        Nothing  -> if isTopLevel then LetConst else LetConstOL 
        Just sig -> case isQualifiedType sig of
                       True  -> LetConstOL
                       False -> LetConst
