{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Iface 

        Description:            Reading and Writing Interface files in 
                                Haskell. 

                                The neat idea is to just make the 
                                files Haskell modules, that way we only
                                need one parser. The only small trick is
                                converting some parse results back into
                                values (reflection!).

        Primary Authors:        Bernie Pope

        Notes:                  I wouldn't say this was 100% perfect at the moment
                                but the cases that don't work are probably of the
                                more obscure kind.

Supporting Haskell modules is tricky. There are three sources of 
difficulty:

    1. Items that are imported or exported can be abbreviated:

       eg.  import Foo (Bool (..))

       Here I must know which constructors of Bool that Foo exports.
       Foo may export all of them, some of them or none of them.

       To know this, each module interface must describe an association
       between type constructors and exported data constructors, and
       type classes and exported members.
 
       We record this information in the interface file for each module:
       For example, presuming that Foo exports Bool(True), then the interface
       file for Foo will contain:

         ... (Bool, [True]) ...

       Furthermore it is legal for a module to export items seperately:

         module Foo (Bool(True), Bool(False))

         .... and in another module ....

         import Foo (Bool(True))
         import Foo (Bool(False))

   2.  A module may export items that are imported from other modules:

          module Roo (fred, jim) where
          import Bill (fred)
          jim = ...

       In fact you can export a whole module:

          module Roo (module Bill) where

       And you can even export your own self:

          module Roo (module Roo) where, which is equivalent to "module Roo"

       This flexibility in the module system means that you have to keep track
       of a large amount of information, to process the modules correctly.

   3.  Items can be hidden from imports:

       import Foo hiding (bar, Ram, Fred(A,B))

       This is a bit painful because it is not exactly the opposite of
       an import (without hiding). It is especially weird because although
       Fred mentions A an B this hides everything from Fred (including the
       type constructor, and other possible data constructors not mentioned
       here.

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Iface 
   ( processImports
   , assocFromDecls
   , plusAssocMap
   , showAssocMap
   , IfaceCache
   , emptyIfaceCache
   , ImportMap
   , emptyImportMap
   , interfaceDirectory
   , showImportMap
   , ModIface (..)
   , mkModIface
   , writeModIface
   , exportSelectors
   , importSelectors
   , AssocMap 
   ) where

import Language.Haskell.Syntax  

import Ident                    
   ( IdentMap
   , showIdentMap
   , plusIdentMap 
   , IdentBind (..)
   , IdentInfo (..)
   , Arity
   , identMapToList
   , lookupIdent
   , emptyIdentMap
   )

import FileIO                   
   ( createFileOverWrite
   , tryOpenFile
   , directorySep
   )

import IO

import Directory

import qualified PPrint 

import qualified Language.Haskell.Pretty as Pretty

import Data.FiniteMap

import ParserUtils              
   ( parseHsSource )

import SyntaxUtils              
   ( maybeIsQualified
   , unQualify
   , reQualify
   , thisModQualified 
   , fromHsQNameQual
   )

import Utils                    
   ( notMembers )

import IfaceUtils               
   ( addListItemsToMap
   , addListItemsToMap_C
   , insertMap
   , IdentQual (..)
   )

import Error                    
   ( abortWithError
   , fatalError
   , ErrorCode (..)
   )

import List                     
   ( nub
   , union
   )

import Infix                    
   ( InfixMap 
   , plusInfixMap
   , Fixity (..)
   , infixMapToList
   , emptyInfixMap
   , showInfixMap
   )

import Synonym                  
   ( SynonymMap 
   , showSynonymMap
   , plusSynonymMap
   , Synonym (..) 
   , synonymMapToList 
   , emptySynonymMap 
   , lookupSynonymMap
   , synonymMapToDecls
   , declsToSynonymMap 
   )

import Record                   
   ( RecordMap
   , showRecordMap
   , plusRecordMap
   , recordMapToList
   , emptyRecordMap
   , addListItemsToRecMap
   , lookupRecord 
   )

import Monad                    
   ( mapAndUnzipM )

import Char                     
   ( isUpper )

import Data.Set                 
   ( Set 
   , elementOf 
   )

--------------------------------------------------------------------------------

-- the representation of an interface to a module
data ModIface 
   = ModIface 
     { modIface_idents     :: IdentMap    -- exported identifiers 
     , modIface_assocs     :: AssocMap    -- type and class associations
     , modIface_infixMap   :: InfixMap    -- infix rules for exported idents
     , modIface_synonymMap :: SynonymMap  -- exported type synonyms
     , modIface_recordMap  :: RecordMap   -- exported record information 
     }

showModIface :: ModIface -> String
showModIface iface
   = unlines [ "Idents", showIdentMap $ modIface_idents iface,
               "Assocs", showAssocMap $ modIface_assocs iface,
               "Infix", showInfixMap $ modIface_infixMap iface,
               "Synonyms", showSynonymMap $ modIface_synonymMap iface,
               "Records", showRecordMap $ modIface_recordMap iface ]

-- join two module interfaces together
-- useful for accumulating interfaces
plusModIface :: ModIface -> ModIface -> ModIface
plusModIface iface1 iface2
   = ModIface { modIface_idents     = idents1 `plusIdentMap` idents2
              , modIface_assocs     = assocs1 `plusAssocMap` assocs2
              , modIface_infixMap   = infixs1 `plusInfixMap` infixs2
              , modIface_synonymMap = syns1   `plusSynonymMap` syns2
              , modIface_recordMap  = recs1   `plusRecordMap` recs2
              }
   where
   idents1 = modIface_idents iface1
   idents2 = modIface_idents iface2
   assocs1 = modIface_assocs iface1
   assocs2 = modIface_assocs iface2
   infixs1 = modIface_infixMap iface1
   infixs2 = modIface_infixMap iface2
   syns1   = modIface_synonymMap iface1
   syns2   = modIface_synonymMap iface2
   recs1   = modIface_recordMap iface1
   recs2   = modIface_recordMap iface2

--------------------------------------------------------------------------------

{- 
   associates a type constructor to its data constructors
   and classes to their members
 
   note that we record the qualified and unqualified associations of tycons
   and constructors, because some modules are imported qualified 
   
   eg. Bool -> [True, False]
       Maybe -> [Just, Nothing]
       Show  -> [show, showsPrec] 
       Foo.Show -> [show]
-}
type AssocMap 
   = FiniteMap 
        HsQName    -- the type or class name
        [HsName]   -- the datacons or members

emptyAssocMap :: AssocMap
emptyAssocMap = emptyFM

-- lookup a class or tycon in an association to find its members
lookupAssocMap :: AssocMap -> HsQName -> Maybe [HsName]
lookupAssocMap = lookupFM 

  -- pretty crude printing of an Association 
showAssocMap :: AssocMap -> String
showAssocMap assocMap
   = unlines $ map show tableList
   where      
   tableList = assocMapToList assocMap 

-- convert a list into an AssocMap
listToAssocMap :: [(HsQName, [HsName])] -> AssocMap
listToAssocMap = listToFM 

listToAssocMap_C :: ([HsName] -> [HsName] -> [HsName])
              -> [(HsQName, [HsName])] -> AssocMap
listToAssocMap_C combiner [] = emptyAssocMap
listToAssocMap_C combiner ((name, members):items)
   = addToFM_C combiner (listToAssocMap_C combiner items) name members 

assocMapToList :: AssocMap -> [(HsQName, [HsName])]
assocMapToList = fmToList

-- join two association maps together
-- there is potential for overlap between the two
-- so we must sometimes combine them
plusAssocMap :: AssocMap -> AssocMap -> AssocMap
-- plusAssocMap = plusFM_C (\mems1 mems2 -> nub (mems1 ++ mems2))
plusAssocMap = plusFM_C union 

-- collect the associations from a list of declarations
-- type synonyms result in empty associations!
assocFromDecls :: Module -> [HsDecl] -> AssocMap
assocFromDecls _modName [] = emptyAssocMap
assocFromDecls modName (HsTypeDecl _sloc name _args _rhs : decls)
   = insertMap QualAndUnQual (assocFromDecls modName decls) (Qual modName name) []
assocFromDecls modName ((HsDataDecl _sloc _cntxt tyName _args condecls _derives):decls)
   = insertMap QualAndUnQual (assocFromDecls modName decls) (Qual modName tyName) (conNamesAndFields condecls) 
assocFromDecls modName ((HsNewTypeDecl _sloc _cntxt tyName _args condecl _derives):decls)
   = insertMap QualAndUnQual (assocFromDecls modName decls) (Qual modName tyName) (conNamesAndFields [condecl]) 
assocFromDecls modName ((HsClassDecl _sloc _cntxt className _args members):decls)
   = insertMap QualAndUnQual (assocFromDecls modName decls) (Qual modName className) (memberNames members)
assocFromDecls modName (_other:decls)
   = assocFromDecls modName decls

-- collect the names of data constructors from a list of constructor decls
-- also collect the field names
conNamesAndFields :: [HsConDecl] -> [HsName]
conNamesAndFields decls
   = concatMap conName decls
   where
   conName :: HsConDecl -> [HsName]
   conName (HsConDecl _sloc name _types) = [name]
   conName (HsRecDecl _sloc name members) = name : fieldNames members
   fieldNames :: [([HsName],HsBangType)] -> [HsName]
   fieldNames fields = concatMap fst fields

-- collect the names of the member functions in a type class
memberNames :: [HsDecl] -> [HsName]
memberNames 
   = concatMap memberName
   where
   memberName :: HsDecl -> [HsName]
   memberName (HsTypeSig _sloc names _types) = names
   memberName other = [] 

--------------------------------------------------------------------------------

{- creating the module interface 
   in the IO monad so we can print nice error messages
   and exit gracefully -}

mkModIface :: Module 
           -> ImportMap
           -> ModIface
           -> Maybe [HsExportSpec]
           -> IO ModIface

-- export everything from this module (not imported from others!)
mkModIface modName _imports totalIface Nothing
   = do let identMap   = modIface_idents totalIface
            assocMap   = modIface_assocs totalIface
            infixMap   = modIface_infixMap totalIface
            synonymMap = modIface_synonymMap totalIface
            recordMap  = modIface_recordMap totalIface
        return ModIface { -- since there will be a qualified and unqualified entry for each
                          -- item in this module's ident map we must remove half, doesn't matter
                          -- which half, we choose to remove the unqualified versions
                          modIface_idents  = filterMapQualified identMap modName
                          -- as above but for associations
                        , modIface_assocs  = filterMapQualified assocMap modName
                          -- as above but for infix rules
                        , modIface_infixMap = filterMapQualified infixMap modName
                          -- as above but for synonyms
                        , modIface_synonymMap = filterMapQualified synonymMap modName
                          -- as above but for synonyms
                        , modIface_recordMap = filterMapQualified recordMap modName
                        }

-- accumulate the interface from each of the import specifications
mkModIface modName imports totalModIface (Just specs)
   = mkModIfaceSpecs modName ifaceInit imports totalModIface specs
   where
   -- an initial empty interface
   ifaceInit = ModIface { modIface_idents     = emptyIdentMap
                        , modIface_assocs     = emptyAssocMap
                        , modIface_infixMap   = emptyInfixMap
                        , modIface_synonymMap = emptySynonymMap 
                        , modIface_recordMap  = emptyRecordMap }

mkModIfaceSpecs :: Module 
                -> ModIface 
                -> ImportMap
                -> ModIface
                -> [HsExportSpec]
                -> IO ModIface
mkModIfaceSpecs _modName ifaceAcc _imports _totalIface [] = return ifaceAcc

-- export a variable
-- its ident information and any fixity information
mkModIfaceSpecs modName ifaceAcc imports totalIface (HsEVar qName : specs)
   = do let identMap = modIface_idents totalIface
        let infixMap = modIface_infixMap totalIface
        case lookupIdent identMap qName of
           Nothing -> abortWithError ExportError errMsg
           Just info -> do let newIdentMapAcc = addToFM (modIface_idents ifaceAcc) qName info
                               extraFixitys   = filterFM (\name _ -> name == qName) infixMap  
                               newInfixMap    = extraFixitys `plusInfixMap` (modIface_infixMap ifaceAcc)
                               newIfaceAcc    = ifaceAcc { modIface_idents   = newIdentMapAcc  
                                                         , modIface_infixMap = newInfixMap
                                                         } 
                           mkModIfaceSpecs modName newIfaceAcc imports totalIface specs
   where
   errMsg = "attempt to export a variable that is not in scope: " ++ fromHsQNameQual qName

-- export an abstract entity
-- perhaps it is a synonym - include this in the interface
-- check that it is in scope
-- record an empty association for it
mkModIfaceSpecs modName ifaceAcc imports totalIface (HsEAbs qName : specs)
   = do let synMap = modIface_synonymMap totalIface
            assocMap = modIface_assocs totalIface
            -- check that this thing is in scope (should be in the assoc map)
        case lookupAssocMap assocMap qName of
            Nothing -> abortWithError ExportError errMsg
            Just _  -> do let newSynMapAcc 
                                 = case lookupSynonymMap synMap qName of
                                      Nothing -> modIface_synonymMap ifaceAcc  
                                      Just info -> addToFM (modIface_synonymMap ifaceAcc) qName info 
                              newAssocsAcc = addToFM_C union (modIface_assocs ifaceAcc) qName []
                              newIfaceAcc  = ifaceAcc { modIface_synonymMap = newSynMapAcc 
                                                      , modIface_assocs = newAssocsAcc
                                                      } 
                          mkModIfaceSpecs modName newIfaceAcc imports totalIface specs
   where
   errMsg = "attempt to export either a tycon, class or synonym that is not in scope: " ++ 
            show qName

-- export T(..), turn it into an explicit export spec
-- export fixity information for any of the members
-- export record selectors for any members
mkModIfaceSpecs modName ifaceAcc imports totalIface (HsEThingAll qName : specs)
   = do let assocMap  = modIface_assocs totalIface
        case lookupAssocMap assocMap qName of
           Nothing   -> abortWithError ExportError errMsg
           Just mems -> do let items = map fromNameToCName mems
                           mkModIfaceSpecs modName ifaceAcc imports totalIface
                                        (HsEThingWith qName items : specs)
{-
           Just mems -> do let qMems = map UnQual mems 
                           memsIdentInfoList 
                              <- identListFromItems ExportError 
                                                    (modIface_idents totalIface)
                                                    qMems
                           let newIdentsAcc = addListItemsToMap UnQualified
                                                                memsIdentInfoList
                                                                (modIface_idents ifaceAcc)  
                               newAssocsAcc = addToFM_C union (modIface_assocs ifaceAcc) qName mems 
                               extraFixitys = filterFM (\name _ -> name `elem` qMems) infixMap  
                               newInfixMap = extraFixitys `plusInfixMap` (modIface_infixMap ifaceAcc)
                               extraRecords = filterFM (\name _ -> name `elem` qMems) recordMap 
                               newRecordsAcc = addToFM_C union (modIface_recordMap ifaceAcc) qName extraRecords 
                               newIfaceAcc = ifaceAcc { modIface_idents = newIdentsAcc 
                                                      , modIface_assocs = newAssocsAcc
                                                      , modIface_infixMap = newInfixMap
                                                      } 
                           mkModIfaceSpecs modName newIfaceAcc imports totalIface specs
-}
   where
   errMsg = "attempt to export either a tycon, class that is not in scope: " ++ show qName 

-- export T(Foo,Bar)
-- export fixity information for any of the members
mkModIfaceSpecs modName ifaceAcc imports totalIface (HsEThingWith qName items: specs)
   = do let assocMap  = modIface_assocs totalIface
            infixMap  = modIface_infixMap totalIface
            recordMap = modIface_recordMap totalIface
        case lookupAssocMap assocMap qName of
           Nothing   -> abortWithError ExportError errMsg1
           Just mems -> do let itemNames = map fromCNameToName items
                               qItemNames = case maybeIsQualified qName of
                                               Just qualifier -> map (Qual qualifier) itemNames
                                               Nothing -> map UnQual itemNames 
                           case notMembers itemNames mems of
                              notMems@(_:_) -> abortWithError ExportError (errMsg2 notMems)
                              [] -> do let recordInfos = selectRecordInfos recordMap qItemNames
                                           -- selectorNames = nub $ concatMap snd recordInfos
                                       memsIdentInfoList 
                                          <- identListFromItems ExportError 
                                                    (modIface_idents totalIface)
                                                    -- (qItemNames ++ selectorNames)
                                                    qItemNames 
                                       let newIdentsAcc = addListItemsToMap UnQualified
                                                                memsIdentInfoList
                                                                (modIface_idents ifaceAcc)  
                                           newAssocsAcc = addToFM_C union (modIface_assocs ifaceAcc) qName itemNames 
                                           extraFixitys = filterFM (\name _ -> name `elem` qItemNames) infixMap  
                                           newInfixMap = extraFixitys `plusInfixMap` (modIface_infixMap ifaceAcc)
                                           newRecordsAcc = addListItemsToMap UnQualified
                                                              recordInfos
                                                              (modIface_recordMap ifaceAcc)
                                           newIfaceAcc = ifaceAcc { modIface_idents   = newIdentsAcc 
                                                                  , modIface_assocs   = newAssocsAcc
                                                                  , modIface_infixMap = newInfixMap
                                                                  , modIface_recordMap = newRecordsAcc
                                                                  } 
                                          
                                       mkModIfaceSpecs modName newIfaceAcc imports totalIface specs
   where
   errMsg1 = "attempt to export either a tycon, class that is not in scope: " ++ show qName
   errMsg2 items
      = "type or class " ++ show qName ++ " does not export these items: " ++ unwords (map show items)
                                
{- export a whole module
   
   There are two key behaviours here:
   1. A module is exporting itself, as in
         module Foo (module Foo)
   2. A module is exporting another module, as in
        module Foo (module Bar)

   The first case is treated just as:
      module Foo
   in other words export everything defined in this module that is
   exportable

   In the second case we export everything that was brought in scope
   by the accumulation of _unqualified_ import declarations for that module.
   To keep track of this accumulation we use the ImportMap, in fact this is
   the sole reason for the import map - just in case we have to export
   a whole module.
-}

mkModIfaceSpecs modName ifaceAcc imports totalIface (HsEModuleContents exportModName : specs)
   = do let identMap   = modIface_idents totalIface
            assocMap   = modIface_assocs totalIface
            infixMap   = modIface_infixMap totalIface
            synonymMap = modIface_synonymMap totalIface
            recordMap  = modIface_recordMap totalIface
        newIface  
           <- case modName == exportModName of
                 True  -> return ModIface 
                                 { modIface_idents     = filterMapQualified identMap modName
                                 , modIface_assocs     = filterMapQualified assocMap modName
                                 , modIface_infixMap   = filterMapQualified infixMap modName
                                 , modIface_synonymMap = filterMapQualified synonymMap modName
                                 , modIface_recordMap  = filterMapQualified recordMap modName
                                 }
                 False 
                    -> case lookupImportMap imports exportModName of
                          Nothing -> abortWithError ExportError errMsg
                          Just importInfo 
                             -> do let exportIdents = importInfo_idents importInfo
                                       exportTycons = importInfo_tycons importInfo
                                   return ModIface
                                          { modIface_idents     
                                               = filterFM (\name _ -> name `elem` exportIdents) identMap 
                                          , modIface_assocs     
                                               = filterFM (\name _ -> name `elem` exportTycons) assocMap 
                                          , modIface_infixMap   
                                               = filterFM (\name _ -> name `elem` exportIdents) infixMap 
                                          , modIface_synonymMap 
                                               = filterFM (\name _ -> name `elem` exportTycons) synonymMap
                                          , modIface_recordMap
                                               = filterFM (\name _ -> name `elem` exportIdents) recordMap 
                                          }
        let newIfaceAcc = ifaceAcc `plusModIface` newIface
        mkModIfaceSpecs modName newIfaceAcc imports totalIface specs
   where
   errMsg = "attempt to export a module which is not in scope: " ++ show exportModName


-- find all the items in a finitemap whose key is a name which is
-- qualified to a certain module
filterMapQualified :: FiniteMap HsQName a -> Module -> FiniteMap HsQName a
filterMapQualified map modName
   = filterFM (\name _ -> thisModQualified modName name) map

fromCNameToName :: HsCName -> HsName
fromCNameToName (HsVarName name) = name
fromCNameToName (HsConName name) = name

fromNameToCName :: HsName -> HsCName
fromNameToCName name@(HsIdent str@(s:ss))
   | isUpper  s = HsConName name
   | otherwise = HsVarName name
fromNameToCName name@(HsSymbol str@(s:ss))
   | isColon s = HsConName name
   | otherwise = HsVarName name
   where
   isColon :: Char -> Bool
   isColon c = c == ':'

-- find all the record selectors that match the constructors
selectRecordInfos :: RecordMap -> [HsQName] -> [(HsQName, [HsQName])]
selectRecordInfos recMap [] = []
selectRecordInfos recMap (con : rest)
   = case lookupRecord recMap con of
        Nothing -> restInfos
        Just selectors -> (con, selectors) : restInfos
   where
   restInfos = selectRecordInfos recMap rest

--------------------------------------------------------------------------------   

{- writing the interface to file -}

-- the suffix for interface files
interfaceSuffix = ".i"

-- buddha's private directory
interfaceDirectory = "Buddha"

-- the name for the list of idents in the interface file
identsName = "idents"

-- the name for the list of assocs in the interface file
assocsName = "assocs"

-- the name for the list of records in the interface file
recordsName = "records"

-- the name for the list of infix information in the interface file
infixName = "fixity"

interfaceMessage :: String
interfaceMessage
    = "{- This is an interface file for Buddha, a declarative debugger\n" ++ 
      "   for Haskell 98. Please do not modify this file -}\n"

-- create an interface file for a module and write its contents using Haskell syntax
writeModIface :: Module -> SynonymMap -> ModIface -> IO ()
writeModIface modName@(Module thisModuleName) allSynonyms modIface 
   = do currentDir <- getCurrentDirectory
        let newFileName = thisModuleName ++ interfaceSuffix 
        handle <- createFileOverWrite (currentDir ++ "/" ++ interfaceDirectory) 
                                      newFileName
        hPutStrLn handle interfaceMessage
        let identStr = identsName ++ " = " ++ (PPrint.render $ prettyIdents $ modIface_idents modIface)
        hPutStrLn handle "\n{- The ident info for all constructors and top level identifiers -}\n"
        hPutStrLn handle identStr 
        let assocStr = assocsName ++ " = " ++ (PPrint.render $ prettyAssocs $ modIface_assocs modIface)
        hPutStrLn handle "\n{- The class and tycon associations -}\n"
        hPutStrLn handle assocStr 
        let infixStr = infixName ++ " = " ++ (PPrint.render $ prettyInfix $ modIface_infixMap modIface)
        hPutStrLn handle "\n{- The infix rules -}\n"
        hPutStrLn handle infixStr 
        let synDecls = synonymMapToDecls allSynonyms $ modIface_synonymMap modIface
        let synDeclsStr = concatMap Pretty.prettyPrint synDecls 
        hPutStrLn handle "\n{- The type synonyms -}"
        hPutStrLn handle synDeclsStr 
        let recordStr = recordsName ++ " = " ++ (PPrint.render $ prettyRecords $ modIface_recordMap modIface)
        hPutStrLn handle "\n{- The record constructors -}\n"
        hPutStrLn handle recordStr 
        hClose handle

-- pretty print the ident information as a list
prettyIdents :: IdentMap -> PPrint.Doc
prettyIdents idents 
   = PPrint.pplist selectiveIdentInfo 
   where
   selectiveIdentInfo :: [(HsQName, IdentBind, Arity)] 
   selectiveIdentInfo 
      = [(name, identInfo_bindType info, identInfo_arity info) 
                      | (name, info) <- identMapToList idents] 

-- pretty print the associations as a list
prettyAssocs :: AssocMap -> PPrint.Doc
prettyAssocs assocMap
   =  PPrint.pplist $ assocMapToList assocMap

-- pretty print the associations as a list
prettyRecords :: RecordMap -> PPrint.Doc
prettyRecords recordMap
   =  PPrint.pplist $ recordMapToList recordMap

-- pretty print the fixity information as a list
prettyInfix :: InfixMap -> PPrint.Doc
prettyInfix infixMap
   = PPrint.pplist selectiveInfixInfo
   where
   selectiveInfixInfo 
      = [(name, fixity_assoc fixity, fixity_prec fixity) 
                      | (name, fixity) <- infixMapToList infixMap]

instance PPrint.PPrint HsQName where
   pprint name
      = case name of
           Qual _mod name -> PPrint.pprint name
           UnQual    name -> PPrint.pprint name
           Special _spec  
              -> fatalError __FILE__ __LINE__ $ 
                    "found special constructor whilst writing interface file: " ++ 
                              show name

instance PPrint.PPrint HsName where
   pprint name
      = case name of
           HsIdent  string -> PPrint.text string
           HsSymbol string -> PPrint.parens (PPrint.text string)

-- pretty print an identifier binding type
instance PPrint.PPrint IdentBind where
   pprint Pat        = PPrint.text "Pat"
   pprint LetFun     = PPrint.text "LetFun"
   pprint LetConst   = PPrint.text "LetConst"
   pprint LetConstOL = PPrint.text "LetConstOL"
   pprint Con        = PPrint.text "Con"

-- pretty print an associativity
instance PPrint.PPrint HsAssoc where
   pprint HsAssocNone  = PPrint.text "N"
   pprint HsAssocLeft  = PPrint.text "L"
   pprint HsAssocRight = PPrint.text "R"

--------------------------------------------------------------------------------

-- reading an interface file

{- 
   read the basic contents of an interface file for a given module
   for module Foo, the file is currentDir/Buddha/Foo.i
-}
readModIface :: FilePath -> Module -> IO ModIface 
readModIface ifacePath thisModule@(Module name) 
   = do let interfaceFile = name ++ interfaceSuffix
        currentDir <- getCurrentDirectory 
        let filePath1 
               = currentDir ++ directorySep ++ interfaceDirectory ++ directorySep ++ interfaceFile
            filePath2 = ifacePath ++ directorySep ++ interfaceFile
        -- try to open the file
        handle <- tryOpenFile ReadMode [filePath1, filePath2]
        -- read its contents
        srcText <- hGetContents handle
        -- parse the contents (as Haskell source)
        interfaceSyntax <- parseHsSource interfaceFile srcText
        -- deconstruct the module
        let (HsModule _srcLoc _name _exports _imports decls) = interfaceSyntax 
        -- find the ident info, parse it, interpret it, and convert into IdentMap
        identMap <- case findDecl identsName decls of
                         -- couldn't find the ident information
                         Nothing 
                            -> abortWithError 
                                  MiscError $ 
                                     "could not find ident information in the interface file for module: " ++ 
                                     name
                         -- try to interpret the ident information (XXX very fragile)
                         Just d -> case interpretIdents d of
                                      -- something weird with the ident information
                                      Nothing -> abortWithError
                                                    MiscError $
                                                    "could not interpret ident information for module: " ++ 
                                                    name 
                                      Just identMap -> return identMap 
        associations 
           <- case findDecl assocsName decls of
                         -- couldn't find the assoc information
                         Nothing 
                            -> abortWithError 
                                  MiscError $ 
                                     "could not find association information in the interface file for module: " ++ name
                         -- try to interpret the assoc information (XXX very fragile)
                         Just d -> case interpretAssocs d of
                                   -- something weird with the association information
                                      Nothing -> abortWithError
                                                    MiscError $
                                                    "could not interpret association information for module: " ++ 
                                                    name 
                                      Just identMap -> return identMap 
        infixMap 
           <- case findDecl infixName decls of
                         -- couldn't find the infix information
                         Nothing 
                            -> abortWithError 
                                  MiscError $ 
                                     "could not find infix information in the interface file for module: " ++ name
                         -- try to interpret the infix information (XXX very fragile)
                         Just d -> case interpretInfix d of
                                   -- something weird with the association information
                                      Nothing -> abortWithError
                                                    MiscError $
                                                    "could not interpret infix information for module: " ++ 
                                                    name 
                                      Just infixMap -> return infixMap
        let synonymMap = declsToSynonymMap decls 
        recordMap 
           <- case findDecl recordsName decls of
                 -- couldn't find the record information
                 Nothing 
                    -> abortWithError MiscError $ 
                       "could not find record information in the interface file for module: " ++ 
                       name
                         -- try to interpret the record information (XXX very fragile)
                 Just d -> case interpretRecords d of
                              -- something weird with the record information
                              Nothing -> abortWithError MiscError $
                                         "could not interpret record information for module: " ++ 
                                         name 
                              Just recMap -> return recMap
        let iface = ModIface { modIface_idents   = identMap
                             , modIface_assocs   = associations
                             , modIface_infixMap = infixMap
                             , modIface_synonymMap = synonymMap 
                             , modIface_recordMap = recordMap }
        -- error $ showModIface iface
        return iface

-- find a declaration from a list of declarations  
findDecl :: String -> [HsDecl] -> Maybe HsDecl
findDecl _name [] = Nothing
findDecl name (d@(HsPatBind _srcLoc (HsPVar (HsIdent s)) _rhs _wheres) : decls)
   | s == name = Just d
   | otherwise = findDecl name decls
findDecl name (_anythingElse : decls) = findDecl name decls

-- convert the syntactic representation of the ident data back into
-- an ident map
interpretIdents :: HsDecl -> Maybe IdentMap 
interpretIdents (HsPatBind _srcLoc _name (HsUnGuardedRhs (HsList exps)) _wheres)
   -- = Just $ addListIdsToMap UnQualified listNameInfo emptyIdentMap
   = Just $ addListItemsToMap UnQualified listNameInfo emptyIdentMap
   where
   listNameInfo = map interpretIdentTuple exps
   interpretIdentTuple :: HsExp -> (HsQName, IdentInfo)
   interpretIdentTuple (HsTuple [name, bind, numArgs])
      = (interpretName name,
         IdentInfo { identInfo_bindType = interpretBind bind  
                   , identInfo_arity    = interpretLitInt numArgs })
   interpretIdentTuple other 
      = fatalError __FILE__ __LINE__ $ 
           "interpretIdentTuple: strange data in ident info in interface file: " ++ show other
   interpretBind :: HsExp -> IdentBind
   interpretBind (HsCon (UnQual (HsIdent "Pat")))        = Pat
   interpretBind (HsCon (UnQual (HsIdent "LetFun")))     = LetFun
   interpretBind (HsCon (UnQual (HsIdent "LetConst")))   = LetConst
   interpretBind (HsCon (UnQual (HsIdent "LetConstOL"))) = LetConstOL
   interpretBind (HsCon (UnQual (HsIdent "Con")))        = Con
   interpretBind other
      = fatalError __FILE__ __LINE__ $ 
           "interpretBind: strange data in ident info in interface file: " ++ show other
interpretIdents _anythingElse = Nothing

-- convert the syntactic representation of the assoc data back into
-- an AssocMap 
interpretAssocs :: HsDecl -> Maybe AssocMap 
interpretAssocs (HsPatBind _srcLoc _name (HsUnGuardedRhs (HsList exps)) _wheres)
   = Just $ addListItemsToMap UnQualified listAssocs emptyAssocMap
   where
   listAssocs :: [(HsQName, [HsName])]
   listAssocs = map interpretAssocTuple exps
   interpretAssocTuple :: HsExp -> (HsQName, [HsName])
   interpretAssocTuple (HsTuple [name, members])
      = (interpretName name, interpretMembers members)
   interpretAssocTuple other 
      = fatalError __FILE__ __LINE__ $ 
           "interpretAssocTuple: strange data in assoc info in interface file: " ++ show other
   interpretMembers :: HsExp -> [HsName]
   interpretMembers (HsList mems)
      = map (unQualify . interpretName) mems
   interpretMembers other
      = fatalError __FILE__ __LINE__ $ 
           "interpretMembers: strange data in info in interface file: " ++ show other
interpretAssocs _anythingElse = Nothing

-- convert the syntactic representation of the reocord data back into
-- an RecordMap 
interpretRecords :: HsDecl -> Maybe RecordMap 
interpretRecords (HsPatBind _srcLoc _name (HsUnGuardedRhs (HsList exps)) _wheres)
   = Just $ addListItemsToMap UnQualified listRecords emptyRecordMap
   where
   listRecords :: [(HsQName, [HsQName])]
   listRecords = map interpretRecordTuple exps
   interpretRecordTuple :: HsExp -> (HsQName, [HsQName])
   interpretRecordTuple (HsTuple [name, members])
      = (interpretName name, interpretMembers members)
   interpretRecordTuple other 
      = fatalError __FILE__ __LINE__ $ 
           "interpretRecordTuple: strange data in record info in interface file: " ++ show other
   interpretMembers :: HsExp -> [HsQName]
   interpretMembers (HsList mems) = map interpretName mems
   interpretMembers other
      = fatalError __FILE__ __LINE__ $ 
           "interpretMembers: strange data in interface file: " ++ show other
interpretRecords _anythingElse = Nothing

-- convert the syntactic representation of the inifx data back into
-- an AssocMap 
interpretInfix :: HsDecl -> Maybe InfixMap 
interpretInfix (HsPatBind _srcLoc _name (HsUnGuardedRhs (HsList exps)) _wheres)
   = Just $ addListItemsToMap UnQualified listInfixInfo emptyInfixMap
   where
   listInfixInfo :: [(HsQName, Fixity)]
   listInfixInfo = map interpretInfixTuple exps
   interpretInfixTuple :: HsExp -> (HsQName, Fixity)
   interpretInfixTuple (HsTuple [name, assoc, prec])
      = (interpretName name,
         Fixity { fixity_assoc = interpretAssoc assoc
                , fixity_prec  = interpretLitInt prec})
   interpretInfixTuple other
      = fatalError __FILE__ __LINE__ $ 
           "interpretInfixTuple: strange data in ident info in interface file: " ++ show other
   interpretAssoc :: HsExp -> HsAssoc 
   interpretAssoc (HsCon (UnQual (HsIdent "L"))) = HsAssocLeft 
   interpretAssoc (HsCon (UnQual (HsIdent "N"))) = HsAssocNone
   interpretAssoc (HsCon (UnQual (HsIdent "R"))) = HsAssocRight 
   interpretAssoc other
      = fatalError __FILE__ __LINE__ $ 
           "interpretAssoc: strange data in ident info in interface file: " ++ show other
interpretInfix _anythingElse = Nothing


-- interpret a variable expression as a name (make sure it isn't qualified also)
interpretName :: HsExp -> HsQName
interpretName (HsCon qName) = checkQName qName
interpretName (HsVar qName) = checkQName qName
interpretName other 
   = fatalError __FILE__ __LINE__ $ 
        "interpretName: strange data in info in interface file: " ++ show other
-- make sure the HsQName isn't really qualified
checkQName :: HsQName -> HsQName
checkQName qName@(Qual _mod _name) 
   = fatalError __FILE__ __LINE__ $ 
        "checkQName: found qualified name in interface file :" ++ show qName
checkQName qName@(UnQual _name) = qName

-- interpret an expression as a literal integer
interpretLitInt :: HsExp -> Int
interpretLitInt (HsLit (HsInt i)) = fromIntegral i 
interpretLitInt other
   = fatalError __FILE__ __LINE__ $ 
          "interpretLitInt: strange data in ident info in interface file: " ++ show other


--------------------------------------------------------------------------------

{- process all the import Decls 
   this is quite complex due to the presence of hiding and qualified
   imports and aliases
-}

processImports :: FilePath
               -> [HsImportDecl]
               -> IfaceCache
               -> IO (IfaceCache, ImportMap, ModIface)
processImports ifacePath imports cache 
   = do (cache, imports, iface) <- processImports' imports cache emptyImportMap initModIface
        -- putStrLn $ showModIface iface
        return (cache, imports, iface)
   where
   initModIface = ModIface { modIface_idents     = emptyIdentMap
                           , modIface_assocs     = emptyAssocMap
                           , modIface_infixMap   = emptyInfixMap
                           , modIface_synonymMap = emptySynonymMap
                           , modIface_recordMap  = emptyRecordMap
                           }
   processImports' :: [HsImportDecl] 
                   -> IfaceCache 
                   -> ImportMap 
                   -> ModIface
                   -> IO (IfaceCache, ImportMap, ModIface)
   processImports' [] cache importMap modIface 
      = return (cache, importMap, modIface)
   processImports' (decl:decls) cache importMap modIface 
      = do (newCache, (importModName, importInfo), nextModIface) <- importDecl ifacePath decl cache 
           let newModIface = modIface `plusModIface` nextModIface
               newImportMap = addToImportMap importMap importModName importInfo 
           processImports' decls newCache newImportMap newModIface 
        
{- process one import decl -}

importDecl :: FilePath 
           -> HsImportDecl    -- this import statement (ie, import qualified Foo (bar)) 
           -> IfaceCache      -- a cache of module interfaces
           -> IO (IfaceCache, (Module, ImportInfo), ModIface)

-- import everything from the module
-- must cater for possible qualification and also use of an alias
importDecl ifacePath (HsImportDecl _sloc modName isQualified asName Nothing) cache 
   = do -- read the module interface
        (importModIface, newCache) <- getModIface ifacePath modName cache 
            -- unpack the module interface
        let identList = identMapToList   $ modIface_idents importModIface
            infixList = infixMapToList   $ modIface_infixMap importModIface
            assocList = assocMapToList   $ modIface_assocs importModIface
            synList   = synonymMapToList $ modIface_synonymMap importModIface
            recList   = recordMapToList  $ modIface_recordMap importModIface
            -- check for qualification
            qualification = case isQualified of
                               True  -> Qualified  
                               False -> QualAndUnQual
            -- check for an alias
            thisModName = case asName of
                             Nothing    -> modName
                             Just alias -> alias
            -- rebuild the module interface accounting for alias and qualification
            -- (when we have a unqualified import we get two versions of every
            --  item in each map - this is done by addListItemsToMap)
            newIdentMap = addListItemsToMap qualification (qualifyList thisModName identList) emptyIdentMap
            newInfixMap = addListItemsToMap qualification (qualifyList thisModName infixList) emptyInfixMap
            -- don't need to do any association combining because we are just
            -- renaming an existing map
            newAssocMap = addListItemsToMap qualification (qualifyList thisModName assocList) emptyAssocMap
            newSynMap   = addListItemsToMap qualification (qualifyList thisModName synList) emptySynonymMap 
            newRecMap   = addListItemsToRecMap qualification thisModName recList emptyRecordMap
            -- rebuild the new module interface
            newModIface = ModIface { modIface_idents     = newIdentMap
                                   , modIface_assocs     = newAssocMap
                                   , modIface_infixMap   = newInfixMap
                                   , modIface_synonymMap = newSynMap
                                   , modIface_recordMap  = newRecMap
                                   } 
            -- record the import info for this particular import
            -- if the import is qualified then don't record anything
            newImportInfo 
                = case isQualified of
                     True  -> emptyImportInfo 
                     False -> ImportInfo { importInfo_idents = map fst identList 
                                         , importInfo_tycons = map fst assocList 
                                         }

        return (newCache, (thisModName, newImportInfo), newModIface)

-- explicit imports (may be hiding)
importDecl ifacePath (HsImportDecl _sloc modName isQualified asName (Just (isHiding, specs))) cache 
   = do -- read the module interface
        (importModIface, newCache) <- getModIface ifacePath modName cache 
        -- unpack the interface
        let ifaceIdents   = modIface_idents     importModIface
            ifaceAssocs   = modIface_assocs     importModIface
            ifaceInfixs   = modIface_infixMap   importModIface
            ifaceSynonyms = modIface_synonymMap importModIface
            ifaceRecords  = modIface_recordMap  importModIface
        -- check for an alias
        let thisModName = case asName of
                             Nothing    -> modName
                             Just alias -> alias
        -- expand the import specs into the imported idents and the imported assocs
        (importedIdents, importedAssocs)
            <- expandImportSpecList modName ifaceAssocs ifaceRecords specs
        let importedTycons = map fst importedAssocs
        -- determine whether we want to include names or hide them
        let membership = case isHiding of
                            True  -> notElem
                            False -> elem
        -- filter out the things we need from the module interface
        let identList = identMapToList   $ filterFM (\name _ -> membership name importedIdents) 
                                                    ifaceIdents 
            infixList = infixMapToList   $ filterFM (\name _ -> membership name importedIdents) 
                                                    ifaceInfixs 
            synList   = synonymMapToList $ filterFM (\name _ -> membership name importedTycons) 
                                                    ifaceSynonyms 
            assocList = case isHiding of
                           -- remember to remove any items from the associations that are now
                           -- hidden
                           True -> assocMapToList $ removeHiddenAssocItems importedIdents $ 
                                                    filterFM (\name _ -> notElem name importedTycons)
                                                    ifaceAssocs
                           False -> importedAssocs
        let importedRecords 
                = filterFM (\name _ -> membership name importedIdents)
                                                    ifaceRecords
            recordList = case isHiding of
                            True -> recordMapToList $ removeHiddenRecordItems importedIdents $ importedRecords
                            False -> recordMapToList importedRecords
            -- check for qualification
            qualification = case isQualified of
                               True  -> Qualified  
                               False -> QualAndUnQual
            -- rebuild the module interface accounting for alias and qualification
            -- (when we have a unqualified import we get two versions of every
            --  item in each map - this is done by addListItemsToMap)
            newIdentMap = addListItemsToMap qualification (qualifyList thisModName identList) emptyIdentMap
            newInfixMap = addListItemsToMap qualification (qualifyList thisModName infixList) emptyInfixMap
            -- the associations may overlap and need combination
            newAssocMap = addListItemsToMap_C union 
                                              qualification 
                                              (qualifyList thisModName assocList) emptyAssocMap
            newSynMap   = addListItemsToMap qualification (qualifyList thisModName synList) emptySynonymMap 
            newRecordMap = addListItemsToRecMap qualification thisModName recordList emptyRecordMap
            -- re build the new module interface
            newModIface = ModIface { modIface_idents     = newIdentMap
                                   , modIface_assocs     = newAssocMap
                                   , modIface_infixMap   = newInfixMap
                                   , modIface_synonymMap = newSynMap
                                   , modIface_recordMap  = newRecordMap
                                   } 
            -- record the import info for this particular import
            -- if the import is qualified then don't record anything
            newImportInfo 
                = case isQualified of
                     True  -> emptyImportInfo 
                     False -> ImportInfo { importInfo_idents = map fst identList 
                                         , importInfo_tycons = map fst assocList 
                                         }
        return (newCache, (thisModName, newImportInfo), newModIface)
   where
   removeHiddenAssocItems :: [HsQName] -> AssocMap -> AssocMap
   removeHiddenAssocItems importedIdents assocs 
      = mapFM (\ _name items -> [item | item <- items, (UnQual item) `notElem` importedIdents])
              assocs
   removeHiddenRecordItems :: [HsQName] -> RecordMap -> RecordMap
   removeHiddenRecordItems importedIdents records 
      = mapFM (\ _name items -> [item | item <- items, item `notElem` importedIdents])
              records 
               
     
-- ensure everything is qualified to the right name
qualifyList :: Module -> [(HsQName, b)] -> [(HsQName, b)]
qualifyList modName list 
   = map (reQualifyItem modName) list
   where
   reQualifyItem :: Module -> (HsQName, b) -> (HsQName, b)
   reQualifyItem modName (qName, x) = (reQualify modName qName, x)
                          

-- read a module interface either from within the cache or
-- else from the file system
getModIface :: FilePath -> Module -> IfaceCache -> IO (ModIface, IfaceCache)
getModIface ifacePath modName cache
   = do case lookupIfaceCache cache modName of
           -- found it in the cache
           Just iface -> return (iface, cache)
           -- read it from file
           Nothing -> do iface <- readModIface ifacePath modName
                         return (iface, addIfaceToCache cache modName iface )

-- expand a whole list of import specs
expandImportSpecList :: Module -> AssocMap -> RecordMap -> [HsImportSpec] -> IO ([HsQName], [(HsQName, [HsName])])
expandImportSpecList modName assocs recordMap specs
   = do (varss, assocss) <- mapAndUnzipM (expandImportSpec modName assocs recordMap) specs
        return (concat varss, concat assocss)

-- return the vars/cons and assocs syns that are
-- mentioned by an import spec (implicitly or explicitly)
expandImportSpec :: Module -> AssocMap -> RecordMap -> HsImportSpec -> IO ([HsQName], [(HsQName, [HsName])])
-- just a variable
expandImportSpec _modName assocs _recordMap (HsIVar name)
   = return ([UnQual name], [])
-- abstract tycon/class, or synonym
expandImportSpec _modName assocs _recordMap (HsIAbs name)
   = return ([], [(UnQual name, [])])
-- a thing with all its items eg. T (..)
-- turn it into the more specific form, HsIThingWith
expandImportSpec modName assocs recordMap (HsIThingAll name)
   = case lookupAssocMap assocs (UnQual name) of
        Nothing -> abortWithError ImportError errMsg
        Just mems -> do let items = map fromNameToCName mems
                        expandImportSpec modName assocs recordMap (HsIThingWith name items)
   where
   errMsg = "module: " ++ show modName ++ " does not export: " ++ show name
-- a thing with all some items eg. T (A,B,C)
expandImportSpec modName assocs recordMap (HsIThingWith name items)
   = case lookupAssocMap assocs (UnQual name) of
        Nothing -> abortWithError ImportError errMsg
        Just mems -> 
           do let itemsNames = map fromCNameToName items
              -- check that the items we are trying to export are actually 
              -- provided by the type or class
              case notMembers itemsNames mems of
                 notMems@(_:_)  
                    -> abortWithError ExportError (errMsg2 notMems)
                 [] -> do let qualMems = map UnQual itemsNames 
                          -- let recordInfos = selectRecordInfos recordMap qualMems 
                          --    selectorNames = nub $ concatMap snd recordInfos
                          -- return (qualMems ++ selectorNames, [(UnQual name, itemsNames)])
                          return (qualMems, [(UnQual name, itemsNames)])
   where
   errMsg = "module: " ++ show modName ++ " does not export: " ++ show name
   errMsg2 items
      = "type or class: " ++ show name ++ 
        "from module: " ++ show modName ++
        " does not export these items: " ++ unwords (map show items)

--------------------------------------------------------------------------------

{- in the IO monad for better error messages -}
identListFromItems :: ErrorCode -> IdentMap -> [HsQName] -> IO [(HsQName, IdentInfo)]
identListFromItems _errCode _idents [] = return [] 
identListFromItems errCode idents (item:items)
   = do case lookupIdent idents item of
           Nothing -> abortWithError errCode $ errorMsg item errCode
           Just info -> do restItems <- identListFromItems errCode idents items
                           return $ (item, info) : restItems 
   where
   errorMsg :: HsQName -> ErrorCode -> String
   errorMsg name errCode
      = "attempt to " ++ importOrExport errCode 
                      ++ " an item which is not in scope: " 
                      ++ show name
      where
      importOrExport :: ErrorCode -> String
      importOrExport ImportError = "import"
      importOrExport ExportError = "export" 

--------------------------------------------------------------------------------

{- The Import Map 
   associates a module with the items that are imported from that module,
   naming the things that are explicitly qualified and those that are not.

   import Foo (bar, ram)
   import qualified Foo (fred)

   Foo -> ([bar, ram], [fred])

   The reason we distinguish between qualified and unqualified imports is
   because when a module is exported such as:

   module Bar (module Foo, jack, daxter)
               ^^^^^^^^^^
   only the unqualified imported items are re-exported, due to the peculiar nature
   of the Haskell definition
-}

data ImportInfo
   = ImportInfo { importInfo_idents :: [HsQName]  -- vars and cons 
                , importInfo_tycons :: [HsQName]  -- tycons, class and synonyms 
                }
     deriving Show

emptyImportInfo = ImportInfo { importInfo_idents   = []
                             , importInfo_tycons = []
                             }

type ImportMap = FiniteMap Module ImportInfo

-- crude printing of an import map
showImportMap :: ImportMap -> String
showImportMap importMap 
   -- = show mapList 
   = unlines $ map show mapList 
   where
   mapList :: [(Module, ImportInfo)]
   mapList = importMapToList importMap

lookupImportMap :: ImportMap -> Module -> Maybe ImportInfo
lookupImportMap = lookupFM

importMapToList :: ImportMap -> [(Module, ImportInfo)]
importMapToList = fmToList

emptyImportMap :: ImportMap
emptyImportMap = emptyFM

addToImportMap :: ImportMap -> Module -> ImportInfo -> ImportMap
addToImportMap map modName info
   = addToFM_C combiner map modName info
   where
   combiner :: ImportInfo -> ImportInfo -> ImportInfo
   combiner info1 info2
      = ImportInfo { importInfo_idents   = union vars1 vars2
                   , importInfo_tycons = union tycons1 tycons2
                   }
      where
      vars1 = importInfo_idents info1
      vars2 = importInfo_idents info2
      tycons1 = importInfo_tycons info1
      tycons2 = importInfo_tycons info2


--------------------------------------------------------------------------------

{- the interface cache -}

-- map a module to its interface
type IfaceCache = FiniteMap Module ModIface

lookupIfaceCache :: IfaceCache -> Module -> Maybe ModIface
lookupIfaceCache = lookupFM

addIfaceToCache :: IfaceCache -> Module -> ModIface -> IfaceCache
addIfaceToCache = addToFM 

emptyIfaceCache :: IfaceCache
emptyIfaceCache = emptyFM

--------------------------------------------------------------------------------

{- add all the selectors to the export list if necessary -}

exportSelectors :: Set HsQName -> AssocMap -> Maybe [HsExportSpec] -> Maybe [HsExportSpec]
-- export everything
exportSelectors selNames assocs Nothing = Nothing 
-- export selected items
exportSelectors selNames assocs (Just specs) 
   = Just $ nub $ concatMap exportSelectorsSpecs specs 
   where
   exportSelectorsSpecs :: HsExportSpec -> [HsExportSpec]
   exportSelectorsSpecs spec@(HsEVar name) = [spec] 
   exportSelectorsSpecs spec@(HsEAbs name) = [spec]
   exportSelectorsSpecs spec@(HsEThingAll thingName)
      = spec : impliedSelectorsThingAll thingName
   exportSelectorsSpecs spec@(HsEThingWith thingName items)
      = let (newSpec, newItems) = impliedSelectorsWith spec 
        in newSpec : newItems 
   exportSelectorsSpecs spec@(HsEModuleContents modName) = [spec]
   impliedSelectorsThingAll :: HsQName -> [HsExportSpec]
   impliedSelectorsThingAll name
      = case lookupAssocMap assocs name of
           Nothing -> []
           Just items -> map HsEVar $ filter (\x -> elementOf x selNames) (map UnQual items)
   impliedSelectorsWith :: HsExportSpec -> (HsExportSpec, [HsExportSpec])
   impliedSelectorsWith (HsEThingWith thingName items)
      = (HsEThingWith thingName newItems, selectors)
      where
      (newItems, selectors) = partitionItems items
      partitionItems :: [HsCName] -> ([HsCName], [HsExportSpec])
      partitionItems [] = ([], [])
      partitionItems (item : rest)
         | qualifiedItemName `elementOf` selNames 
              = (restItems, HsEVar qualifiedItemName : restSelectors)
         | otherwise = (item : restItems, restSelectors)
         where
         qualifiedItemName = UnQual $ fromCNameToName item
         (restItems, restSelectors) = partitionItems rest


{- add all the selectors to an import list if necessary -}

importSelectors :: Set HsQName -> AssocMap -> Maybe (Bool, [HsImportSpec]) 
                                           -> Maybe (Bool, [HsImportSpec]) 
importSelectors selNames assocs Nothing = Nothing
importSelectors selNames assocs (Just (hiding, specs)) 
   = Just (hiding, nub $ concatMap importSelectorsSpecs specs)
   where
   importSelectorsSpecs :: HsImportSpec -> [HsImportSpec]
   importSelectorsSpecs spec@(HsIVar name) = [spec] 
   importSelectorsSpecs spec@(HsIAbs name) = [spec]
   importSelectorsSpecs spec@(HsIThingAll thingName)
      = spec : impliedSelectorsThingAll thingName
   importSelectorsSpecs spec@(HsIThingWith thingName items)
      = let (newSpec, newItems) = impliedSelectorsWith spec 
        in newSpec : newItems 
   impliedSelectorsThingAll :: HsName -> [HsImportSpec]
   impliedSelectorsThingAll name
      = case lookupAssocMap assocs (UnQual name) of
           Nothing -> []
           Just items -> map (HsIVar . unQualify) $ filter (\x -> elementOf x selNames) (map UnQual items)
   impliedSelectorsWith :: HsImportSpec -> (HsImportSpec, [HsImportSpec])
   impliedSelectorsWith (HsIThingWith thingName items)
      = (HsIThingWith thingName newItems, selectors)
      where
      (newItems, selectors) = partitionItems items
      partitionItems :: [HsCName] -> ([HsCName], [HsImportSpec])
      partitionItems [] = ([], [])
      partitionItems (item : rest)
         | UnQual itemName `elementOf` selNames 
              = (restItems, HsIVar itemName : restSelectors)
         | otherwise = (item : restItems, restSelectors)
         where
         itemName = fromCNameToName item
         (restItems, restSelectors) = partitionItems rest
