{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Main

        Description:            The driver for the transformation part of
                                buddha.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Main where

import Monad                    
   ( when 
   , unless 
   )

import System                   
   ( getArgs 
   , system 
   , exitWith 
   , ExitCode (..) 
   )

import System.Environment
   ( getEnv )

import Opts                     
   ( processCmdLine 
   , CmdLine (..)
   , makeUsageInfo 
   , flagIsSet
   )

import Transform                
   ( transformSrcFiles )

import Error                    
   ( abortWithError
   , ErrorCode (..) 
   )

import Directory                
   ( getDirectoryContents
   , doesFileExist 
   , createDirectory 
   , doesDirectoryExist 
   , setCurrentDirectory 
   , getCurrentDirectory 
   )

import Depend                   
   ( dependencies )

import FileIO                   
   ( directorySep 
   , fileExistsAndIsReadable
   , copyFile
   )

import ChatIO                   
   ( setChat
   , ChatLevel (..)
   , isChatty 
   , ifChat 
   )

import List                     
   ( intersperse )

import BuddhaName               
   ( buddhaName 
   , buddhaNameVersion
   )

import Statistics               
   ( showStats
   , initStats
   )

import IO                       
   ( hPutStrLn
   , stderr 
   , stdout
   )

import Control.Exception as E
   ( try )

--------------------------------------------------------------------------------

-- here's where the action begins

main 
   = do -- find the paths to the interface files  
        (ifacePath, dataPath) <- pathsFromEnvironment  
        -- process the command line
        cmdline <- getArgs
        let invocation = processCmdLine buddhaName cmdline
        progOpts <- case invocation of
                       Left okay -> return okay   
                       Right err -> abortWithError CmdLineError err
        -- print a help message if requested and exit
        when (needHelp progOpts) $ 
           do {putStrLn $ makeUsageInfo buddhaName; exitWith ExitSuccess} 
        -- print the version number if requested and exit
        when (needVersion progOpts) $ 
           do {putStrLn $ buddhaName ++ " version " ++ VERSION; exitWith ExitSuccess} 
        -- unpack the command line arguments
        let srcFiles  = fileNames progOpts
        let dumps     = dumpFlags progOpts
        let skips     = skipFlags progOpts
        setChat $ chatLevel progOpts
        -- locate the Haskell files to transform
        haskellSrcFiles
           <- case null srcFiles of
                 True  -> allHsFilesInCWD
                 False -> checkFilesExist $ filter isHaskellSrcFileName srcFiles 
        -- if we have zero source files then print an error message and quit
        -- it is important to do this step first, so that if no files are found
        -- we don't make any changes to the filesystem
        when (null haskellSrcFiles) $
           abortWithError CmdLineError "no Haskell files found"
        -- Initialise the ./Buddha directory and files needed in it
        unless (flagIsSet skips "init") $ buddhaInit dataPath
        -- calculate the dependencies amongst the modules
        filesToTrans 
           <- case flagIsSet skips "depend" of
                 True  -> return haskellSrcFiles
                 False -> do modules <- dependencies haskellSrcFiles
                             return $ identifyModules haskellSrcFiles modules
        -- transform each module 
        stats <- if flagIsSet skips "trans"
                    then return initStats
                    else do ifChat Normal $ putStrLn $ buddhaNameVersion ++ ": transforming: " ++ 
                                            (concat $ intersperse " " filesToTrans) 
                            transformSrcFiles ifacePath progOpts filesToTrans 
        -- print statistics if requested
        when (flagIsSet dumps "stats") $
           do hPutStrLn stdout "total stats for program" 
              hPutStrLn stdout $ showStats stats
        -- build the debug executable
        unless (flagIsSet skips "build") $ buildBuddha
        ifChat Normal $ putStrLn $ buddhaNameVersion ++ ": done"
        -- done
        exitWith ExitSuccess 

-- get the names of all the Haskell src files in the current working directory
allHsFilesInCWD
   = do cwd <- getCurrentDirectory
        contents <- getDirectoryContents cwd
        return $ filter isHaskellSrcFileName contents

-- true if it ends in either .hs or .lhs
isHaskellSrcFileName :: FilePath -> Bool
isHaskellSrcFileName filePath
   = case break (=='.') (reverse filePath) of
        ("sh", _)  -> True
        ("shl", _) -> True
        _          -> False

-- this matches module names to their extensions
-- it doesn't work if there is Foo.hs and Foo.lhs on the command line
-- but normally that shouldn't happen
-- could be slow with a big number of files!
identifyModules :: [String] -> [String] -> [String]
identifyModules _ [] = []
identifyModules cmdLineFileNames (m:mods) 
  | hsExists m  = hsMod m  : rest
  | lhsExists m = lhsMod m : rest 
  | otherwise   = rest 
  where
  rest = identifyModules cmdLineFileNames mods
  hsMod m = m ++ ".hs"
  lhsMod m = m ++ ".lhs"
  hsExists m  = hsMod m  `elem` cmdLineFileNames
  lhsExists m = lhsMod m `elem` cmdLineFileNames  

--------------------------------------------------------------------------------

-- initialisation of the Buddha directory

buddhaDir :: String 
buddhaDir = "Buddha"

buddhaMake :: String 
buddhaMake = "buddha-mk"

buddhaMain :: String
buddhaMain = "Main.hs" 

-- XXX this is all very fragile and could use some more error checking
-- and catching of exceptions - this will be especially useful in diagnosing
-- cases where buddha is not installed properly
buddhaInit :: FilePath -> IO ()
buddhaInit dataPath
   = do ifChat Normal $ putStrLn $ buddhaNameVersion ++ ": initialising" 
        initDir
        initFile dataPath buddhaMake
        initFile dataPath buddhaMain 

initDir :: IO ()
initDir
   = do dirExists <- doesDirectoryExist buddhaDir
        case dirExists of
           True -> return ()
           False -> do ifChat Verbose $ putStrLn $ buddhaNameVersion ++ ": creating directory " ++
                                                   buddhaDir
                       createDirectory buddhaDir 

initFile :: FilePath -> FilePath -> IO ()
initFile dataPath file
   = do let destination = buddhaDir ++ directorySep ++ file
        fileExists <- doesFileExist destination 
        case fileExists of
           True -> return ()
           False -> do let from = dataPath ++ directorySep ++ file
                           to   = destination
                       status <- copyFile from to
                       unless (status == ExitSuccess) $ do
                          abortWithError FileIOError $
                                         "initialisation failed, can't copy " ++
                                         from ++ " to " ++ to

--------------------------------------------------------------------------------

-- call buddha-mk, give it the argument "silent" if we
-- want a quiet build, otherwise it is chatty
-- XXX this probably won't work on windows

buildBuddha :: IO ()
buildBuddha
   = do setCurrentDirectory buddhaDir 
        isSilentBuild <- isChatty Silent 
        let cmd = case isSilentBuild of
                     True  -> "./" ++ buddhaMake ++ " " ++ "silent"
                     False -> "./" ++ buddhaMake
        ifChat Normal $ putStrLn $ buddhaNameVersion ++ ": compiling"
        exitStatus <- system cmd 
        return ()

--------------------------------------------------------------------------------

-- the name of the environment variable that says where the interface files are
envIfacePath :: String
envIfacePath = "BUDDHA_IFACE_PATH"

-- the name of the environment variable that says where the data files are
envDataPath :: String
envDataPath = "BUDDHA_DATA_PATH"

-- look for the paths to buddha's data directory and
-- its interface files
pathsFromEnvironment :: IO (FilePath, FilePath)
pathsFromEnvironment
   = do maybeIfacePath <- E.try $ getEnv envIfacePath 
        case maybeIfacePath of
           Left e -> abortWithError EnvironmentError $ 
                        "environment " ++ envIfacePath ++ " not set" 
           Right ifacePath
              -> do maybeDataPath <- E.try $ getEnv envDataPath
                    case maybeDataPath of
                       Left e -> abortWithError EnvironmentError $
                                    "environment " ++ envDataPath ++ " not set" 
                       Right dataPath -> return (ifacePath, dataPath)

--------------------------------------------------------------------------------

-- check that all the files that were specified actually exist
-- abort if any of them do not
checkFilesExist :: [FilePath] -> IO [FilePath]
checkFilesExist [] = return []
checkFilesExist (f:fs)
   = do existAndReadable <- fileExistsAndIsReadable f 
        if existAndReadable 
           then do goodFiles <- checkFilesExist fs
                   return (f:goodFiles)
           else abortWithError FileIOError $
                   "file: " ++ f ++ " does not exist or is not readable"
